#pragma once

#include "Types/GamebaseContactTypes.h"

class GAMEBASE_API IGamebaseContact
{
public:
    virtual ~IGamebaseContact() = default;

    /**
     * Open TOAST Contact WebView with CS URL which is wrote in TOAST Gamebase App Console.
     * If the CS URL is empty or invalid, you will receive a GamebaseException.
     *
     * @param CloseCallback
     *
     * Example Usage:
     * @code
     *  void OpenContact()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetContact()->OpenContact(FGamebaseErrorDelegate::CreateLambda([](const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("OpenContact succeeded."));
     *          }
     *          else
     *          {
     *              if (Error->code == GamebaseErrorCode::WEBVIEW_INVALID_URL)
     *              {
     *                  // Gamebase Console Service Center URL is invalid.
     *                  // Please check the url field in the TOAST Gamebase Console.
     *                  auto launchingInfo = GamebaseSubsystem->GetLaunching().GetLaunchingInformations();
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("OpenContact failed. (csUrl: %s)"), *launchingInfo.launching.app.relatedUrls.csUrl);
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("OpenContact failed. (Error: %d)"), Error->code);
     *              }
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void OpenContact(const FGamebaseErrorDelegate& CloseCallback) = 0;

    /**
     * Open TOAST Contact WebView with CS URL which is wrote in TOAST Gamebase App Console.
     * If the CS URL is empty or invalid, you will receive a GamebaseException.
     *
     * @param Configuration     The init settings of contact URL.
     * @param CloseCallback   called when the browser closed.
     *
     * Example Usage:
     * @code
     *  void OpenContact(const FString& UserName, const FString& AdditionalURL)
     *  {
     *      FGamebaseContactConfiguration Configuration;
     *      Configuration.UserName = UserName;
     *      Configuration.AdditionalURL = AdditionalURL;
     *      
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetContact()->OpenContact(Configuration, FGamebaseErrorDelegate::CreateLambda([](const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("OpenContact succeeded."));
     *          }
     *          else
     *          {
     *              if (Error->code == GamebaseErrorCode::WEBVIEW_INVALID_URL)
     *              {
     *                  // Gamebase Console Service Center URL is invalid.
     *                  // Please check the url field in the TOAST Gamebase Console.
     *                  auto launchingInfo = GamebaseSubsystem->GetLaunching().GetLaunchingInformations();
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("OpenContact failed. (csUrl: %s)"), *launchingInfo.launching.app.relatedUrls.csUrl);
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("OpenContact failed. (Error: %d)"), Error->code);
     *              }
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void OpenContact(const FGamebaseContactConfiguration& Configuration, const FGamebaseErrorDelegate& CloseCallback) = 0;

    /**
     * Returns the customer service URL.
     * You can open webview with this URL.
     *
     * @param Callback    Returns the customer service URL.
     *
     * Example Usage:
     * @code
     *  void RequestContactURL()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetContact()->RequestContactURL(FGamebaseContactUrlDelegate::CreateLambda([=](FString url, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestContactURL succeeded. %s"), *url);
     *          }
     *          else
     *          {
     *              if (Error->code == GamebaseErrorCode::UI_CONTACT_FAIL_INVALID_URL)
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RequestContactURL failed. Please check the customer URL field in the TOAST Gamebase Console."));
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RequestContactURL failed. (Error: %d)"), Error->code);
     *              }
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestContactURL(const FGamebaseContactUrlDelegate& Callback) = 0;

    /**
     * Returns the customer service URL.
     * You can open webview with this URL.
     *
     * @param Configuration The init settings of contact URL.
     * @param Callback      Returns the customer service URL.
     *
     * Example Usage:
     * @code
     *  void RequestContactURL(const FString& UserName, const FString& AdditionalURL)
     *  {
     *      FGamebaseContactConfiguration Configuration;
     *      Configuration.UserName = UserName;
     *      Configuration.AdditionalURL = AdditionalURL;
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetContact()->RequestContactURL(Configuration, FGamebaseContactUrlDelegate::CreateLambda([=](FString Url, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestContactURL succeeded. %s"), *Url);
     *          }
     *          else
     *          {
     *              if (Error->code == GamebaseErrorCode::UI_CONTACT_FAIL_INVALID_URL)
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RequestContactURL failed. Please check the customer URL field in the TOAST Gamebase Console."));
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RequestContactURL failed. (Error: %d)"), Error->code);
     *              }
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestContactURL(const FGamebaseContactConfiguration& Configuration, const FGamebaseContactUrlDelegate& Callback) = 0;
};