#pragma once

#include "Types/GamebaseAnalyticsTypes.h"

class GAMEBASE_API IGamebaseAnalytics
{
public:
    virtual ~IGamebaseAnalytics() = default;

    /**
     * Send a game user data to Gamebase Server for analyzing the data.
     * This method should be called after login.
     *
     * @param GameUserData Game User Data
     *
     * @code
     *  void SetGameUserData(int32 UserLevel, const FString& ChannelId, const FString& CharacterId, const FString& CharacterClassId)
     *  {
     *      FGamebaseAnalyticsUserData Data;
     *      Data.UserLevel = UserLevel;
     *      Data.ChannelId = ChannelId;
     *      Data.CharacterId = CharacterId;
     *      Data.CharacterClassId = CharacterClassId;
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetAnalytics()->SetGameUserData(Data);
     *  }
     * @endcode
     */
    virtual void SetGameUserData(const FGamebaseAnalyticsUserData& GameUserData) = 0;

    /**
     * Send a game user data to Gamebase server for analyzing the data.
     * This method should be called after level up event.
     *
     * @param LevelUpData
     *
     * @code
     *  void TraceLevelUp(int32 UserLevel)
     *  {
     *      FGamebaseAnalyticsLevelUpData Data;
     *      Data.UserLevel = UserLevel;
     *      Data.LevelUpTime = FDateTime::Now().ToUnixTimestamp();
     *      
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetAnalytics()->TraceLevelUp(Data);
     *  }
     * @endcode
     */
    virtual void TraceLevelUp(const FGamebaseAnalyticsLevelUpData& LevelUpData) = 0;
};