#pragma once

#include "CoreMinimal.h"
#include "Gamebase.h"
#include "IGamebaseAnalytics.h"
#include "IGamebaseCommunity.h"
#include "IGamebaseContact.h"
#include "IGamebaseGameNotice.h"
#include "IGamebaseImageNotice.h"
#include "IGamebaseLaunching.h"
#include "IGamebaseLogger.h"
#include "IGamebaseNetwork.h"
#include "IGamebasePurchase.h"
#include "IGamebasePush.h"
#include "IGamebaseTemporaryWithdrawal.h"
#include "IGamebaseTerms.h"
#include "IGamebaseUtil.h"
#include "IGamebaseWebView.h"
#include "GamebaseSubsystem.generated.h"

UCLASS(Abstract)
class GAMEBASE_API UGamebaseSubsystem : public UGameInstanceSubsystem
{
    GENERATED_BODY()
    
public:
    /**
     * Used to enable or disable logging, and other debug features.
     * 
     * @param bIsDebugMode Debug features (like logging) are enabled if true, disabled if false.
     */
    virtual void SetDebugMode(bool bIsDebugMode)
        PURE_VIRTUAL(UGamebaseSubsystem::SetDebugMode,);

    /**
     * Sets the Gamebase displayLanguage.
     *
     * @param LanguageCode The Gamebase displayLanguage. Please use the GamebaseDisplayLanguageCode class.
     */
    virtual void SetDisplayLanguageCode(const FString& LanguageCode) const 
        PURE_VIRTUAL(UGamebaseSubsystem::SetDisplayLanguageCode,);
    
    /**
     * This function initializes the Gamebase SDK.
     * If this function is not called, the Gamebase SDK function will not work.
     *
     * @param Configuration The configurations required to run the Gamebase SDK.
     * @param Callback    Initialization result callback, returns the launching information as a result of initialization.
     *
     * Example Usage:
     * @code
     *  void Initialize(const FString& appID, const FString& appVersion)
     *  {
     *      FGamebaseConfiguration configuration;
     *      configuration.AppID = appID;
     *      configuration.AppVersion = appVersion;
     *      configuration.StoreCode = GamebaseStoreCode.Google;
     *      configuration.bEnablePopup = true;
     *      configuration.bEnableLaunchingStatusPopup = true;
     *      configuration.bEnableBanPopup = true;
     *  
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->Initialize(Configuration, FGamebaseLaunchingInfoDelegate::CreateLambda([=](const FGamebaseLaunchingInfo* launchingInfo, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Initialize succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Initialize failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void Initialize(const FGamebaseConfiguration& Configuration, const FGamebaseLaunchingInfoDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::Initialize,);
    
    /**
     * Add a Gamebase event handler to be called when every events are arrived.
     * You have to convert the message data to VO below according to the category value.<br>
     *  - GamebaseEventCategory::LoggedOut : FGamebaseEventLoggedOutData
     *  - GamebaseEventCategory::IdPRevoked : FGamebaseEventIdPRevokedData
     *  - GamebaseEventCategory::ServerPushAppKickOut : FGamebaseEventServerPushData
     *  - GamebaseEventCategory::ServerPushAppKickOutMessageReceived : FGamebaseEventServerPushData
     *  - GamebaseEventCategory::ServerPushTransferKickout : FGamebaseEventServerPushData
     *  - GamebaseEventCategory::ObserverLaunching : FGamebaseEventObserverData
     *  - GamebaseEventCategory::ObserverNetwork : FGamebaseEventObserverData
     *  - GamebaseEventCategory::ObserverHeartbeat : FGamebaseEventObserverData
     *  - GamebaseEventCategory::PurchaseUpdated : FGamebaseEventPurchasableReceipt
     *  - GamebaseEventCategory::PushReceivedMessage : FGamebaseEventPushMessage
     *  - GamebaseEventCategory::PushClickMessage : FGamebaseEventPushMessage
     *  - GamebaseEventCategory::PushClickAction : FGamebaseEventPushAction<br>
     * 
     * SERVER_PUSH : Receive messages from the Gamebase server.
     * OBSERVER : This is an event that fires when the launch, login account(heartbeat), or network connection 'status changes'.
     * PURCHASE_UPDATED : Promotion payment events can be received.
     * PUSH_RECEIVED_MESSAGE : This event operates when a Push message is received.
     * PUSH_CLICK_MESSAGE : This event is executed when the Push message is clicked.
     * PUSH_CLICK_ACTION : This is an event that is triggered when the action button added through the rich message function is clicked.
     *
     * @param Callback    The callback that will run.
     *
     * Example Usage:
     * @code
     *  void AddEventHandler()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->AddEventHandler(FGamebaseEventDelegate::FDelegate::CreateLambda([=](const FGamebaseEventMessage& message)
     *      {
     *          UE_LOG(GamebaseTestResults, Display, TEXT("Event = %s"), *message.category);
     *
     *          if (message.category.Equals(GamebaseEventCategory::LoggedOut))
     *          {
     *              auto loggedOutData = FGamebaseEventLoggedOutData::From(message.data);
     *          }
     *          else if (message.category.Equals(GamebaseEventCategory::IdPRevoked))
     *          {
     *              auto idPRevokedData = FGamebaseEventIdPRevokedData::From(message.data);
     *          }
     *          else if (message.category.Equals(GamebaseEventCategory::ServerPushAppKickOut) ||
     *              message.category.Equals(GamebaseEventCategory::ServerPushAppKickOutMessageReceived) ||
     *              message.category.Equals(GamebaseEventCategory::ServerPushTransferKickout))
     *          {
     *              auto serverPushData = FGamebaseEventServerPushData::From(message.data);
     *          }
     *          else if (message.category.Equals(GamebaseEventCategory::ObserverLaunching))
     *          {
     *              auto observerData = FGamebaseEventObserverData::From(message.data);
     *          }
     *          else if (message.category.Equals(GamebaseEventCategory::ObserverNetwork))
     *          {
     *              auto observerData = FGamebaseEventObserverData::From(message.data);
     *          }
     *          else if (message.category.Equals(GamebaseEventCategory::ObserverHeartbeat))
     *          {
     *              auto observerData = FGamebaseEventObserverData::From(message.data);
     *          }
     *          else if (message.category.Equals(GamebaseEventCategory::PurchaseUpdated))
     *          {
     *              auto purchasableReceipt = FGamebaseEventPurchasableReceipt::From(message.data);
     *          }
     *          else if (message.category.Equals(GamebaseEventCategory::PushReceivedMessage))
     *          {
     *              auto pushMessage = FGamebaseEventPushMessage::From(message.data);
     *          }
     *          else if (message.category.Equals(GamebaseEventCategory::PushClickMessage))
     *          {
     *              auto pushMessage = FGamebaseEventPushMessage::From(message.data);
     *          }
     *          else if (message.category.Equals(GamebaseEventCategory::PushClickAction))
     *          {
     *              auto pushAction = FGamebaseEventPushAction::From(message.data);
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual FDelegateHandle AddEventHandler(const FGamebaseEventDelegate::FDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::AddEventHandler, return FDelegateHandle(););

    /**
     * Remove a observer listener.
     * 
     * @param Handle    The callback that will be removed.
     */
    virtual void RemoveEventHandler(const FDelegateHandle& Handle)
        PURE_VIRTUAL(UGamebaseSubsystem::RemoveEventHandler,);

    /**
     * Remove all observer listeners.
     */
    virtual void RemoveAllEventHandler()
        PURE_VIRTUAL(UGamebaseSubsystem::RemoveAllEventHandler,);

    /**
     * Add a observer to be called when network status, launching status or user status is changed.
     *
     * @param ProviderName  The provider name witch is authentication provider.
     * @param Callback    Login result callback, returns the authentication token as a result of login.
     *
     * Example Usage:
     * @code
     *  void Login()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->Login(GamebaseAuthProvider::Guest, FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Login succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Login failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void Login(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::Login,);

    /**
     * Logs the user in with the credential of external authentication provider.
     *
     * @param CredentialInfo    The CredentialInfo which is credential of authentication provider.
     * @param Callback        Login result callback,  returns the authentication token as a result of login.
     *
     * Example Usage:
     * @code
     *  void Login()
     *  {
     *      FGamebaseVariantMap CredentialInfo;
     *      CredentialInfo.Add(GamebaseAuthProviderCredential::ProviderName, GamebaseAuthProvider::Google);
     *      CredentialInfo.Add(GamebaseAuthProviderCredential::AccessToken, TEXT("Token"));
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->Login(CredentialInfo, FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Login succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Login failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void Login(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::Login,);

    /**
     * Logs the user in with the external authentication provider.
     *
     * @param ProviderName      The provider name witch is authentication provider.
     * @param AdditionalInfo    The CredentialInfo which is credential of authentication provider.
     * @param Callback        Login result callback,  returns the authentication token as a result of login.
     *
     * Example Usage:
     * @code
     *  void Login()
     *  {
     *      FGamebaseVariantMap AdditionalInfo;
     *      AdditionalInfo.Add(TEXT("Key"), TEXT("Value"));
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->Login(GamebaseAuthProvider::XXX, AdditionalInfo, FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Login succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Login failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::Login,);

    /**
     * Logs the user in with last logged in authentication provider.
     *
     * @param Callback    Login result callback, returns the authentication token as a result of login.
     *
     * Example Usage:
     * @code
     *  void LoginForLastLoggedInProvider()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->LoginForLastLoggedInProvider(FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("LoginForLastLoggedInProvider succeeded."));
     *          }
     *          else
     *          {
     *              if (Error->code == GamebaseErrorCode::SOCKET_ERROR || Error->code == GamebaseErrorCode::SOCKET_RESPONSE_TIMEOUT)
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("Retry LoginForLastLoggedInProvider or notify an error message to the user. : %s"), *Error->message);
     *              }
     *              else
     *              {
     *                  FString lastLoggedInProvider = GamebaseSubsystem->GetLastLoggedInProvider();
     *                  if (lastLoggedInProvider.IsEmpty())
     *                  {
     *                      // Display the IdP select menu to user.
     *                  }
     *                  else
     *                  {
     *                      // Login with lastLoggedInProvider
     *                  }
     *              }
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void LoginForLastLoggedInProvider(const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::LoginForLastLoggedInProvider,);
    
    /**
     * Logs the user in with last logged in authentication provider.
     *
     * @param AdditionalInfo    The AdditionalInfo which is additional information using for mapping.
     * @param Callback    Login result callback, returns the authentication token as a result of login.
     *
     * Example Usage:
     * @code
     *  void LoginForLastLoggedInProvider()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->LoginForLastLoggedInProvider(FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("LoginForLastLoggedInProvider succeeded."));
     *          }
     *          else
     *          {
     *              if (Error->code == GamebaseErrorCode::SOCKET_ERROR || Error->code == GamebaseErrorCode::SOCKET_RESPONSE_TIMEOUT)
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("Retry LoginForLastLoggedInProvider or notify an error message to the user. : %s"), *Error->message);
     *              }
     *              else
     *              {
     *                  FString lastLoggedInProvider = GamebaseSubsystem->GetLastLoggedInProvider();
     *                  if (lastLoggedInProvider.IsEmpty())
     *                  {
     *                      // Display the IdP select menu to user.
     *                  }
     *                  else
     *                  {
     *                      // Login with lastLoggedInProvider
     *                  }
     *              }
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void LoginForLastLoggedInProvider(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::LoginForLastLoggedInProvider,);
    
    /**
     * (Windows only)
     * Cancels the login attempt via external browser.
     *
     * This works only when the login process using an external browser is in progress.
     * It aborts the login attempt and returns the result to the login callback.
     *
     * Example Usage:
     * @code
     *  void CancelLogin()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->CancelLoginWithExternalBrowser();
     *  }
     * @endcode
     */
    virtual void CancelLoginWithExternalBrowser()
        PURE_VIRTUAL(UGamebaseSubsystem::CancelLoginWithExternalBrowser,);
    
    /**
     * Mapping the currently authenticated user identifier of Gamebase with another external authentication.
     *
     * @param ProviderName The ProviderName which is authentication provider.
     * @param Callback   Mapping result callback, returns the authentication token as a result of mapping.
     *
     * Example Usage:
     * @code
     *  void AddMappingSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->AddMapping(GamebaseAuthProvider::XXX, FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping succeeded. Gamebase userId is %s"), *authToken->member.userId);
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void AddMapping(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::AddMapping,);
    
    /**
     * Mapping the currently authenticated user identifier of Gamebase with another external authentication.
     *
     * @param ProviderName   The ProviderName which is authentication provider.
     * @param AdditionalInfo The AdditionalInfo which is additional information using for mapping.
     * @param Callback     Callbacks the results of mappings, returns the authentication token as a result of mappings.
     *
     * Example Usage:
     * @code
     *  void AddMappingSample()
     *  {
     *      FGamebaseVariantMap AdditionalInfo;
     *      AdditionalInfo->SetStringField(TEXT("Key"), TEXT("Value"));
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->AddMapping(GamebaseAuthProvider::XXX, AdditionalInfo, FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping succeeded. Gamebase userId is %s"), *authToken->member.userId);
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void AddMapping(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::AddMapping,);

    /**
     * Mapping the currently authenticated user identifier of Gamebase with the credential of external authentication provider.
     *
     * @param CredentialInfo    The CredentialInfo which is credential of authentication provider.
     * @param Callback        Mapping result callback, returns the authentication token as a result of mapping.
     *
     * Example Usage:
     * @code
     *  void AddMappingSample()
     *  {
     *      FGamebaseVariantMap CredentialInfo;
     *      CredentialInfo.Add(GamebaseAuthProviderCredential::ProviderName, GamebaseAuthProvider::Google);
     *      CredentialInfo.Add(GamebaseAuthProviderCredential::AccessToken, TEXT("Token"));
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->AddMapping(CredentialInfo, FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping succeeded. Gamebase userId is %s"), *authToken->member.userId);
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void AddMapping(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::AddMapping,);
    
    /**
     * Change logged in account with ForcingMappingTicket.
     *
     * @param ForcingMappingTicket  The mapping information which is necessary to log in another account.
     * @param Callback            Resume mapping result callback, returns the authentication token as a result of mapping.
     *
     * Example Usage:
     * @code
     *  void ChangeLoginSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->AddMapping(GamebaseAuthProvider::XXX, FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping succeeded. Gamebase userId is %s"), *authToken->member.userId);
     *          }
     *          else
     *          {
     *              // If you got this error code(AUTH_ADD_MAPPING_ALREADY_MAPPED_TO_OTHER_MEMBER) that means this user already has another account of the AuthProvider.XXX),
     *              // You can call this method, AddMappingForcibly() which can try to map forcibly with the AuthProvider.XXX.
     *              if (Error->code == GamebaseErrorCode::AUTH_ADD_MAPPING_ALREADY_MAPPED_TO_OTHER_MEMBER)
     *              {
     *                  // Before calling the mapping forcibly api, You should get this ForcingMappingTicket for this method parameter.
     *                  FGamebaseForcingMappingTicket* ForcingMappingTicket = FGamebaseForcingMappingTicket::From(Error);
     *                  if (ForcingMappingTicket == nullptr)
     *                  {
     *                      // Unexpected error occurred. Contact Administrator.
     *                  }
     *
     *                  // Try to change log in account with the ForcingMappingTicket.
     *                  GamebaseSubsystem->ChangeLogin(ForcingMappingTicket,
     *                      FGamebaseAuthTokenDelegate::CreateLambda([](const FGamebaseAuthToken* innerAuthToken, const FGamebaseError* innerError)
     *                  {
     *                      if (Gamebase::IsSuccess(innerError))
     *                      {
     *                          UE_LOG(GamebaseTestResults, Display, TEXT("AddMappingForcibly succeeded. Gamebase userId is %s"), *innerAuthToken->member.userId);
     *                      }
     *                      else
     *                      {
     *                          // Check the error code and Handle the error appropriately.
     *                          UE_LOG(GamebaseTestResults, Display, TEXT("AddMappingForcibly failed."));
     *                      }
     *                  }));
     *              }
     *              else
     *              {
     *                  // Add Mapping Forcibly Failed.
     *                  // Check the error code and Handle the error appropriately.
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping failed."));
     *              }
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void ChangeLogin(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::ChangeLogin,);
    
    /**
     * Forcibly trying to map the currently authenticated user identifier of Gamebase with the credential of external authentication provider.
     *
     * @param ForcingMappingTicket  The mapping information which is necessary to map forcibly with the provider that is already mapped with another account.
     * @param Callback            Mapping result callback, returns the authentication token as a result of mapping.
     *
     * Example Usage:
     * @code
     *  void AddMappingForciblySample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->AddMapping(GamebaseAuthProvider::XXX, *AdditionalInfo, FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping succeeded. Gamebase userId is %s"), *authToken->member.userId);
     *          }
     *          else
     *          {
     *              // If you got this error code(AUTH_ADD_MAPPING_ALREADY_MAPPED_TO_OTHER_MEMBER) that means this user already has another account of the AuthProvider.XXX),
     *              // You can call this method, AddMappingForcibly() which can try to map forcibly with the AuthProvider.XXX.
     *              if (Error->code == GamebaseErrorCode::AUTH_ADD_MAPPING_ALREADY_MAPPED_TO_OTHER_MEMBER)
     *              {
     *                  // Before calling the mapping forcibly api, You should get this ForcingMappingTicket for this method parameter.
     *                  FGamebaseForcingMappingTicket* ForcingMappingTicket = FGamebaseForcingMappingTicket::From(Error);
     *                  if (ForcingMappingTicket == nullptr)
     *                  {
     *                      // Unexpected error occurred. Contact Administrator.
     *                  }
     *
     *                  // Try to add mapping forcibly with the ForcingMappingTicket.
     *                  GamebaseSubsystem->ChangeLogin(ForcingMappingTicket,
     *                      FGamebaseAuthTokenDelegate::CreateLambda([](const FGamebaseAuthToken* innerAuthToken, const FGamebaseError* innerError)
     *                  {
     *                      if (Gamebase::IsSuccess(innerError))
     *                      {
     *                          // Log in account changed successfully.
     *                          UE_LOG(GamebaseTestResults, Display, TEXT("ChangeLogin succeeded. Gamebase userId is %s"), *innerAuthToken->member.userId);
     *                      }
     *                      else
     *                      {
     *                          // Change login failed.
     *                          // The UserID will not changed.
     *                          UE_LOG(GamebaseTestResults, Display, TEXT("ChangeLogin failed."));
     *                      }
     *                  }));
     *              }
     *              else
     *              {
     *                  // Add Mapping Forcibly Failed.
     *                  // Check the error code and Handle the error appropriately.
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("AddMapping failed."));
     *              }
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void AddMappingForcibly(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::AddMappingForcibly,);

    /**
     * Removes external authentication that is mapped to the current user identifier.
     *
     * @param ProviderName  The ProviderName which is authentication provider.
     * @param Callback    Callbacks the result of removing the mapping.
     *
     * Example Usage:
     * @code
     *  void RemoveMapping()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->RemoveMapping(GamebaseAuthProvider::XXX, FGamebaseErrorDelegate::CreateLambda([=](const FGamebaseError& Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RemoveMapping succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RemoveMapping failed. (Error: %d)"), Error->code);
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RemoveMapping(const FString& ProviderName, const FGamebaseErrorDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::RemoveMapping,);

    /**
     * Logs out the user.
     *
     * @param Callback    Callbacks the result of logout.
     *
     * Example Usage:
     * @code
     *  void LogoutSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->Logout(FGamebaseErrorDelegate::CreateLambda([=](const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Logout succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Logout failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void Logout(const FGamebaseErrorDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::Logout,);

    /**
     * Withdraws the user.
     *
     * @param Callback    Callbacks the result of withdraw.
     *
     * Example Usage:
     * @code
     *  void WithdrawSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->Withdraw(FGamebaseErrorDelegate::CreateLambda([=](const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Withdraw succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("Withdraw failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void Withdraw(const FGamebaseErrorDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::Withdraw,);

    /**
     * Try to issue the transfer account.
     *
     * @param Callback    Callbacks the result of IssueTransferAccount
     *
     * Example Usage:
     * @code
     *  void IssueTransferAccountSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->IssueTransferAccount(FGamebaseTransferAccountDelegate::CreateLambda([=](const FGamebaseTransferAccountInfo* transferAccountInfo, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("IssueTransferAccount succeeded. (ID: %s, Password: %s)"), *transferAccountInfo->account.id, *transferAccountInfo->account.password);
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("IssueTransferAccount failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void IssueTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::IssueTransferAccount,);

    /**
     * Try to query the transfer account to Gamebase.
     * If the transfer account is already issued, it returns.
     *
     * @param Callback    Callbacks the result of QueryTransferAccount
     *
     * Example Usage:
     * @code
     *  void QueryTransferAccountSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->QueryTransferAccount(FGamebaseTransferAccountDelegate::CreateLambda([=](const FGamebaseTransferAccountInfo* transferAccountInfo, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("QueryTransferAccount succeeded. (ID: %s)"), *transferAccountInfo->account.id);
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("QueryTransferAccount failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void QueryTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::QueryTransferAccount,);

    /**
     * Try to renew the transfer account.
     * If you want to renew the account automatically or manually,
     * you should pass the proper TransferAccountRenewConfiguration object to the first parameter.
     *
     * @param Configuration     Configuration for Transfer Account renewal.
     * @param Callback        Callbacks the result of RenewTransferAccountManualIdPassword
     *
     * Example Usage:
     * @code
     *  void RenewTransferAccount(const FString& AccountId, const FString& AccountPassword)
     *  {
     *      FGamebaseTransferAccountRenewConfiguration Configuration;
     *      Configuration.AccountId = AccountId;
     *      Configuration.AccountPassword = AccountPassword;
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->RenewTransferAccount(Configuration, FGamebaseTransferAccountDelegate::CreateLambda([=](const FGamebaseTransferAccountInfo* transferAccountInfo, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RenewTransferAccount succeeded. (ID: %s, Password: %s)"), *transferAccountInfo->account.id, *transferAccountInfo->account.password);
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RenewTransferAccount failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RenewTransferAccount(const FGamebaseTransferAccountRenewConfiguration& Configuration, const FGamebaseTransferAccountDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::RenewTransferAccount,);

    /**
     * Try to transfer account with transfer account id and password.
     * If this process is finished successfully, it proceeds login process and return AuthToken.
     *
     * @param AccountId         TransferAccount id received from old device.
     * @param AccountPassword   TransferAccount password from old device.
     * @param Callback        Callbacks the result of TransferAccountWithIdPLogin
     *
     * Example Usage:
        * @code
     *  void TransferAccountWithIdPLoginSample(const FString& AccountId, const FString& AccountPassword)
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->TransferAccountWithIdPLogin(AccountId, AccountPassword, FGamebaseAuthTokenDelegate::CreateLambda([=](const FGamebaseAuthToken* authToken, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("TransferAccountWithIdPLogin succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("TransferAccountWithIdPLogin failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void TransferAccountWithIdPLogin(const FString& AccountId, const FString& AccountPassword, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::TransferAccountWithIdPLogin,);

    /**
     * Gets the name of the last logged-in authentication provider.
     *
     * Example Usage:
     *  @code
     *  void RequestLastLoggedInProviderSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->RequestLastLoggedInProvider(FGamebaseLastLoggedInProviderDelegate::CreateLambda([=](const FString& LastLoggedInProvider, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestLastLoggedInProviderSample succeeded. (LastLoggedInProvider: %s)"), *LastLoggedInProvider);
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestLastLoggedInProviderSample failed."));
     *          }
     *      }));
     *  }
     *  @endcode 
     */
    virtual void RequestLastLoggedInProvider(const FGamebaseLastLoggedInProviderDelegate& Callback) const
        PURE_VIRTUAL(UGamebaseSubsystem::RequestLastLoggedInProvider,);

    /**
     * Gets the name of the last logged-in authentication provider.
     *
     * @return The name that is last logged-in authentication provider.
     */
    virtual FString GetLastLoggedInProvider() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetLastLoggedInProvider, return FString(););

    /**
     * Gets the user ID that is currently logged in.
     *
     * @return The user id that is currently logged in.
     */
    virtual FString GetUserID() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetUserID, return FString(););

    /**
     * Gets the access token for the current user.
     *
     * @return The access token for Gamebase platform.
     */
    virtual FString GetAccessToken() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetAccessToken, return FString(););
    
    /**
     * Gets the list of external authentication providers mapped to the current user identifier.
     *
     * @return The list of external authentication providers mapped to the current user identifier.
     */
    virtual TArray<FString> GetAuthMappingList()
        PURE_VIRTUAL(UGamebaseSubsystem::GetAuthMappingList, return {};);
    
    /**
     * Gets the user ID from the authentication provider.
     *
     * @param ProviderName  The ProviderName which is authentication provider.
     * @return The user ID from the authentication provider.
     */
    virtual FString GetAuthProviderUserID(const FString& ProviderName)
        PURE_VIRTUAL(UGamebaseSubsystem::GetAuthProviderUserID, return FString(););
    
    /**
     * Gets the access token from the authentication provider.
     *
     * @param ProviderName  The ProviderName which is authentication provider.
     * @return The access token from the authentication provider.
     */
    virtual FString GetAuthProviderAccessToken(const FString& ProviderName)
        PURE_VIRTUAL(UGamebaseSubsystem::GetAuthProviderAccessToken, return FString(););

    /**
     * Gets the profile from the authentication provider.
     *
     * @param ProviderName  The ProviderName which is authentication provider.
     * @return The profile from the authentication provider.
     */
    virtual FGamebaseAuthProviderProfilePtr GetAuthProviderProfile(const FString& ProviderName)
        PURE_VIRTUAL(UGamebaseSubsystem::GetAuthProviderProfile, return nullptr;);

    /**
     * Gets the ban information of the suspended user.
     *
     * @return The ban information of the suspended user.
     */
    virtual FGamebaseBanInfoPtr GetBanInfo()
        PURE_VIRTUAL(UGamebaseSubsystem::GetBanInfo, return nullptr;);

    /**
     * Returns the current version of the Gamebase SDK for Android as a string.
     *
     * @return The current version of the Gamebase SDK.
     */
    virtual FString GetSDKVersion() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetSDKVersion, return FString(););
    
    /**
     * Returns the current version of the Gamebase SDK for Android as a string.
     *
     * @return The current version of the Gamebase SDK.
     */
    virtual bool IsSandbox() const
        PURE_VIRTUAL(UGamebaseSubsystem::IsSandbox, return false;);
    
    /**
     * Gets the language code set for the current device.
     *
     * @return The device language code that is currently set.
     */
    virtual FString GetDeviceLanguageCode() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetDeviceLanguageCode, return FString(););
    
    /**
     * Gets the carrier code set for the current device.
     *
     * @return The carrier code set for the current device.
     */
    virtual FString GetCarrierCode() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetCarrierCode, return FString(););
    
    /**
     * Gets the carrier name set for the current device.
     *
     * @return The carrier name set for the current device.
     */
    virtual FString GetCarrierName() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetCarrierName, return FString(););
    
    /**
     * Gets the country code.
     * First, get the country code set in USIM, <br>
     * and if there is no USIM, get the country code set in the device.
     *
     * @return The country code.
     */
    virtual FString GetCountryCode() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetCountryCode, return FString(););
    
    /**
     * Gets the country code set in USIM.
     *
     * @return The country code that is set in USIM.
     */
    virtual FString GetCountryCodeOfUSIM() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetCountryCodeOfUSIM, return FString(););
    
    /**
     * Gets the country code currently set on the device.
     *
     * @return The country code currently set on the device.
     */
    virtual FString GetCountryCodeOfDevice() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetCountryCodeOfDevice, return FString(););
    
    /**
     * Gets the Gamebase displayLanguage.
     *
     * @return The display language code that is currently set.
     */
    virtual FString GetDisplayLanguageCode() const
        PURE_VIRTUAL(UGamebaseSubsystem::GetDisplayLanguageCode, return FString(););
    
    /**
     * The Analytics class can send the game indicator to the Gamebase Server.
     */
    virtual IGamebaseAnalytics* GetAnalytics()
        PURE_VIRTUAL(UGamebaseSubsystem::GetAnalytics, return nullptr;);
    
    /**
     * This class provides wrapping of function execution related to NHN Cloud Contact.
     */
    virtual IGamebaseContact* GetContact()
        PURE_VIRTUAL(UGamebaseSubsystem::GetContact, return nullptr;);
    
    /**
     * This class is provided to check information such as launch information, status, and so on.
     */
    virtual IGamebaseLaunching* GetLaunching()
        PURE_VIRTUAL(UGamebaseSubsystem::GetLaunching, return nullptr;);
    
    /**
     * Send a log to Log & Crash Server.
     * The log levels are as follows: debug < info < warn < error < fatal.
     * Log & Crash Search allows fast and patterned search of any logs you want among a large volume.
     */
    virtual IGamebaseLogger* GetLogger()
        PURE_VIRTUAL(UGamebaseSubsystem::GetLogger, return nullptr;);
    
    /**
     * This class provides network status information.
     */
    virtual IGamebaseNetwork* GetNetwork()
        PURE_VIRTUAL(UGamebaseSubsystem::GetNetwork, return nullptr;);
    
    /**
     * This class provides wrapping of function execution related to payment.
     */
    virtual IGamebasePurchase* GetPurchase()
        PURE_VIRTUAL(UGamebaseSubsystem::GetPurchase, return nullptr;);
    
    /**
     * This class provides wrapping of function execution related to push notification.
     */
    virtual IGamebasePush* GetPush()
        PURE_VIRTUAL(UGamebaseSubsystem::GetPush, return nullptr;);
    
    /**
     * This class provides utility functions.
     */
    virtual IGamebaseUtil* GetUtil()
        PURE_VIRTUAL(UGamebaseSubsystem::GetUtil, return nullptr;);
    
    /**
     * This class implements the Webview call function.
     */
    virtual IGamebaseWebView* GetWebView()
        PURE_VIRTUAL(UGamebaseSubsystem::GetWebView, return nullptr;);

    /**
     * This class implements the game notice call function.
     */
    virtual IGamebaseGameNotice* GetGameNotice()
        PURE_VIRTUAL(UGamebaseSubsystem::GetGameNotice, return nullptr;);
    
    /**
     * This class implements the image notice call function.
     */
    virtual IGamebaseImageNotice* GetImageNotice()
        PURE_VIRTUAL(UGamebaseSubsystem::GetImageNotice, return nullptr;);
    
    /**
     * This class provides wrapping of function execution related to temporary withdrawal.
     */
    virtual IGamebaseTemporaryWithdrawal* GetTemporaryWithdrawal()
        PURE_VIRTUAL(UGamebaseSubsystem::GetTemporaryWithdrawal, return nullptr;);
    
    /**
    * This class provides terms functions.
    */
    virtual IGamebaseTerms* GetTerms()
        PURE_VIRTUAL(UGamebaseSubsystem::GetTerms, return nullptr;);
    
    /**
    * This class provides community functions.
    */
    virtual IGamebaseCommunity* GetCommunity()
        PURE_VIRTUAL(UGamebaseSubsystem::GetCommunity, return nullptr;);
    
    /**
    * This API is legacy and no longer maintained.
    * Please refer to the URL below for migrating to the new API:
    * https://docs.nhncloud.com/ko/Game/Gamebase/ko/unreal-authentication/#add-mapping-forcibly
    * @note The legacy API may be deprecated in the future. Transitioning is recommended.
    */
    virtual void AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::AddMappingForcibly,);

    /**
    * This API is legacy and no longer maintained.
    * Please refer to the URL below for migrating to the new API:
    * https://docs.nhncloud.com/ko/Game/Gamebase/ko/unreal-authentication/#add-mapping-forcibly
    * @note The legacy API may be deprecated in the future. Transitioning is recommended.
    */
    virtual void AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::AddMappingForcibly,);
    
    /**
    * This API is legacy and no longer maintained.
    * Please refer to the URL below for migrating to the new API:
    * https://docs.nhncloud.com/ko/Game/Gamebase/ko/unreal-authentication/#add-mapping-forcibly
    * @note The legacy API may be deprecated in the future. Transitioning is recommended.
    */
    virtual void AddMappingForcibly(const FGamebaseVariantMap& CredentialInfo, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
        PURE_VIRTUAL(UGamebaseSubsystem::AddMappingForcibly,);
    
protected:
    // USubsystem interface
    virtual void Initialize(FSubsystemCollectionBase& Collection) override;
    virtual void Deinitialize() override;
    // ~USubsystem interface
};
