#include "GamebasePushTypes.h"

#include "Dom/JsonObject.h"
#include "Serialization/JsonReader.h"
#include "Serialization/JsonSerializer.h"
#include "NhnJsonSerializeMacros.h"

namespace AgreePushTypes
{
    const FString None{ TEXT("NONE") };
    const FString Day{ TEXT("DAY") };
    const FString Night{ TEXT("NIGHT") };
    const FString All{ TEXT("ALL") };
}

FGamebasePushConfigurationPtr FGamebasePushConfiguration::From(const FGamebaseDataContainer* DataContainer)
{
    if (DataContainer == nullptr || DataContainer->Data.IsEmpty())
    {
        return nullptr;
    }
    
    TSharedPtr<FJsonObject> JsonObject;
    const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(DataContainer->Data);
    if (!FJsonSerializer::Deserialize(Reader, JsonObject) || !JsonObject.IsValid())
    {
        return nullptr;
    }

    const FString AgreePush = JsonObject->GetStringField("agreePush");
    
    FGamebasePushConfigurationPtr PushConfiguration = MakeShared<FGamebasePushConfiguration, ESPMode::ThreadSafe>();
    if (AgreePush.Equals(AgreePushTypes::Day) == true)
    {
        PushConfiguration->bPushEnabled = true;
        PushConfiguration->bADAgreement = true;
        PushConfiguration->bADAgreementNight = false;
    }
    if (AgreePush.Equals(AgreePushTypes::Night) == true)
    {
        PushConfiguration->bPushEnabled = true;
        PushConfiguration->bADAgreement = false;
        PushConfiguration->bADAgreementNight = true;
    }
    if (AgreePush.Equals(AgreePushTypes::All) == true)
    {
        PushConfiguration->bPushEnabled = true;
        PushConfiguration->bADAgreement = true;
        PushConfiguration->bADAgreementNight = true;
    }
    else
    {
        // 'agreePush' values handled in this case: NONE and other (including empty string)
        PushConfiguration->bPushEnabled = true;
        PushConfiguration->bADAgreement = false;
        PushConfiguration->bADAgreementNight = false;
    }

    return PushConfiguration;
}

void FGamebasePushConfiguration::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("pushEnabled", bPushEnabled);
    NHN_JSON_SERIALIZE("adAgreement", bADAgreement);
    NHN_JSON_SERIALIZE("adAgreementNight", bADAgreementNight);
    NHN_JSON_SERIALIZE("displayLanguageCode", DisplayLanguageCode);
    NHN_JSON_SERIALIZE_OPTIONAL("requestNotificationPermission", bRequestNotificationPermission);
    NHN_JSON_SERIALIZE_OPTIONAL("alwaysAllowTokenRegistration", bAlwaysAllowTokenRegistration);
    END_NHN_VO_SERIALIZER
}

void FGamebasePushTokenInfo::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("sandbox", bSandbox);
    NHN_JSON_SERIALIZE("pushType", PushType);
    NHN_JSON_SERIALIZE("token", Token);
    NHN_JSON_SERIALIZE("userId", UserId);
    NHN_JSON_SERIALIZE("deviceCountryCode", DeviceCountryCode);
    NHN_JSON_SERIALIZE("timezone", Timezone);
    NHN_JSON_SERIALIZE("registeredDateTime", RegisteredDateTime);
    NHN_JSON_SERIALIZE("languageCode", LanguageCode);
    NHN_JSON_SERIALIZE_OBJECT_SERIALIZABLE("agreement", Agreement);
    END_NHN_VO_SERIALIZER
}

void FGamebaseNotificationOptions::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("foregroundEnabled", bForegroundEnabled);
    NHN_JSON_SERIALIZE("badgeEnabled", bBadgeEnabled);
    NHN_JSON_SERIALIZE("soundEnabled", bSoundEnabled);
    NHN_JSON_SERIALIZE("priority", Priority);
    NHN_JSON_SERIALIZE("smallIconName", SmallIconName);
    NHN_JSON_SERIALIZE("soundFileName", SoundFileName);
    END_NHN_VO_SERIALIZER
}

void FGamebasePushTokenInfo::FAgreement::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("pushEnabled", bPushEnabled);
    NHN_JSON_SERIALIZE("adAgreement", bAdAgreement);
    NHN_JSON_SERIALIZE("adAgreementNight", bAdAgreementNight);
    END_NHN_VO_SERIALIZER
}