#include "GamebaseAuthTypes.h"

#include "GamebaseAuthProviderCredential.h"
#include "GamebaseError.h"
#include "NhnJsonSerializeMacros.h"

namespace GamebaseErrorExtras
{
    const FString BanInfoKey{ TEXT("ban") };
    const FString ForcingMappingTicket{ TEXT("forcingMappingTicket") };
    const FString FailTransferAccount{ TEXT("failTransferAccount") };
}

void FGamebaseGraceBanInfo::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{   
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("gracePeriodDate", GracePeriodDate);
    NHN_JSON_SERIALIZE("message", Message);
    NHN_JSON_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE("paymentStatus", PaymentStatus);
    NHN_JSON_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE("releaseRuleCondition", ReleaseRuleCondition);
    END_NHN_VO_SERIALIZER
}

void FGamebaseGraceBanInfo::FPaymentStatus::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("amount", Amount);
    NHN_JSON_SERIALIZE("count", Count);
    NHN_JSON_SERIALIZE("currency", Currency);
    END_NHN_VO_SERIALIZER
}

void FGamebaseGraceBanInfo::FReleaseRuleCondition::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("amount", Amount);
    NHN_JSON_SERIALIZE("count", Count);
    NHN_JSON_SERIALIZE("currency", Currency);
    NHN_JSON_SERIALIZE("conditionType", ConditionType);
    END_NHN_VO_SERIALIZER
}

void FGamebaseTemporaryWithdrawalInfo::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("gracePeriodDate", GracePeriodDate);
    END_NHN_VO_SERIALIZER
}

void FGamebaseAuthMappingInfo::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("authKey", AuthKey);
    NHN_JSON_SERIALIZE("authSystem", AuthSystem);
    NHN_JSON_SERIALIZE("idPCode", IdpCode);
    NHN_JSON_SERIALIZE("regDate", RegDate);
    NHN_JSON_SERIALIZE("userId", UserId);
    END_NHN_VO_SERIALIZER
}

void FGamebaseAuthToken::FToken::Serialize(
    FJsonSerializerBase& Serializer,
    const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("sourceIdPCode", SourceIdpCode);
    NHN_JSON_SERIALIZE("accessToken", AccessToken);
    NHN_JSON_SERIALIZE_OPTIONAL("accessTokenSecret", AccessTokenSecret);
    NHN_JSON_SERIALIZE_OPTIONAL("subCode", SubCode);
    NHN_JSON_SERIALIZE_OPTIONAL("expiresIn", ExpiresIn);
    NHN_JSON_SERIALIZE_OPTIONAL_VARIANT_MAP("extraParams", ExtraParams);
    END_NHN_VO_SERIALIZER
}

FGamebaseVariantMap FGamebaseAuthToken::FToken::ToCredentialInfo() const
{
    FGamebaseVariantMap CredentialInfo;
    
    CredentialInfo.Add(GamebaseAuthProviderCredential::ProviderName, SourceIdpCode);
    CredentialInfo.Add(GamebaseAuthProviderCredential::GamebaseAccessToken, AccessToken);

    if (SubCode.IsSet())
    {
        CredentialInfo.Add(GamebaseAuthProviderCredential::SubCode, *SubCode);
    }
    
    if (ExtraParams.IsSet())
    {
        CredentialInfo.Add(GamebaseAuthProviderCredential::ExtraParams, *ExtraParams);
    }
    
    return CredentialInfo;
}

void FGamebaseAuthToken::FMember::Serialize(
    FJsonSerializerBase& Serializer,
    const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("appId", AppId);
    NHN_JSON_SERIALIZE_ARRAY_SERIALIZABLE("authList", AuthList, FGamebaseAuthMappingInfo);
    NHN_JSON_SERIALIZE("lastLoginDate", LastLoginDate);
    NHN_JSON_SERIALIZE("regDate", RegDate);
    NHN_JSON_SERIALIZE("userId", UserId);
    NHN_JSON_SERIALIZE("valid", Valid);
    NHN_JSON_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE("temporaryWithdrawal", TemporaryWithdrawal);
    NHN_JSON_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE("graceBan", GraceBan);
    END_NHN_VO_SERIALIZER
}

void FGamebaseAuthToken::Serialize(
    FJsonSerializerBase& Serializer,
    const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE_OBJECT_SERIALIZABLE("token", Token);
    NHN_JSON_SERIALIZE_OBJECT_SERIALIZABLE("member", Member);
    END_NHN_VO_SERIALIZER
}

void FGamebaseAuthProviderProfile::Serialize(
    FJsonSerializerBase& Serializer,
    const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE_VARIANT_MAP("information", Information);
    END_NHN_VO_SERIALIZER
}

FGamebaseBanInfoPtr FGamebaseBanInfo::From(const FGamebaseError* Error)
{
    if (Error == nullptr)
    {
        return nullptr;
    }
    
    const TOptional<FString>& JsonString = Error->FindExtra(GamebaseErrorExtras::BanInfoKey);
    if (JsonString.IsSet() == false)
    {
        return nullptr;
    }

    const FGamebaseBanInfoPtr BanInfo = MakeShared<FGamebaseBanInfo, ESPMode::ThreadSafe>();
    BanInfo->FromJson(JsonString.GetValue());
    
    return BanInfo;
}

void FGamebaseBanInfo::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("userId", UserId);
    NHN_JSON_SERIALIZE("banType", BanType);
    NHN_JSON_SERIALIZE("beginDate", BeginDate);
    NHN_JSON_SERIALIZE("endDate", EndDate);
    NHN_JSON_SERIALIZE("message", Message);
    NHN_JSON_SERIALIZE("csInfo", CsInfo);
    NHN_JSON_SERIALIZE("csUrl", CsUrl);
    END_NHN_VO_SERIALIZER
}

void FGamebaseTransferAccountInfo::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("issuedType", IssuedType);
    NHN_JSON_SERIALIZE_OBJECT_SERIALIZABLE("account", Account);
    NHN_JSON_SERIALIZE_OBJECT_SERIALIZABLE("condition", Condition);
    END_NHN_VO_SERIALIZER
}

void FGamebaseTransferAccountInfo::FAccount::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("id", Id);
    NHN_JSON_SERIALIZE("password", Password);
    END_NHN_VO_SERIALIZER
}

void FGamebaseTransferAccountInfo::FCondition::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("transferAccountType", TransferAccountType);
    NHN_JSON_SERIALIZE("expirationType", ExpirationType);
    NHN_JSON_SERIALIZE("expirationDate", ExpirationDate);
    END_NHN_VO_SERIALIZER
}

FGamebaseTransferAccountFailInfoPtr FGamebaseTransferAccountFailInfo::From(const FGamebaseError* Error)
{
    if (Error == nullptr)
    {
        return nullptr;
    }
    
    const TOptional<FString>& JsonString = Error->FindExtra(GamebaseErrorExtras::FailTransferAccount);
    if (!JsonString.IsSet())
    {
        return nullptr;
    }
    
    const FGamebaseTransferAccountFailInfoPtr TransferAccountFailInfo = MakeShared<FGamebaseTransferAccountFailInfo, ESPMode::ThreadSafe>();
    TransferAccountFailInfo->FromJson(JsonString.GetValue());
    
    return TransferAccountFailInfo;
}

void FGamebaseTransferAccountFailInfo::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("appId", AppId);
    NHN_JSON_SERIALIZE("id", Id);
    NHN_JSON_SERIALIZE("status", Status);
    NHN_JSON_SERIALIZE("failCount", FailCount);
    NHN_JSON_SERIALIZE("blockEndDate", BlockEndDate);
    NHN_JSON_SERIALIZE("regDate", RegDate);
    END_NHN_VO_SERIALIZER
}

FGamebaseForcingMappingTicketPtr FGamebaseForcingMappingTicket::From(const FGamebaseError* Error)
{
    if (Error == nullptr)
    {
        return nullptr;
    }
    
    const TOptional<FString>& JsonString = Error->FindExtra(GamebaseErrorExtras::ForcingMappingTicket);
    if (!JsonString.IsSet())
    {
        return nullptr;
    }
    
    const FGamebaseForcingMappingTicketPtr ForcingMappingTicket = MakeShared<FGamebaseForcingMappingTicket, ESPMode::ThreadSafe>();
    ForcingMappingTicket->FromJson(JsonString.GetValue());
    
    return ForcingMappingTicket;
}

void FGamebaseForcingMappingTicket::Serialize(FJsonSerializerBase& Serializer, const bool bFlatObject)
{
    BEGIN_NHN_VO_SERIALIZER
    NHN_JSON_SERIALIZE("userId", UserId);
    NHN_JSON_SERIALIZE("mappedUserId", MappedUserId);
    NHN_JSON_SERIALIZE("mappedUserValid", MappedUserValid);
    NHN_JSON_SERIALIZE("idPCode", IdPCode);
    NHN_JSON_SERIALIZE("forcingMappingKey", ForcingMappingKey);
    NHN_JSON_SERIALIZE("expirationDate", ExpirationDate);
    NHN_JSON_SERIALIZE("accessToken", AccessToken);
    NHN_JSON_SERIALIZE_OPTIONAL("subCode", SubCode);
    NHN_JSON_SERIALIZE_OPTIONAL_VARIANT_MAP("extraParams", ExtraParams);
    END_NHN_VO_SERIALIZER
}