﻿#include "SGamebaseWebBrowser.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseWebBrowserSchemeHandler.h"
#include "GamebaseWebBrowserTypes.h"
#include "GamebaseWebBrowserUtils.h"
#include "GamebaseWebBrowserViewModel.h"
#include "SWebBrowser.h"
#include "IWebBrowserWindow.h"
#include "IWebBrowserPopupFeatures.h"
#include "IWebBrowserDialog.h"
#include "Widgets/Images/SImage.h"
#include "WebBrowserModule.h"

class SGamebaseWebBrowserImpl final
    : public SGamebaseWebBrowser
{
public:
    virtual ~SGamebaseWebBrowserImpl() override;
    
    virtual void Construct(const FArguments& InArgs, const TSharedRef<FGamebaseWebBrowserViewModel>& InViewModel) override;

    virtual void Tick(const FGeometry& AllottedGeometry, const double InCurrentTime, const float InDeltaTime) override;
    virtual bool SupportsKeyboardFocus() const override;
    virtual FReply OnFocusReceived(const FGeometry& MyGeometry, const FFocusEvent& InFocusEvent) override;
    virtual FReply OnKeyDown(const FGeometry& MyGeometry, const FKeyEvent& InKeyEvent) override;

private:
    void ConstructWindowView(const FArguments& InArgs, const TSharedRef<FGamebaseWebBrowserViewModel>& InViewModel);
    void ConstructPopupView(const FArguments& InArgs, const TSharedRef<FGamebaseWebBrowserViewModel>& InViewModel);
    
    EVisibility IsVisibleForward() const;
    bool CanGoForward() const;
    
    EVisibility IsVisibleBack() const;
    bool CanGoBack() const;
    
    FReply OnForwardButtonClicked() const;
    FReply OnBackButtonClicked() const;
    FReply OnCloseButtonClicked() const;
    
    void AddWebOverlay(const TSharedRef<SWidget>& Content);
    void RemoveWebOverlay(const TSharedRef<SWidget>& Content);
    void CloseTopOverlayBrowser();
    bool IsShowingOverlay() const;
    
    void HandleLoadError();
    bool HandleLoadUrl(const FString& Method, const FString& Url, FString& OutResponse);
    void HandleBrowserUrlChanged(const FText& Url);
    void HandleTitleChanged(const FText& NewTitle);
    bool HandleBeforePopup(FString Url, FString Target);
    bool HandleBeforeBrowse(const FString& Url, const FWebNavigationRequest& Request);
    void HandleOverlayBrowserLoadError(TWeakPtr<IWebBrowserWindow> BrowserWindowPtr);
    bool HandleBrowserCreateWindow(const TWeakPtr<IWebBrowserWindow>& NewBrowserWindow, const TWeakPtr<IWebBrowserPopupFeatures>& PopupFeatures);
    bool HandleBrowserCloseWindow(const TWeakPtr<IWebBrowserWindow>& BrowserWindowPtr);
    EWebBrowserDialogEventResponse HandleShowDialog(const TWeakPtr<IWebBrowserDialog>& DialogPtr);
    bool HandleKey(const FKeyEvent& KeyEvent);
    bool HandleKeyChar(const FCharacterEvent& CharacterEvent);
    FReply HandleCloseOverlayClicked();
    EVisibility GetCloseOverlayVisibility() const;
    
private:
    TSharedPtr<FGamebaseWebBrowserViewModel> ViewModel;

    TSharedPtr<STextBlock> TitleWidget;
    
    TSharedPtr<SOverlay> BrowserContainer;
    TMap<TWeakPtr<IWebBrowserWindow>, TWeakPtr<SWebBrowserView>> BrowserOverlayWidgets;

    /* The persistent main login flow browser */
    TSharedPtr<SWebBrowserView> MainBrowser;

    TMap<TWeakPtr<IWebBrowserWindow>, TWeakPtr<SWindow>> BrowserWindowWidgets;
    
    /** Holds navigation requests that are requested outside of the game thread. */
    TArray<FString> NavigationQueue;
    FCriticalSection NavigationQueueCS;

    TArray<FGamebaseSchemeHandlerFactoryPtr> SchemeFactories;
    
    bool bEncounteredError = false;
    bool bOpenDevTools = false;
    const ISlateStyle* StyleSet = nullptr;

    FString Title;
    bool bUpdateTitle = false;
    bool bShowAlwaysButtons = false;
    bool bShowBackButton = false;
    bool bShowForwardButton = false;
};

SGamebaseWebBrowserImpl::~SGamebaseWebBrowserImpl()
{
    IWebBrowserSingleton* WebBrowserSingleton = IWebBrowserModule::Get().GetSingleton();
    for (const auto& Scheme : SchemeFactories)
    {
        if (!WebBrowserSingleton->UnregisterSchemeHandlerFactory(Scheme.Get()))
        {
            GAMEBASE_LOG_GLOBAL_WARNING("UnregisterSchemeHandlerFactory failed.");
        }
    }
}

void SGamebaseWebBrowserImpl::Construct(const FArguments& InArgs, const TSharedRef<FGamebaseWebBrowserViewModel>& InViewModel)
{
    bEncounteredError = false;
    bOpenDevTools = false;
    ViewModel = InViewModel;
    StyleSet = InArgs._StyleSet;

    const FString Url = ViewModel->GetUrl();
    const auto& Configuration = ViewModel->GetBrowserConfiguration();
    
    IWebBrowserSingleton* WebBrowserSingleton = IWebBrowserModule::Get().GetSingleton();
    for (const auto& SchemeUrl : ViewModel->GetSchemeList())
    {
        FGamebaseSchemeHandlerFactoryPtr SchemeHandler = MakeShared<FGamebaseSchemeHandlerFactory>();
        SchemeFactories.Add(SchemeHandler);
        
        FString Scheme, Domain;
        GamebaseWebBrowserUtils::ParseScheme(SchemeUrl, Scheme, Domain);
        if (!WebBrowserSingleton->RegisterSchemeHandlerFactory(Scheme, Domain, SchemeHandler.Get()))
        {
            GAMEBASE_LOG_GLOBAL_WARNING("RegisterSchemeHandlerFactory failed. (Scheme: %s, Domain: %s)", *Scheme, *Domain);
        }
    }

    switch (Configuration->ViewType)
    {
        case EGamebaseWebBrowserViewType::Window:
            ConstructWindowView(InArgs, InViewModel);
            break;
        
        case EGamebaseWebBrowserViewType::FloatingPopup:
            ConstructPopupView(InArgs, InViewModel);
            break;

        default:
            checkNoEntry();
            break;
    }

    if (Url.IsEmpty())
    {
        bEncounteredError = true;
    }
}

void SGamebaseWebBrowserImpl::Tick(const FGeometry& AllottedGeometry, const double InCurrentTime, const float InDeltaTime)
{
    check(IsInGameThread());

    if (bEncounteredError)
    {
        HandleLoadError();
    }

    {
        // Flush the navigation queue
        FScopeLock ScopeLock(&NavigationQueueCS);
        while(NavigationQueue.Num())
        {
            const FString Url = NavigationQueue[0];
            NavigationQueue.RemoveAt(0);
            HandleBeforePopup(Url, TEXT(""));
        }
    }
}

bool SGamebaseWebBrowserImpl::SupportsKeyboardFocus() const
{
    return true;
}

FReply SGamebaseWebBrowserImpl::OnFocusReceived(const FGeometry& MyGeometry, const FFocusEvent& InFocusEvent)
{
    FReply Reply = FReply::Handled();

    if (InFocusEvent.GetCause() != EFocusCause::Cleared)
    {
        Reply.SetUserFocus(MainBrowser.ToSharedRef(), InFocusEvent.GetCause());
    }

    return Reply;
}

FReply SGamebaseWebBrowserImpl::OnKeyDown(const FGeometry& MyGeometry, const FKeyEvent& InKeyEvent)
{
    if (InKeyEvent.GetKey() == EKeys::Escape)
    {
        // Close any overlays first.
        if (IsShowingOverlay())
        {
            CloseTopOverlayBrowser();
        }
        else
        {
            ViewModel->HandleRequestClose(TEXT("Escape"));
        }

        return FReply::Handled();
    }
    return FReply::Unhandled();
}

void SGamebaseWebBrowserImpl::ConstructWindowView(
    const FArguments& InArgs,
    const TSharedRef<FGamebaseWebBrowserViewModel>& InViewModel)
{
    bEncounteredError = false;
    bOpenDevTools = false;
    ViewModel = InViewModel;
    StyleSet = InArgs._StyleSet;

    const auto& Configuration = ViewModel->GetBrowserConfiguration();
    int32 NavigationBarHeight = 0;
    FColor NavigationBarColor;
    FColor TitleColor;
    FColor IconTintColor;
    
    const bool bIsNavigationBar = Configuration->Navigation.IsSet();
    if (bIsNavigationBar)
    {
        const auto& NavigationInfo = Configuration->Navigation.GetValue();
        Title = NavigationInfo.Title;
        bUpdateTitle = NavigationInfo.Title.IsEmpty();
        bShowAlwaysButtons = NavigationInfo.bVisibleButtons;
        bShowBackButton = NavigationInfo.bIsBackButton;
        bShowForwardButton = NavigationInfo.bIsForwardButton;
        NavigationBarHeight = NavigationInfo.BarHeight;
        NavigationBarColor = NavigationInfo.BarColor;
        TitleColor = NavigationInfo.TitleColor;
        IconTintColor = NavigationInfo.IconTintColor;
    }
    else
    {
        bUpdateTitle = false;
        bShowAlwaysButtons = false;
        bShowBackButton = false;
        bShowForwardButton = false;
    }
    
    const FString HomeUrl = ViewModel->GetUrl();
    
    FVector2D CurrentViewportSize;
    if (Configuration->ViewportSize == FVector2D::ZeroVector)
    {
        //GameViewport->GetViewportSize(CurrentViewportSize);
    }
    else
    {
        CurrentViewportSize = Configuration->ViewportSize;
    }
    
    const FVector2D ViewportSize(CurrentViewportSize.X, CurrentViewportSize.Y - NavigationBarHeight);
    
    SUserWidget::Construct(SUserWidget::FArguments()
    [
        SNew(SOverlay)
        + SOverlay::Slot()
        [
            SNew(SOverlay)
            + SOverlay::Slot()
            .HAlign(HAlign_Fill)
            .VAlign(VAlign_Fill)
            [
                SNew(SBox)
                .HAlign(HAlign_Fill)
                .VAlign(VAlign_Fill)
                [
                    SNew(SImage)
                    .Image(StyleSet->GetBrush("CefWebView.Background"))
                    .ColorAndOpacity(FLinearColor(Configuration->BackgroundColor))
                ]
            ]
            + SOverlay::Slot()
            .HAlign(HAlign_Fill)
            .VAlign(VAlign_Fill)
            [
                SNew(SBox)
                .HAlign(HAlign_Fill)
                .VAlign(VAlign_Fill)
                [
                    SNew(SButton)
                    .OnClicked(this, &SGamebaseWebBrowserImpl::OnCloseButtonClicked)
                    .ClickMethod(EButtonClickMethod::PreciseClick)
                    .ButtonColorAndOpacity(FColor::Transparent)
                    .IsFocusable(true)
                ]
            ]
            + SOverlay::Slot()
            [
                SNew(SBox)
                .VAlign(Configuration->ViewportSize.Y == 0.0f ? VAlign_Fill : VAlign_Center)
                .HAlign(Configuration->ViewportSize.Y == 0.0f ? HAlign_Fill : HAlign_Center)
                .WidthOverride(CurrentViewportSize.X)
                .HeightOverride(CurrentViewportSize.Y)
                [
                    SNew(SVerticalBox)
                    + SVerticalBox::Slot()
                    .VAlign(VAlign_Top)
                    .HAlign(HAlign_Fill)
                    .AutoHeight()
                    [
                        SNew(SOverlay)
                        .Visibility(bIsNavigationBar ? EVisibility::Visible : EVisibility::Collapsed)
                        + SOverlay::Slot()
                        .VAlign(VAlign_Fill)
                        .HAlign(HAlign_Fill)
                        [
                            SNew(SBox)
                            .HAlign(HAlign_Fill)
                            .VAlign(VAlign_Fill)
                            .HeightOverride(NavigationBarHeight)
                            [
                                SNew(SImage)
                                .Image(StyleSet->GetBrush("CefWebView.Navigation.Background"))
                                .ColorAndOpacity(NavigationBarColor)
                            ]
                        ]
                        + SOverlay::Slot()
                        .VAlign(VAlign_Fill)
                        .HAlign(HAlign_Fill)
                        [
                            SNew(SHorizontalBox)
                            +SHorizontalBox::Slot()
                            .HAlign(HAlign_Left)
                            [
                                SNew(SHorizontalBox)
                                +SHorizontalBox::Slot()
                                .HAlign(HAlign_Left)
                                [
                                    SNew(SButton)
                                    .Visibility(this, &SGamebaseWebBrowserImpl::IsVisibleBack)
                                    .IsEnabled(this, &SGamebaseWebBrowserImpl::CanGoBack)
                                    .OnClicked(this, &SGamebaseWebBrowserImpl::OnBackButtonClicked)
                                    .ButtonColorAndOpacity(FColor::Transparent)
                                    .ClickMethod(EButtonClickMethod::MouseDown)
                                    .IsFocusable(false)
                                    [
                                        SNew(SImage)
                                        .Image(StyleSet->GetBrush("CefWebView.BackButton.Image"))
                                        .ColorAndOpacity(IconTintColor)
                                    ]
                                ]
                                +SHorizontalBox::Slot()
                                .HAlign(HAlign_Right)
                                [
                                    SNew(SButton)
                                    .Visibility(this, &SGamebaseWebBrowserImpl::IsVisibleForward)
                                    .IsEnabled(this, &SGamebaseWebBrowserImpl::CanGoForward)
                                    .OnClicked(this, &SGamebaseWebBrowserImpl::OnForwardButtonClicked)
                                    .ButtonColorAndOpacity(FColor::Transparent)
                                    .ClickMethod(EButtonClickMethod::MouseDown)
                                    .IsFocusable(false)
                                    [
                                        SNew(SImage)
                                        .Image(StyleSet->GetBrush("CefWebView.BackButton.Image"))
                                        .RenderTransformPivot(FVector2D(0.5f, 0.5f))
                                        .RenderTransform(FSlateRenderTransform(FQuat2D(FMath::DegreesToRadians(180.0f))))
                                        .ColorAndOpacity(IconTintColor)
                                    ]
                                ]
                            ]
                            +SHorizontalBox::Slot()
                            .HAlign(HAlign_Center)
                            .VAlign(VAlign_Center)
                            [
                                SAssignNew(TitleWidget, STextBlock)
                                .Text(FText::FromString(Title))
                                .TextStyle(StyleSet, "CefWebView.SubTitleText")
                                .ColorAndOpacity(TitleColor)
                            ]
                            +SHorizontalBox::Slot()
                            .HAlign(HAlign_Right)
                            [
                                SNew(SButton)
                                .OnClicked(this, &SGamebaseWebBrowserImpl::OnCloseButtonClicked)
                                .ClickMethod(EButtonClickMethod::MouseDown)
                                .ButtonColorAndOpacity(FColor::Transparent)
                                .IsFocusable(false)
                                [
                                    SNew(SImage)
                                    .Image(StyleSet->GetBrush("CefWebView.CloseButton.Image"))
                                    .ColorAndOpacity(IconTintColor)
                                ]
                            ]
                        ]
                    ]
                    + SVerticalBox::Slot()
                    .VAlign(VAlign_Fill)
                    .HAlign(HAlign_Fill)
                    [
                        SNew(SOverlay)
                        +SOverlay::Slot()
                        [
                            SAssignNew(BrowserContainer, SOverlay)
                            + SOverlay::Slot()
                            .HAlign(HAlign_Fill)
                            .VAlign(VAlign_Fill)
                            [
                                SAssignNew(MainBrowser, SWebBrowserView)
                                .ShowErrorMessage(false)
                                .SupportsTransparency(false)
                                .InitialURL(HomeUrl)
                                .BackgroundColor(FColor::White)
                                .ViewportSize(ViewportSize)
                                .OnLoadError(this, &SGamebaseWebBrowserImpl::HandleLoadError)
                                .OnLoadUrl(this, &SGamebaseWebBrowserImpl::HandleLoadUrl)
                                .OnUrlChanged(this, &SGamebaseWebBrowserImpl::HandleBrowserUrlChanged)
                                .OnTitleChanged(this, &SGamebaseWebBrowserImpl::HandleTitleChanged)
                                .OnBeforePopup(this, &SGamebaseWebBrowserImpl::HandleBeforePopup)
                                .OnBeforeNavigation(this, &SGamebaseWebBrowserImpl::HandleBeforeBrowse)
                                .OnCreateWindow(this, &SGamebaseWebBrowserImpl::HandleBrowserCreateWindow)
                                .OnCloseWindow(this, &SGamebaseWebBrowserImpl::HandleBrowserCloseWindow)
                                .OnShowDialog(this, &SGamebaseWebBrowserImpl::HandleShowDialog)
                                .OnUnhandledKeyDown(this, &SGamebaseWebBrowserImpl::HandleKey)
                                .OnUnhandledKeyUp(this, &SGamebaseWebBrowserImpl::HandleKey)
                                .OnUnhandledKeyChar(this, &SGamebaseWebBrowserImpl::HandleKeyChar)
                                .ContextSettings(ViewModel->GetBrowserContextSettings())
                            ]
                        ]
                    ]
                ]
            ]
        ]
    ]);

    if (HomeUrl.IsEmpty())
    {
        bEncounteredError = true;
    }
}

void SGamebaseWebBrowserImpl::ConstructPopupView(
    const FArguments& InArgs,
    const TSharedRef<FGamebaseWebBrowserViewModel>& InViewModel)
{
    bEncounteredError = false;
    bOpenDevTools = false;
    ViewModel = InViewModel;
    StyleSet = InArgs._StyleSet;

    bUpdateTitle = false;
    bShowAlwaysButtons = false;
    bShowBackButton = false;
    bShowForwardButton = false;
    
    const auto& Configuration = ViewModel->GetBrowserConfiguration();
    const FString HomeUrl = ViewModel->GetUrl();
    
    FVector2D CurrentViewportSize;
    if (Configuration->ViewportSize == FVector2D::ZeroVector)
    {
        //GameViewport->GetViewportSize(CurrentViewportSize);
    }
    else
    {
        CurrentViewportSize = Configuration->ViewportSize;
    }

    const float DpiScale = FSlateApplication::Get().GetApplicationScale();
    const FVector2D ViewportSize(CurrentViewportSize.X, CurrentViewportSize.Y);

    constexpr float ButtonSize = 24;
    constexpr float SpaceSize = 8;
    constexpr float PaddingSize = ButtonSize + SpaceSize;

    const FSlateRenderTransform InverseDPIScale(FScale2D(1.0f / DpiScale));
    
    SUserWidget::Construct(SUserWidget::FArguments()
    [
        SNew(SOverlay)
            + SOverlay::Slot()
            .HAlign(HAlign_Fill)
            .VAlign(VAlign_Fill)
            [
                SNew(SImage)
                    .Image(StyleSet->GetBrush("CefWebView.Background"))
                    .ColorAndOpacity(FLinearColor(Configuration->BackgroundColor))
            ]
            + SOverlay::Slot()
            [
                SNew(SBox)
                .RenderTransform(InverseDPIScale)
                .VAlign(Configuration->ViewportSize.Y == 0.0f ? VAlign_Fill : VAlign_Center)
                .HAlign(Configuration->ViewportSize.Y == 0.0f ? HAlign_Fill : HAlign_Center)
                .WidthOverride(ViewportSize.X)
                .HeightOverride(ViewportSize.Y + PaddingSize * 2)
                .Padding(FMargin(0, SpaceSize, 0, PaddingSize))
                [
                    SNew(SVerticalBox)
                    + SVerticalBox::Slot()
                    .VAlign(VAlign_Top)
                    .HAlign(HAlign_Right)
                    .Padding(0, SpaceSize, -SpaceSize, 8)
                    .AutoHeight()
                    [
                        SNew(SButton)
                        .VAlign(VAlign_Top)
                        .HAlign(HAlign_Right)
                        .OnClicked(this, &SGamebaseWebBrowserImpl::OnCloseButtonClicked)
                        .ClickMethod(EButtonClickMethod::MouseDown)
                        .ButtonColorAndOpacity(FColor::Transparent)
                        .IsFocusable(false)
                        [
                            SNew(SImage)
                            .Image(StyleSet->GetBrush("CefWebView.ClosePopupButton.Image"))
                        ]
                    ]
                    + SVerticalBox::Slot()
                    .VAlign(VAlign_Fill)
                    .HAlign(HAlign_Fill)
                    [
                        SAssignNew(BrowserContainer, SOverlay)
                        + SOverlay::Slot()
                        .HAlign(HAlign_Fill)
                        .VAlign(VAlign_Fill)
                        [
                            SNew(SBox)
                            .WidthOverride(CurrentViewportSize.X)
                            .HeightOverride(CurrentViewportSize.Y)
                            [
                                SAssignNew(MainBrowser, SWebBrowserView)
                                .ShowErrorMessage(false)
                                .SupportsTransparency(false)
                                .InitialURL(HomeUrl)
                                .BackgroundColor(FColor::White)
                                .ViewportSize(ViewportSize)
                                .OnLoadError(this, &SGamebaseWebBrowserImpl::HandleLoadError)
                                .OnLoadUrl(this, &SGamebaseWebBrowserImpl::HandleLoadUrl)
                                .OnUrlChanged(this, &SGamebaseWebBrowserImpl::HandleBrowserUrlChanged)
                                .OnTitleChanged(this, &SGamebaseWebBrowserImpl::HandleTitleChanged)
                                .OnBeforePopup(this, &SGamebaseWebBrowserImpl::HandleBeforePopup)
                                .OnBeforeNavigation(this, &SGamebaseWebBrowserImpl::HandleBeforeBrowse)
                                .OnCreateWindow(this, &SGamebaseWebBrowserImpl::HandleBrowserCreateWindow)
                                .OnCloseWindow(this, &SGamebaseWebBrowserImpl::HandleBrowserCloseWindow)
                                .OnShowDialog(this, &SGamebaseWebBrowserImpl::HandleShowDialog)
                                .OnUnhandledKeyDown(this, &SGamebaseWebBrowserImpl::HandleKey)
                                .OnUnhandledKeyUp(this, &SGamebaseWebBrowserImpl::HandleKey)
                                .OnUnhandledKeyChar(this, &SGamebaseWebBrowserImpl::HandleKeyChar)
                                .ContextSettings(ViewModel->GetBrowserContextSettings())
                            ]
                        ]
                    ]
                ]
            ]
        ]
    );

    if (HomeUrl.IsEmpty())
    {
        bEncounteredError = true;
    }
}

EVisibility SGamebaseWebBrowserImpl::IsVisibleForward() const
{
    if (!bShowForwardButton)
    {
        return EVisibility::Collapsed;
    }
    
    if (bShowAlwaysButtons)
    {
        return EVisibility::Visible;
    }
    
    return CanGoForward() ? EVisibility::Visible : EVisibility::Collapsed;
}

bool SGamebaseWebBrowserImpl::CanGoForward() const
{
    return MainBrowser.IsValid() && MainBrowser->CanGoForward();
}

EVisibility SGamebaseWebBrowserImpl::IsVisibleBack() const
{
    if (!bShowBackButton)
    {
        return EVisibility::Collapsed;
    }
    
    if (bShowAlwaysButtons)
    {
        return EVisibility::Visible;
    }
    
    return CanGoForward() ? EVisibility::Visible : EVisibility::Collapsed;
}

bool SGamebaseWebBrowserImpl::CanGoBack() const
{
    return MainBrowser.IsValid() && MainBrowser->CanGoBack();
}

FReply SGamebaseWebBrowserImpl::OnForwardButtonClicked() const
{
    if (MainBrowser.IsValid() && MainBrowser->CanGoForward())
    {
        MainBrowser->GoForward();
    }
    
    return FReply::Handled();
}

FReply SGamebaseWebBrowserImpl::OnBackButtonClicked() const
{
    if (MainBrowser.IsValid() && MainBrowser->CanGoBack())
    {
        MainBrowser->GoBack();
    }
    
    return FReply::Handled();
}

FReply SGamebaseWebBrowserImpl::OnCloseButtonClicked() const
{
    if (IsShowingOverlay())
    {
        const_cast<SGamebaseWebBrowserImpl&>(*this).CloseTopOverlayBrowser();
    }
    else
    {
        ViewModel->HandleRequestClose(TEXT("Escape"));
    }

    return FReply::Handled();
}

void SGamebaseWebBrowserImpl::AddWebOverlay(const TSharedRef<SWidget>& Content)
{
    if (BrowserContainer.IsValid())
    {
        BrowserContainer->AddSlot()
                        .Padding(35.0f)
        [
            Content
        ];
    }
}

void SGamebaseWebBrowserImpl::RemoveWebOverlay(const TSharedRef<SWidget>& Content)
{
    if (IsShowingOverlay())
    {
        BrowserContainer->RemoveSlot(Content);
    }
}

void SGamebaseWebBrowserImpl::CloseTopOverlayBrowser()
{
    if (IsShowingOverlay())
    {
        // Find the browser widget in the top most overlay slot
        const auto BrowserContainerChildren = BrowserContainer->GetChildren();
        TSharedRef<SWidget> LastSlotWidget = BrowserContainerChildren->GetChildAt(BrowserContainerChildren->Num() - 1);

        // Find and call close on the IWebBrowserWindow associated with the widget in the last overlay slot
        for (auto& Kvp : BrowserOverlayWidgets)
        {
            const TSharedPtr<SWebBrowserView> FoundBrowserViewWidget = Kvp.Value.Pin();
            if (FoundBrowserViewWidget.IsValid() && FoundBrowserViewWidget == LastSlotWidget)
            {
                TSharedPtr<IWebBrowserWindow> WebBrowserWindow = Kvp.Key.Pin();
                if (WebBrowserWindow->IsValid() && !WebBrowserWindow->IsClosing())
                {
                    WebBrowserWindow->CloseBrowser(false);
                }
            }
        }
    }
}

bool SGamebaseWebBrowserImpl::IsShowingOverlay() const
{
    // We are showing overlaid browsers if the browser container contains more than one slot.
    return BrowserContainer->GetNumWidgets() > 1;
}

void SGamebaseWebBrowserImpl::HandleLoadError()
{
    //@todo: Should forward whatever info we can about the load error
    ViewModel->HandleLoadError();
    bEncounteredError = false;
}

bool SGamebaseWebBrowserImpl::HandleLoadUrl(const FString& Method, const FString& Url, FString& OutResponse)
{
    return false;
}

void SGamebaseWebBrowserImpl::HandleBrowserUrlChanged(const FText& Url)
{
    if (true) // HandleBrowserBeforeBrowse seems to do all that is required atm
    {
        FString CurrentURL = MainBrowser->GetUrl();
        //UE_LOG(LogLoginFlow, Log, TEXT("HandleBrowserUrlChanged Current: %s New: %s"), *CurrentURL, *Url.ToString());

        ViewModel->HandleBrowserUrlChanged(Url);
    }
}

void SGamebaseWebBrowserImpl::HandleTitleChanged(const FText& NewTitle)
{
    if (bUpdateTitle)
    {
        TitleWidget->SetText(FText::FromString(NewTitle.ToString()));
    }
}

bool SGamebaseWebBrowserImpl::HandleBeforePopup(FString Url, FString Target)
{
    // Chrome debug tools are allowed to open a popup window.
    if (Url.Contains(TEXT("chrome-devtools")))
    {
        bOpenDevTools = true;
        return false;
    }

    // Popups that provide valid Target will be spawned as an overlay on top of the login flow.
    if (!Target.IsEmpty() && 
        !Target.StartsWith("_blank") &&
        !Target.StartsWith("blank"))
    {
        return false;
    }

    // Remaining popups are redirected to external browser.
    if (IsInGameThread())
    {
        ViewModel->HandleNavigation(Url);
    }
    else
    {
        FScopeLock ScopeLock(&NavigationQueueCS);
        NavigationQueue.Add(Url);
    }
    
    return true;
}

bool SGamebaseWebBrowserImpl::HandleBeforeBrowse(const FString& Url, const FWebNavigationRequest& Request)
{
    //UE_LOG(LogLoginFlow, Log, TEXT("HandleBrowserBeforeBrowse Main:%d Redirect:%d URL: %s"), Request.bIsMainFrame, Request.bIsRedirect, *Url);
    return ViewModel->HandleBeforeBrowse(Url);
}

void SGamebaseWebBrowserImpl::HandleOverlayBrowserLoadError(TWeakPtr<IWebBrowserWindow> BrowserWindowPtr)
{
    TSharedPtr<IWebBrowserWindow> WebBrowserWindow = BrowserWindowPtr.Pin();
    if (WebBrowserWindow->IsValid() && !WebBrowserWindow->IsClosing())
    {
        // We encountered an error so just close the browser overlay.
        // @todo: Relay this error info to the user.
        WebBrowserWindow->CloseBrowser(false);
    }
}

bool SGamebaseWebBrowserImpl::HandleBrowserCreateWindow(const TWeakPtr<IWebBrowserWindow>& NewBrowserWindow, const TWeakPtr<IWebBrowserPopupFeatures>& PopupFeatures)
{
    TSharedPtr<IWebBrowserPopupFeatures> PopupFeaturesSP = PopupFeatures.Pin();
    check(PopupFeatures.IsValid())

    // All allowed popups, with the exception of the dev tools, are spawned as an overlay on top of the login flow browser.
    if (bOpenDevTools)
    {
        bOpenDevTools = false;
        // Dev tools spawn in a new window.
        TSharedPtr<SWindow> ParentWindow = FSlateApplication::Get().FindWidgetWindow(SharedThis(this));
        if (ParentWindow.IsValid())
        {
            const int PosX = PopupFeaturesSP->IsXSet() ? PopupFeaturesSP->GetX() : 100;
            const int PosY = PopupFeaturesSP->IsYSet() ? PopupFeaturesSP->GetY() : 100;
            const FVector2D BrowserWindowPosition(PosX, PosY);

            const int Width = PopupFeaturesSP->IsWidthSet() ? PopupFeaturesSP->GetWidth() : 800;
            const int Height = PopupFeaturesSP->IsHeightSet() ? PopupFeaturesSP->GetHeight() : 600;
            const FVector2D BrowserWindowSize(Width, Height);

            const ESizingRule SizeingRule = PopupFeaturesSP->IsResizable() ? ESizingRule::UserSized : ESizingRule::FixedSize;

            TSharedPtr<IWebBrowserWindow> NewBrowserWindowSP = NewBrowserWindow.Pin();
            check(NewBrowserWindowSP.IsValid());

            TSharedRef<SWindow> BrowserWindowWidget =
                SNew(SWindow)
                .Title(FText::GetEmpty())
                .ClientSize(BrowserWindowSize)
                .ScreenPosition(BrowserWindowPosition)
                .AutoCenter(EAutoCenter::None)
                .SizingRule(SizeingRule)
                .SupportsMaximize(SizeingRule != ESizingRule::FixedSize)
                .SupportsMinimize(SizeingRule != ESizingRule::FixedSize)
                .HasCloseButton(true)
                .CreateTitleBar(true)
                .IsInitiallyMaximized(PopupFeaturesSP->IsFullscreen())
                .LayoutBorder(FMargin(0));

            // Setup browser widget.
            TSharedPtr<SWebBrowser> BrowserWidget;
                BrowserWindowWidget->SetContent(
                    SNew(SBorder)
                    .VAlign(VAlign_Fill)
                    .HAlign(HAlign_Fill)
                    .Padding(0)
                    [
                    SAssignNew(BrowserWidget, SWebBrowser, NewBrowserWindowSP)
                    .ShowControls(PopupFeaturesSP->IsToolBarVisible())
                    .ShowAddressBar(PopupFeaturesSP->IsLocationBarVisible())
                    .OnCreateWindow(this, &SGamebaseWebBrowserImpl::HandleBrowserCreateWindow)
                    .OnCloseWindow(this, &SGamebaseWebBrowserImpl::HandleBrowserCloseWindow)
                    .OnShowDialog(this, &SGamebaseWebBrowserImpl::HandleShowDialog)
                ]);

            // Setup some OnClose stuff.
        {
            struct FLocal
            {
                static void RequestDestroyWindowOverride(const TSharedRef<SWindow>& Window, TWeakPtr<IWebBrowserWindow> BrowserWindowPtr)
                {
                    TSharedPtr<IWebBrowserWindow> BrowserWindow = BrowserWindowPtr.Pin();
                    if (BrowserWindow.IsValid())
                    {
                        if (BrowserWindow->IsClosing())
                        {
                            FSlateApplicationBase::Get().RequestDestroyWindow(Window);
                        }
                        else
                        {
                            // Notify the browser window that we would like to close it.  On the CEF side, this will 
                            //  result in a call to FWebBrowserHandler::DoClose only if the JavaScript onbeforeunload
                            //  event handler allows it.
                            BrowserWindow->CloseBrowser(false);
                        }
                    }
                }
            };

            BrowserWindowWidget->SetRequestDestroyWindowOverride(FRequestDestroyWindowOverride::CreateStatic(&FLocal::RequestDestroyWindowOverride, TWeakPtr<IWebBrowserWindow>(NewBrowserWindow)));
        }

        FSlateApplication::Get().AddWindow(BrowserWindowWidget);
        BrowserWindowWidget->BringToFront();
        FSlateApplication::Get().SetKeyboardFocus(BrowserWidget, EFocusCause::SetDirectly);

        BrowserWindowWidgets.Add(NewBrowserWindow, BrowserWindowWidget);
        return true;
        }
    }
    else
    {
        TSharedPtr<IWebBrowserWindow> NewBrowserWindowSP = NewBrowserWindow.Pin();
        check(NewBrowserWindowSP.IsValid());

        TSharedRef<SWebBrowserView> NewBrowserToOverlay =
            SNew(SWebBrowserView, NewBrowserWindowSP)
            .ShowErrorMessage(false)
            .SupportsTransparency(true)
            .OnLoadError(this, &SGamebaseWebBrowserImpl::HandleOverlayBrowserLoadError, NewBrowserWindow)
            .OnBeforePopup(this, &SGamebaseWebBrowserImpl::HandleBeforePopup)
            .OnCreateWindow(this, &SGamebaseWebBrowserImpl::HandleBrowserCreateWindow)
            .OnCloseWindow(this, &SGamebaseWebBrowserImpl::HandleBrowserCloseWindow)
            .OnBeforeNavigation(this, &SGamebaseWebBrowserImpl::HandleBeforeBrowse)
            .OnShowDialog(this, &SGamebaseWebBrowserImpl::HandleShowDialog);

        AddWebOverlay(NewBrowserToOverlay);
        BrowserOverlayWidgets.Add(NewBrowserWindow, NewBrowserToOverlay);

        return true;
    }

    return false;
}

bool SGamebaseWebBrowserImpl::HandleBrowserCloseWindow(const TWeakPtr<IWebBrowserWindow>& BrowserWindowPtr)
{
    TSharedPtr<IWebBrowserWindow> BrowserWindow = BrowserWindowPtr.Pin();
    if (BrowserWindow.IsValid())
    {
        if(!BrowserWindow->IsClosing())
        {
            // If the browser is not set to close, we tell the browser to close which will call back into this handler function.
            BrowserWindow->CloseBrowser(false);
        }
        else
        {
            // Close any matching overlay
            const TWeakPtr<SWebBrowserView>* FoundBrowserViewWidgetPtr = BrowserOverlayWidgets.Find(BrowserWindow);
            if (FoundBrowserViewWidgetPtr != nullptr)
            {
                TSharedPtr<SWebBrowserView> FoundBrowserViewWidget = FoundBrowserViewWidgetPtr->Pin();
                if (FoundBrowserViewWidget.IsValid())
                {
                    RemoveWebOverlay(FoundBrowserViewWidget.ToSharedRef());
                }
                BrowserOverlayWidgets.Remove(BrowserWindow);
                return true;
            }

            // Close any matching window
            const TWeakPtr<SWindow>* FoundBrowserWindowWidgetPtr = BrowserWindowWidgets.Find(BrowserWindow);
            if(FoundBrowserWindowWidgetPtr != nullptr)
            {
                TSharedPtr<SWindow> FoundBrowserWindowWidget = FoundBrowserWindowWidgetPtr->Pin();
                if(FoundBrowserWindowWidget.IsValid())
                {
                    FoundBrowserWindowWidget->RequestDestroyWindow();
                }
                BrowserWindowWidgets.Remove(BrowserWindow);
                return true;
            }
        }
    }

    return false;
}

EWebBrowserDialogEventResponse SGamebaseWebBrowserImpl::HandleShowDialog(const TWeakPtr<IWebBrowserDialog>& DialogPtr)
{
    // @todo: Due to an OS X crash, we continue all dialogs with the default action to prevent them from displaying using native windows.  In the future, we should add custom handlers/ui for these dialogs.
    EWebBrowserDialogEventResponse WebDialogHandling = EWebBrowserDialogEventResponse::Continue;
    return WebDialogHandling;
}

bool SGamebaseWebBrowserImpl::HandleKey(const FKeyEvent& KeyEvent)
{
    return ViewModel->ShouldConsumeInput();
}

bool SGamebaseWebBrowserImpl::HandleKeyChar(const FCharacterEvent& CharacterEvent)
{
    return ViewModel->ShouldConsumeInput();
}

FReply SGamebaseWebBrowserImpl::HandleCloseOverlayClicked()
{
    CloseTopOverlayBrowser();
    return FReply::Handled();
}

EVisibility SGamebaseWebBrowserImpl::GetCloseOverlayVisibility() const
{
    return IsShowingOverlay() ? EVisibility::Visible : EVisibility::Hidden;
}

TSharedRef<SGamebaseWebBrowser> SGamebaseWebBrowser::New()
{
    return MakeShareable(new SGamebaseWebBrowserImpl());
}
