﻿#include "GamebaseWebBrowserViewModel.h"

#include "GamebaseErrorCode.h"
#include "IWebBrowserSingleton.h"

class FGamebaseWebBrowserViewModelImpl final
    : public FGamebaseWebBrowserViewModel
{
public:
    virtual ~FGamebaseWebBrowserViewModelImpl() override = default;
    
    virtual FString GetUrl() override;
    virtual const TSharedRef<FGamebaseWebBrowserWidgetSettings>& GetBrowserConfiguration() const override;
    virtual const TOptional<FBrowserContextSettings> GetBrowserContextSettings() const override;
    virtual const TArray<FString> GetSchemeList() const override;
    virtual bool ShouldConsumeInput() const override;
    
    virtual void HandleRequestClose(const FString& CloseInfo) override;
    virtual void HandleLoadError() override;
    virtual bool HandleBrowserUrlChanged(const FText& Url) override;
    virtual bool HandleBeforeBrowse(const FString& Url) override;
    virtual bool HandleNavigation(const FString& Url) override;
    
    FGamebaseWebBrowserViewModelImpl(
        const FString& StartingUrl,
        const TSharedRef<FGamebaseWebBrowserWidgetSettings>& WidgetSettings,
        const TSharedRef<FGamebaseWebBrowserContextSettings>& ContextSettings,
        const TArray<FString>& SchemeList,
        const FCloseFunc& CloseCallback,
        const FErrorFunc& ErrorCallback,
        const FSchemeFunc& SchemeCallback,
        const bool bInConsumeInput)
        : StartingUrl(StartingUrl)
        , WidgetSettings(WidgetSettings)
        , ContextSettings(ContextSettings)
        , SchemeList(SchemeList)
        , CloseCallback(CloseCallback)
        , ErrorCallback(ErrorCallback)
        , SchemeCallback(SchemeCallback)
        , bConsumeInput(bInConsumeInput)
    { }
    
private:
    const FString StartingUrl;
    const TSharedRef<FGamebaseWebBrowserWidgetSettings> WidgetSettings;
    const TSharedRef<FGamebaseWebBrowserContextSettings> ContextSettings;
    const TArray<FString> SchemeList;
    const FCloseFunc CloseCallback;
    const FErrorFunc ErrorCallback;
    const FSchemeFunc SchemeCallback;
    bool bConsumeInput;
};

FString FGamebaseWebBrowserViewModelImpl::GetUrl()
{
    return StartingUrl;
}

const TSharedRef<FGamebaseWebBrowserWidgetSettings>& FGamebaseWebBrowserViewModelImpl::GetBrowserConfiguration() const
{
    return WidgetSettings;
}

const TOptional<FBrowserContextSettings> FGamebaseWebBrowserViewModelImpl::GetBrowserContextSettings() const
{
    FBrowserContextSettings Result { ContextSettings->Id };
    
    return TOptional<FBrowserContextSettings>();
}

const TArray<FString> FGamebaseWebBrowserViewModelImpl::GetSchemeList() const
{
    return SchemeList;
}

void FGamebaseWebBrowserViewModelImpl::HandleRequestClose(const FString& CloseInfo)
{
    CloseCallback();
}

void FGamebaseWebBrowserViewModelImpl::HandleLoadError()
{
    ErrorCallback(GamebaseErrorCode::WEBVIEW_UNKNOWN_ERROR, TEXT("Failed to load web browser."));
}

bool FGamebaseWebBrowserViewModelImpl::HandleBrowserUrlChanged(const FText& Url)
{
    return false;
}

bool FGamebaseWebBrowserViewModelImpl::HandleBeforeBrowse(const FString& Url)
{
    for (const FString& Scheme : SchemeList)
    {
        if (Url.StartsWith(Scheme))
        {
            SchemeCallback(Url);
            break;
        }
    }
    
    return false;
}

bool FGamebaseWebBrowserViewModelImpl::HandleNavigation(const FString& Url)
{
    constexpr bool bHandled = true;
    check(!Url.IsEmpty())
    FPlatformProcess::LaunchURL(*Url, nullptr, nullptr);
    return bHandled;
}

bool FGamebaseWebBrowserViewModelImpl::ShouldConsumeInput() const
{
    return bConsumeInput;
}

TSharedRef<FGamebaseWebBrowserViewModel> FGamebaseWebBrowserViewModel::Create(
    const FString& Url,
    const TSharedRef<FGamebaseWebBrowserWidgetSettings>& BrowserConfiguration,
    const TSharedRef<FGamebaseWebBrowserContextSettings>& BrowserContextSettings,
    const TArray<FString>& SchemeList,
    const FCloseFunc& OnRequestClose,
    const FErrorFunc& OnRedirectURL,
    const FSchemeFunc& OnSchemeEvent,
    const bool bConsumeInput)
{
    TSharedRef<FGamebaseWebBrowserViewModelImpl> ViewModel = MakeShareable(
        new FGamebaseWebBrowserViewModelImpl(
                Url,
                BrowserConfiguration,
                BrowserContextSettings,
                SchemeList,
                OnRequestClose,
                OnRedirectURL,
                OnSchemeEvent,
                bConsumeInput));
    //ViewModel->Initialize();
    return ViewModel;
}