#include "GpWindowsIapManager.h"

#include <string>

#include "GamebaseDebugLogger.h"
#include "GamebaseGameThreadDispatcher.h"
#include "GamebaseStandalonePurchaseConstants.h"
#include "GamebaseSystemUtils.h"
#include "GamebaseVerifySignatureHelper.h"
#include "GpWindowsIapResultCode.h"
#include "Constants/GamebaseErrorCode.h"

namespace GamebaseIAP
{
    constexpr float AsyncTimeout = 30;
    const FString DllFileName = TEXT("IapSDK.dll");
    const FString IapSdkDomain = TEXT("IAP_SDK");
}

namespace GamebaseErrorCallback
{
    FGamebaseError CreateIapInnerError(const int32 Code, const FString& Message)
    {
        return FGamebaseError(Code, Message, GamebaseIAP::IapSdkDomain);
    }

    void Execute(const FGamebaseIapCallbackDelegate& Callback, const int32 Code, const FString& Message, const int32 GamebaseErrorCode, const FString& GamebaseMessage)
    {
        FGamebaseError Error(
            GamebaseErrorCode,
            GamebaseMessage,
            GamebasePurchaseShared::Domain.ToString(),
            CreateIapInnerError(Code, Message));
        Callback.ExecuteIfBound(MakeShared<FGamebaseError, ESPMode::ThreadSafe>(Error));
    }

    void Execute(const FGamebaseErrorDelegate& Callback, const int32 Code, const FString& Message, const int32 GamebaseErrorCode, const FString& GamebaseMessage)
    {
        const FGamebaseError Error(
            GamebaseErrorCode,
            GamebaseMessage,
            GamebasePurchaseShared::Domain.ToString(),
            CreateIapInnerError(Code, Message));
        Callback.ExecuteIfBound(&Error);
    }

    void Execute(const FGamebaseIapResultFunc& Callback, const int32 Code, const FString& Message, const int32 GamebaseErrorCode, const FString& GamebaseMessage)
    {
        const FGamebaseError Error(
            GamebaseErrorCode,
            GamebaseMessage,
            GamebasePurchaseShared::Domain.ToString(),
            CreateIapInnerError(Code, Message));
        Callback(FGamebaseIapResult(Error));
    }
}
using TGpIapCallbackMethod = void(*)(int Code, const char* Response);
using TGpIapNotifyHandlerMethod = void(*)(int Code, const char* Message, const char* Reserve);

using TGpIapInitialize = void(*)(const char* SdkConfig, TGpIapCallbackMethod Callback);
using TGpIapUpdate = void(*)();
using TGpIapRelease = void(*)();
using TGpIapSetDebugMode = void(*)(bool bIsDebugMode);

using TGpIapSetUserInfo = void(*)(char* GamebaseUserId, char* GamebaseAccessToken);
using TGpIapLogin = void(*)(TGpIapCallbackMethod Callback);
using TGpIapLogout = void(*)(TGpIapCallbackMethod Callback);

using TGpIapAddListenerEvent = void(*)(TGpIapNotifyHandlerMethod NotifyHandler);
using TGpIapRemoveListenerEvent = void(*)();

using TGpIapRequestItemListPurchasable = void(*)(TGpIapCallbackMethod Callback);
using TGpIapRequestItemListOfNotConsumed = void(*)(bool bAllStores, TGpIapCallbackMethod Callback);
using TGpIapRequestPurchaseProductId = void(*)(const char* MarketItemId, const char* GamebasePayload, TGpIapCallbackMethod Callback);
using TGpIapRequestPurchaseProductIdWithPayload = void(*)(const char* MarketItemId, const char* GamebasePayload,
                                                          const char* DeveloperPayload, TGpIapCallbackMethod Callback);
using TGpIapRequestReprocPurchased = void(*)(const char* GamebaseItemList, TGpIapCallbackMethod Callback);

using TGpIapLoginWithDevAuthTool = void(*)(const char* Addr, const char* CredentialName, TGpIapCallbackMethod Callback);
using TGpIapRequestPurchaseEpicGameOffer = void(*)(const char* GameOfferId, TGpIapCallbackMethod Callback);
using TGpIapServerConsumeForTest = void(*)(const char* PaymentSeq, const char* AccessToken, TGpIapCallbackMethod Callback);

#if WITH_GAMEBASE_EOS_SHARED
using TGpIapSetEosPlatformInstance = void(*)(EOS_HPlatform EOSPlatform);
#endif

TGpIapInitialize GpIapInitialize = nullptr;
TGpIapUpdate GpIapUpdate = nullptr;
TGpIapRelease GpIapRelease = nullptr;
TGpIapSetDebugMode GpIapSetDebugMode = nullptr;
TGpIapSetUserInfo GpIapSetUserInfo = nullptr;
TGpIapLogin GpIapLogin = nullptr;
TGpIapLogout GpIapLogout = nullptr;
TGpIapAddListenerEvent GpIapAddListenerEvent = nullptr;
TGpIapRemoveListenerEvent GpIapRemoveListenerEvent = nullptr;
TGpIapRequestItemListPurchasable GpIapRequestItemListPurchasable = nullptr;
TGpIapRequestItemListOfNotConsumed GpIapRequestItemListOfNotConsumed = nullptr;
TGpIapRequestPurchaseProductId GpIapRequestPurchaseProductId = nullptr;
TGpIapRequestPurchaseProductIdWithPayload GpIapRequestPurchaseProductIdWithPayload = nullptr;
TGpIapLoginWithDevAuthTool GpIapLoginWithDevAuthTool = nullptr;
TGpIapRequestPurchaseEpicGameOffer GpIapRequestPurchaseEpicGameOffer = nullptr;
TGpIapServerConsumeForTest GpIapServerConsumeForTest = nullptr;
TGpIapRequestReprocPurchased GpIapRequestReprocPurchased = nullptr;

#if WITH_GAMEBASE_EOS_SHARED
TGpIapSetEosPlatformInstance GpIapSetEosPlatformInstance = nullptr;
#endif

static FGpWindowsIapManager* GWindowsIapSdk = nullptr;

using FAsyncCallback = TFunction<void(int32, const FString&)>;

static void* DllHandle = nullptr;
static FString SavedAsyncCallbackApiName = FString();
static FAsyncCallback SavedAsyncCallback = nullptr;
static FCriticalSection SavedAsyncCallbackCriticalSection;

struct FGamebaseIapAsyncInfo
{
    using FCompleteCallback = TFunction<void(int32, const FString&)>;

    FString Name;
    FCompleteCallback OnCompleted;
};

static TSharedPtr<FGamebaseIapAsyncInfo> SavedAsyncInfo = nullptr;

static void SetSavedAsyncInfo(const FGamebaseIapAsyncInfo& Info)
{
    SavedAsyncInfo.Reset();
}

static void* GetIapDllHandle()
{
    void* Handle = nullptr;
#if PLATFORM_WINDOWS
    const FString Path = GamebaseSystemUtils::GetBinariesPath();
    GAMEBASE_LOG_GLOBAL_DEBUG("Import IapSDK DLL (Path: %s)", *Path);
    FPlatformProcess::PushDllDirectory(*Path);
    const FString FilePath = Path / GamebaseIAP::DllFileName;
    if (FPaths::FileExists(FilePath))
    {
        Handle = FPlatformProcess::GetDllHandle(*FilePath);
        if (Handle == nullptr)
        {
            const int32 ErrorNum = FPlatformMisc::GetLastError();
            TCHAR ErrorMsg[1024];
            FPlatformMisc::GetSystemErrorMessage(ErrorMsg, 1024, ErrorNum);
            GAMEBASE_LOG_GLOBAL_ERROR("Failed to get IapSDK.dll handle for %s: %s (%d))", *FilePath, ErrorMsg, ErrorNum);
        }
    }
    else
    {
        GAMEBASE_LOG_GLOBAL_ERROR("IapSDK.dll file is missing. (Path: %s)", *FilePath);
    }
    FPlatformProcess::PopDllDirectory(*Path);
#endif
    return Handle;
}

FGpWindowsIapManager::FGpWindowsIapManager()
{
    ImportDll();
    GWindowsIapSdk = this;
}

FGpWindowsIapManager::~FGpWindowsIapManager()
{
    ShutdownDll();
    GWindowsIapSdk = nullptr;
}

bool FGpWindowsIapManager::ImportDll()
{
    DllHandle = GetIapDllHandle();
    if (DllHandle == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_ERROR("Failed to import the Windows IAP DLL");
        return false;
    }
    
    void* LambdaDLLHandle = DllHandle;
    
    auto GetIapDllExport = [&LambdaDLLHandle](const TCHAR* FuncName, bool& bInSuccess) -> void*
    {
        if (bInSuccess)
        {
            void* FuncPtr = FPlatformProcess::GetDllExport(LambdaDLLHandle, FuncName);
            if (FuncPtr == nullptr)
            {
                bInSuccess = false;
                GAMEBASE_LOG_GLOBAL_WARNING("Failed to load iap sdk dll (FuncName: %s)", FuncName);
                FPlatformProcess::FreeDllHandle(LambdaDLLHandle);
                LambdaDLLHandle = nullptr;
            }
            return FuncPtr;
        }
        
        return nullptr;
    };

#define GAMEBASE_IAP_DLL_EXPORT(Name) \
    GpIap##Name = static_cast<TGpIap##Name>(GetIapDllExport(TEXT(#Name), bSuccess));

    bool bSuccess = true;
    GAMEBASE_IAP_DLL_EXPORT(Initialize);
    GAMEBASE_IAP_DLL_EXPORT(Update);
    GAMEBASE_IAP_DLL_EXPORT(Release);
    GAMEBASE_IAP_DLL_EXPORT(SetDebugMode);
    GAMEBASE_IAP_DLL_EXPORT(Login);
    GAMEBASE_IAP_DLL_EXPORT(Logout);
    GAMEBASE_IAP_DLL_EXPORT(SetUserInfo);
    GAMEBASE_IAP_DLL_EXPORT(AddListenerEvent);
    GAMEBASE_IAP_DLL_EXPORT(RemoveListenerEvent);
    GAMEBASE_IAP_DLL_EXPORT(RequestItemListPurchasable);
    GAMEBASE_IAP_DLL_EXPORT(RequestItemListOfNotConsumed);
    GAMEBASE_IAP_DLL_EXPORT(RequestPurchaseProductId);
    GAMEBASE_IAP_DLL_EXPORT(RequestPurchaseProductIdWithPayload);
    GAMEBASE_IAP_DLL_EXPORT(RequestReprocPurchased);
    GAMEBASE_IAP_DLL_EXPORT(LoginWithDevAuthTool);
    GAMEBASE_IAP_DLL_EXPORT(RequestPurchaseEpicGameOffer);
    GAMEBASE_IAP_DLL_EXPORT(ServerConsumeForTest);
#if WITH_GAMEBASE_EOS_SHARED
    GAMEBASE_IAP_DLL_EXPORT(SetEosPlatformInstance);
#endif

    check(bSuccess);
    return bSuccess;
}

bool FGpWindowsIapManager::ShutdownDll()
{
    if (DllHandle)
    {
        GpIapRelease();
        
        FPlatformProcess::FreeDllHandle(DllHandle);
        
        DllHandle = nullptr;
        GpIapInitialize = nullptr;
        GpIapUpdate = nullptr;
        GpIapRelease = nullptr;
        GpIapSetUserInfo = nullptr;
        GpIapLogin = nullptr;
        GpIapLogout = nullptr;
        GpIapAddListenerEvent = nullptr;
        GpIapRemoveListenerEvent = nullptr;
        GpIapRequestItemListPurchasable = nullptr;
        GpIapRequestItemListOfNotConsumed = nullptr;
        GpIapRequestPurchaseProductId = nullptr;
        GpIapRequestPurchaseProductIdWithPayload = nullptr;
        GpIapRequestReprocPurchased = nullptr;
        GpIapServerConsumeForTest = nullptr;
#if WITH_GAMEBASE_EOS_SHARED
        GpIapSetEosPlatformInstance = nullptr;
#endif
    }
    
    return true;
}

void FGpWindowsIapManager::Initialize(const FGamebaseIapConfiguration& Configuration, const FGamebaseIapCallbackDelegate& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback.ExecuteIfBound(Error);
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback.ExecuteIfBound(Error);
        return;
    }
    
    GAMEBASE_LOG_GLOBAL_DEBUG("IAP SDK Initialize.\n%s", *Configuration.ToJson(true));
    
    Async(EAsyncExecution::ThreadPool, []
    {
        static const FString Signature = TEXT("NHN Corporation");
        const FString FilePath = GamebaseSystemUtils::GetBinariesPath() / GamebaseIAP::DllFileName;

        TOptional<FString> Error = GamebaseVerifySignatureHelper::VerifyDll(FilePath, Signature);

        AsyncTask(ENamedThreads::GameThread, [Error]
        {
            if (Error.IsSet())
            {
                GAMEBASE_LOG_GLOBAL_WARNING("Failed to validate IapSDK.dll. (Message: %s)", **Error);
            }
            else
            {
                GAMEBASE_LOG_GLOBAL_DEBUG("IapSDK.dll validation successful");
            }
        });
    });
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK initialisation succeeds");
                    Callback.ExecuteIfBound(nullptr);
                    break;    
                }
            default:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_NOT_INITIALIZED,
                        TEXT("Windows IAP SDK initialisation failed"));
                    break;
                }
        }
    };

    const std::string ConvertString = TCHAR_TO_UTF8(*Configuration.ToJson(false));

    GpIapInitialize(ConvertString.c_str(), [](const int Code, const char* Message)
    {
        GWindowsIapSdk->OnCallback(Code, Message);
    });
}

void FGpWindowsIapManager::SetDebugMode(const bool bIsDebugMode)
{
    if (IsAvailableDll() == false)
    {
        return;
    }
    
    GpIapSetDebugMode(bIsDebugMode);
}

void FGpWindowsIapManager::SetUserInfo(const FString& GamebaseUserId, const FString& GamebaseAccessToken)
{
    if (IsAvailableDll() == false)
    {
        return;
    }
    
    GpIapSetUserInfo(TCHAR_TO_UTF8(*GamebaseUserId), TCHAR_TO_UTF8(*GamebaseAccessToken));
}

void FGpWindowsIapManager::Login(const FGamebaseIapCallbackDelegate& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback.ExecuteIfBound(Error);
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback.ExecuteIfBound(Error);
        return;
    }
    
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK Login succeeds");
                    Callback.ExecuteIfBound(nullptr);
                    break;    
                }
            default:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK Login failed (Code: %d, Message: %s)", Code, *FString(Message));
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK Login failed"));
                    break;
                }
        }
    };
    
    GpIapLogin([](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::Logout(const FGamebaseIapCallbackDelegate& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback.ExecuteIfBound(Error);
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback.ExecuteIfBound(Error);
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK Logout succeeds");
                    Callback.ExecuteIfBound(nullptr);
                    break;    
                }
            default:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK Logout failed (Code: %d, Message: %s)", Code, *FString(Message));
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK Logout failed"));
                    break;
                }
        }
    };
    
    GpIapLogout([](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::Release()
{
    if (DllHandle == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("The IAP DLL was not imported.");
        return;
    }
    
    GpIapRelease();
}

void FGpWindowsIapManager::Update()
{
    if (DllHandle == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("The IAP DLL was not imported.");
        return;
    }
    
    GpIapUpdate();
}

void FGpWindowsIapManager::AddListenerEvent()
{
    if (DllHandle == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("The IAP DLL was not imported.");
        return;
    }
    
    GpIapAddListenerEvent([](int Code, const char* Message, const char* Reserve)
        {
            const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
            switch (ErrorCode)
            {
                case EGpWindowsIapResultCode::EventAuthAutoLoggedIn:
                    {
                        GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK event EventAuthAutoLoggedIn (Message: %s, Reserve: %s)", *FString(Message), *FString(Reserve));
                        break;
                    }
                case EGpWindowsIapResultCode::EventUserConnectAuthExpiring:
                    {
                        GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK event EventUserConnectAuthExpiring (Message: %s, Reserve: %s)", *FString(Message), *FString(Reserve));
                        break;
                    }
                case EGpWindowsIapResultCode::EventSdkReProcessingFlow:
                    {
                        GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK event EventSdkReProcessingFlow (Message: %s, Reserve: %s)", *FString(Message), *FString(Reserve));
                        break;
                    }
                default:
                    {
                        GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK event (Code: %d, Message: %s, Reserve: %s)", Code, *FString(Message), *FString(Reserve));
                        break;
                    }
            }
        });
}

void FGpWindowsIapManager::RemoveListenerEvent()
{
    GpIapRemoveListenerEvent();
}

void FGpWindowsIapManager::RequestItemListOfNotConsumed(const bool bAllStores, FGamebaseIapResultFunc&& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback(FGamebaseIapResult(*Error.Get()));
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback(FGamebaseIapResult(*Error.Get()));
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    Callback(FGamebaseIapResult(Message));
                    break;
                }
            case EGpWindowsIapResultCode::SuccessNoUnconsumedItems:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("There are no unconsumed items.");
                    Callback(FGamebaseIapResult(FString()));
                    break;
                }
            default:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK RequestItemListOfNotConsumed failed"));
                    break;
                }
        }
    };
    
    GpIapRequestItemListOfNotConsumed(bAllStores, [](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::RequestPurchaseProductId(
    const FString& MarketItemId,
    const FString& GamebaseProductId,
    FGamebaseIapResultFunc&& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback(FGamebaseIapResult(*Error.Get()));
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback(FGamebaseIapResult(*Error.Get()));
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    Callback(FGamebaseIapResult(Message));
                    break;    
                }
            case EGpWindowsIapResultCode::ErrorIapCheckoutUserCanceled:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_USER_CANCELED,
                        TEXT("Windows IAP SDK RequestPurchaseProductId failed"));
                    break;
                }
            default:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK RequestPurchaseProductId failed"));
                    break;
                }
        }
    };
    
    GpIapRequestPurchaseProductId(
        TCHAR_TO_UTF8(*MarketItemId),
        TCHAR_TO_UTF8(*GamebaseProductId),
        [](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::RequestPurchaseProductIdWithPayload(
    const FString& MarketItemId,
    const FString& GamebaseProductId,
    const FString& Payload,
    FGamebaseIapResultFunc&& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback(FGamebaseIapResult(*Error.Get()));
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback(FGamebaseIapResult(*Error.Get()));
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    Callback(FGamebaseIapResult(Message));
                    break;    
                }
            case EGpWindowsIapResultCode::ErrorIapCheckoutUserCanceled:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_USER_CANCELED,
                        TEXT("Windows IAP SDK RequestPurchaseProductIdWithPayload failed"));
                    break;
                }
            default:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK RequestPurchaseProductIdWithPayload failed (Code: %d, Message: %s)", Code, *FString(Message));
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK RequestPurchaseProductIdWithPayload failed"));
                    break;
                }
        }
    };
    
    GpIapRequestPurchaseProductIdWithPayload(TCHAR_TO_UTF8(*MarketItemId),
        TCHAR_TO_UTF8(*GamebaseProductId),
        TCHAR_TO_UTF8(*Payload),
        [](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::RequestItemListPurchasable(FGamebaseIapResultFunc&& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback(FGamebaseIapResult(*Error.Get()));
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback(FGamebaseIapResult(*Error.Get()));
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    Callback(FGamebaseIapResult(Message));
                    break;    
                }
            default:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK RequestItemListPurchasable failed"));
                    break;
                }
        }
    };
    
    GpIapRequestItemListPurchasable([](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::RequestReprocessPurchased(const FString& GamebaseItemListJsonString, const FGamebaseIapCallbackDelegate& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback.ExecuteIfBound(Error);
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback.ExecuteIfBound(Error);
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK RequestReprocessPurchased succeeds");
                    Callback.ExecuteIfBound(nullptr);
                    break;    
                }
            case EGpWindowsIapResultCode::ErrorNotInitialize:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_NOT_INITIALIZED,
                        TEXT("Proceed without initializing IAP SDK"));
                    break;
                }
            case EGpWindowsIapResultCode::ErrorRequireUserLogin:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Store login does not work properly in IAP SDK"));
                    break;
                }
            case EGpWindowsIapResultCode::ErrorInvalidGamebaseUserInfo:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Gamebase User Info is invalid."));
                    break;
                }
            case EGpWindowsIapResultCode::ErrorInvalidParameter:
            default:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK RequestReprocessPurchased failed"));
                    break;
                }
        }
    };
    
    GpIapRequestReprocPurchased(TCHAR_TO_UTF8(*GamebaseItemListJsonString), [](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

#if WITH_GAMEBASE_EOS_SHARED
void FGpWindowsIapManager::SetEOSPlatformHandle(const EOS_HPlatform PlatformHandle)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        return;
    }
    
    GpIapSetEosPlatformInstance(PlatformHandle);
}
#endif

void FGpWindowsIapManager::OnCallback(const int32 Code, const char* Message)
{
    const FString ConvertedMessage = FString(UTF8_TO_TCHAR(Message));
    
    GamebaseGameThreadDispatcher::Dispatch([Code, ConvertedMessage]
        {
            FScopeLock Lock(&SavedAsyncCallbackCriticalSection);

            const FAsyncCallback TempCallback = MoveTemp(SavedAsyncCallback);
            const FString TempApiName = MoveTemp(SavedAsyncCallbackApiName);
            
            SavedAsyncCallback = nullptr;
            SavedAsyncCallbackApiName = TEXT("");
            
            if (TempCallback)
            {
                GAMEBASE_LOG_GLOBAL_DEBUG("Response received from IAP SDK (API: %s, Code: %d, Message: %s)", *TempApiName, Code, *ConvertedMessage);
                TempCallback(Code, ConvertedMessage);
            }
            else
            {
                GAMEBASE_LOG_GLOBAL_ERROR("SavedAsyncCallback is null! (API: %s, Code: %d, Message: %s)", *TempApiName, Code, *ConvertedMessage);
            }
        });
}

void FGpWindowsIapManager::Login(const FGamebaseErrorDelegate& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback.ExecuteIfBound(Error.Get());
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback.ExecuteIfBound(Error.Get());
        return;
    }

    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__) ;
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK Login succeeds");
                    Callback.ExecuteIfBound(nullptr);
                    break;    
                }
            default:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK Login failed (Code: %d, Message: %s)", Code, *FString(Message));
                    GamebaseErrorCallback::Execute(Callback, Code, Message, GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR, TEXT("Windows IAP SDK Login failed"));
                    break;
                }
        }
    };
    
    GpIapLogin([](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::LoginWithDevAuthTool(const FString& Host, const FString& CredentialName, const FGamebaseErrorDelegate& Callback)
{
    if (IsAvailableDll() == false)
    {
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK Login succeeds");
                    Callback.ExecuteIfBound(nullptr);
                    break;    
                }
            default:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK Login failed (Code: %d, Message: %s)", Code, *FString(Message));
                    GamebaseErrorCallback::Execute(Callback, Code, Message, GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR, TEXT("Windows IAP SDK Login failed"));
                    break;
                }
        }
    };
    
    GpIapLoginWithDevAuthTool(TCHAR_TO_UTF8(*Host), TCHAR_TO_UTF8(*CredentialName), [](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::Logout(const FGamebaseErrorDelegate& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback.ExecuteIfBound(&(*Error));
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback.ExecuteIfBound(&(*Error));
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    Callback.ExecuteIfBound(nullptr);
                    break;    
                }
            default:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK Logout failed"));
                    break;
                }
        }
    };
    
    GpIapLogout([](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::RequestPurchaseEpicGameOffer(const FString& GameOfferId, const FGamebaseErrorDelegate& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback.ExecuteIfBound(&(*Error));
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback.ExecuteIfBound(&(*Error));
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    Callback.ExecuteIfBound(nullptr);
                    break;    
                }
            default:
                {
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK RequestPurchaseEpicGameOffer failed"));
                    break;
                }
        }
    };
    
    GpIapRequestPurchaseEpicGameOffer(TCHAR_TO_UTF8(*GameOfferId), [](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::IapServerConsumeForTest(const FString& PaymentSeq, const FString& AccessToken, const FGamebaseErrorDelegate& Callback)
{
    if (const FGamebaseErrorPtr Error = AvailableDll())
    {
        Callback.ExecuteIfBound(&(*Error));
        return;
    }
    
    if (const FGamebaseErrorPtr Error = ProcessingApi())
    {
        Callback.ExecuteIfBound(&(*Error));
        return;
    }
    
    SavedAsyncCallbackApiName = ANSI_TO_TCHAR(__FUNCTION__);
    SavedAsyncCallback = [Callback](int32 Code, const FString& Message) {
        const EGpWindowsIapResultCode ErrorCode = static_cast<EGpWindowsIapResultCode>(Code);
        switch (ErrorCode)
        {
            case EGpWindowsIapResultCode::Success:
                {
                    Callback.ExecuteIfBound(nullptr);
                    break;    
                }
            default:
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK IapServerConsumeForTest failed (Code: %d, Message: %s)", Code, *FString(Message));
                    GamebaseErrorCallback::Execute(Callback, Code, Message,
                        GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Windows IAP SDK IapServerConsumeForTest failed"));
                    break;
                }
        }
    };
    
    GpIapServerConsumeForTest(TCHAR_TO_UTF8(*PaymentSeq), TCHAR_TO_UTF8(*AccessToken), [](const int Code, const char* Message)
        {
            GWindowsIapSdk->OnCallback(Code, Message);
        });
}

void FGpWindowsIapManager::Shutdown()
{
}

FGamebaseErrorPtr FGpWindowsIapManager::AvailableDll()
{
    if (IsAvailableDll() == false)
    {
        return MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
            GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
            TEXT("Failed to import the Windows IAP DLL."), GamebasePurchaseShared::Domain);
    }

    return nullptr;
}

FGamebaseErrorPtr FGpWindowsIapManager::ProcessingApi()
{
    if (IsProcessingApi() == false)
    {
        const FString CurrentApi = SavedAsyncCallbackApiName.IsEmpty() ? TEXT("Unknown") : SavedAsyncCallbackApiName;
        return MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
            GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR,
            FString::Printf(TEXT("API '%s' is currently being processed. The new API call will be ignored until it completes."), *CurrentApi),
            GamebasePurchaseShared::Domain);
    }

    return nullptr;
}

bool FGpWindowsIapManager::IsAvailableDll()
{
    if (DllHandle == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("Failed to import the Windows IAP DLL.");
        return false;
    }

    return true;
}

bool FGpWindowsIapManager::IsProcessingApi()
{
    if (SavedAsyncCallback != nullptr)
    {
        const FString CurrentApi = SavedAsyncCallbackApiName.IsEmpty() ? TEXT("Unknown") : SavedAsyncCallbackApiName;
        GAMEBASE_LOG_GLOBAL_DEBUG("API '%s' is currently being processed. The new API call will be ignored until it completes.", *CurrentApi);
        return false;
    }

    return true;
}
