#include "GamebaseStandalonePurchaseAdapter.h"

#include "IGamebaseIapManager.h"
#include "GamebaseDebugLogger.h"
#include "GamebaseErrorCode.h"
#include "GamebaseStandalonePurchaseConstants.h"
#include "GpWindowsIapModels.h"
#include "Types/GamebasePurchaseTypes.h"

void UGamebaseStandalonePurchaseAdapter::Initialize(
    const FGamebasePurchaseConfiguration& Configuration,
    FIapResultFunc&& Callback)
{
    IapManager = IGamebaseIapManager::Get();
    if (IapManager == nullptr)
    {
        const auto Error = MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
            GamebaseErrorCode::PURCHASE_NOT_INITIALIZED,
            TEXT("Windows IAP SDK initialisation failed"), GamebasePurchaseShared::Domain);
        Callback(&Error.Get());
        return;
    }
    
    PreInitialize(Configuration);

    if (const TOptional<FString> StoreError = GetStoreError())
    {
        const auto Error = MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
            GamebaseErrorCode::PURCHASE_NOT_SUPPORTED_MARKET,
            *StoreError,
            GamebasePurchaseShared::Domain);
        
        Callback(&Error.Get());
        return;
    }
    
    const auto& IapConfiguration = GetConfiguration(Configuration);
    
    IapManager->Initialize(IapConfiguration, FGamebaseIapCallbackDelegate::CreateLambda([this, Callback](const FGamebaseErrorPtr& Error)
    {
        if (Error.IsValid())
        {
            Callback(Error.Get());
            return;
        }
        
        TickDelegateHandle = GamebaseTicker::AddTicker(FTickerDelegate::CreateUObject(this, &UGamebaseStandalonePurchaseAdapter::Tick));
        IapManager->AddListenerEvent();
    }));
}

void UGamebaseStandalonePurchaseAdapter::SetDebugMode(const bool bIsDebugMode)
{
    if (IapManager == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("IapManager is null");
        return;
    }
    
    IapManager->SetDebugMode(bIsDebugMode);
}

void UGamebaseStandalonePurchaseAdapter::SetGamebaseUserId(const FString& GamebaseUserId, const FString& GamebaseToken)
{
    if (IapManager == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("IapManager is null");
        return;
    }
    
    IapManager->SetUserInfo(GamebaseUserId, GamebaseToken);
}

void UGamebaseStandalonePurchaseAdapter::SetDisplayLanguage(const FString& LanguageCode)
{
    if (IapManager == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("IapManager is null");
        return;
    }
    
    SavedDisplayLanguage = LanguageCode;
}

void UGamebaseStandalonePurchaseAdapter::RequestReprocessPurchased(const FString& GamebaseItemListJsonString, const FIapResultFunc& Callback)
{
    if (IapManager == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("IapManager is null");
        return;
    }
    
    IapManager->RequestReprocessPurchased(GamebaseItemListJsonString, FGamebaseIapCallbackDelegate::CreateLambda([Callback](const FGamebaseErrorPtr& Error)
    {
        if (Error.IsValid())
        {
            Callback(Error.Get());
        }
        else
        {
            Callback(nullptr);
        }
    }));
}

void UGamebaseStandalonePurchaseAdapter::RequestPurchase(const FString& MarketItemId, const FString& GamebaseProductId, const FIapPurchaseReceiptDelegate& Callback)
{
    if (IapManager == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("IapManager is null");
        return;
    }
    
    IapManager->RequestPurchaseProductId(MarketItemId, GamebaseProductId,
        [Callback, StoreCode = GetUniqueKey()](const FGamebaseIapResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(TOptional<FIapPurchaseReceipt>(), &Result.GetErrorValue());
                return;
            }

            const FString JsonString = Result.GetOkValue();
            if (JsonString.IsEmpty())
            {
                //Result message is null
                Callback.ExecuteIfBound(TOptional<FIapPurchaseReceipt>(), nullptr);
                return;
            }
            
            using namespace GpWindowsIapSdk;
                        
            FPurchaseReceiptData ResponseData;
            if (!ResponseData.FromJson(JsonString))
            {
                //Json format error
                Callback.ExecuteIfBound(TOptional<FIapPurchaseReceipt>(), nullptr);
                return;
            }
            
            FIapPurchaseReceipt ResultData;
            ResultData.ItemSeq = 0;
            ResultData.Price = ResponseData.Price;
            ResultData.Currency = ResponseData.Currency;
            ResultData.PaymentSeq = ResponseData.PaymentSeq;
            ResultData.PurchaseToken = ResponseData.PurchaseToken;
            ResultData.MarketItemId = ResponseData.MarketItemId;
            ResultData.UserId = ResponseData.UserId;
            ResultData.PaymentId = ResponseData.PaymentId;
            ResultData.ProductType = TEXT("CONSUMABLE");
            ResultData.OriginalPaymentId = ResponseData.PaymentSeq;
            ResultData.PurchaseTime = ResponseData.PurchaseTime;
            ResultData.ExpiryTime = 0;
            ResultData.StoreCode = StoreCode;
            ResultData.Payload = ResponseData.GamebasePayload;
            ResultData.bIsPromotion = false;
            ResultData.bIsTestPurchase = false;

            Callback.ExecuteIfBound(ResultData, nullptr);
        });
}

void UGamebaseStandalonePurchaseAdapter::RequestPurchase(const FString& MarketItemId, const FString& GamebaseProductId, const FString& Payload, const FIapPurchaseReceiptDelegate& Callback)
{
    if (IapManager == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("IapManager is null");
        return;
    }
    
    IapManager->RequestPurchaseProductIdWithPayload(MarketItemId, GamebaseProductId, Payload,
        [Callback, StoreCode = GetUniqueKey(), Payload](const FGamebaseIapResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(TOptional<FIapPurchaseReceipt>(), &Result.GetErrorValue());
                return;
            }
            
            const FString JsonString = Result.GetOkValue();
            if (JsonString.IsEmpty())
            {
                //Result message is null
                Callback.ExecuteIfBound(TOptional<FIapPurchaseReceipt>(), nullptr);
                return;
            }
            
            using namespace GpWindowsIapSdk;
            
            FPurchaseReceiptData ResponseData;
            if (!ResponseData.FromJson(JsonString))
            {
                //Json format error
                Callback.ExecuteIfBound(TOptional<FIapPurchaseReceipt>(), nullptr);
                return;
            }
            
            GAMEBASE_LOG_GLOBAL_DEBUG("Windows IAP SDK RequestPurchaseProductIdWithPayload succeeds");
            
            FIapPurchaseReceipt ResultData;
            ResultData.ItemSeq = 0;
            ResultData.Price = ResponseData.Price;
            ResultData.Currency = ResponseData.Currency;
            ResultData.PaymentSeq = ResponseData.PaymentSeq;
            ResultData.PurchaseToken = ResponseData.PurchaseToken;
            ResultData.MarketItemId = ResponseData.MarketItemId;
            ResultData.UserId = ResponseData.UserId;
            ResultData.PaymentId = ResponseData.PaymentId;
            ResultData.ProductType = TEXT("CONSUMABLE");
            ResultData.OriginalPaymentId = ResponseData.PaymentSeq;
            ResultData.PurchaseTime = ResponseData.PurchaseTime;
            ResultData.ExpiryTime = 0;
            ResultData.StoreCode = StoreCode;
            ResultData.Payload = Payload;
            ResultData.bIsPromotion = false;
            ResultData.bIsTestPurchase = false;

            Callback.ExecuteIfBound(ResultData, nullptr);
        });
}

void UGamebaseStandalonePurchaseAdapter::RequestItemListPurchasable(const FIapPurchasableItemListDelegate& Callback)
{
    if (IapManager == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("IapManager is null");
        return;
    }
    
    IapManager->RequestItemListPurchasable(
        [this, Callback](const FGamebaseIapResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(TOptional<FIapPurchasableItemArray>(), &Result.GetErrorValue());
                return;
            }
            
            const FString JsonString = Result.GetOkValue();
            if (JsonString.IsEmpty())
            {
                //Result message is null
                Callback.ExecuteIfBound(TOptional<FIapPurchasableItemArray>(), nullptr);
                return;
            }

            const FIapPurchasableItemArray ResultArray = ConvertItemListPurchasable(JsonString);
            Callback.ExecuteIfBound(ResultArray, nullptr);
        });
}

void UGamebaseStandalonePurchaseAdapter::RequestItemListAtIAPConsole(const FIapPurchasableItemListDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandalonePurchaseAdapter::RequestItemListOfNotConsumed(const FGamebasePurchasableConfiguration& Configuration, const FIapPurchaseReceiptListDelegate& Callback)
{
    if (IapManager == nullptr)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("IapManager is null");
        return;
    }
    
    IapManager->RequestItemListOfNotConsumed(Configuration.bAllStores,
        [Callback, StoreCode = GetUniqueKey()](const FGamebaseIapResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(TOptional<FIapPurchaseReceiptArray>(), &Result.GetErrorValue());
                return;
            }
            
            const FString JsonString = Result.GetOkValue();
            if (JsonString.IsEmpty())
            {
                Callback.ExecuteIfBound(TOptional<FIapPurchaseReceiptArray>(), nullptr);
                return;
            }
            
            using namespace GpWindowsIapSdk;
            
            FIapPurchaseReceiptArray ResultArray;
                        
            TArray<FPurchaseReceiptData> IapItems = NHN::Json::ConvertJsonStringToJsonArray<FPurchaseReceiptData>(JsonString);
            for (const FPurchaseReceiptData& IapData : IapItems)
            {
                FIapPurchaseReceipt ResultData;
                ResultData.GamebaseProductId = IapData.GamebaseProductId;
                ResultData.Price = IapData.Price;
                ResultData.Currency = IapData.Currency;
                ResultData.PaymentSeq = IapData.PaymentSeq;
                ResultData.PurchaseToken = IapData.PurchaseToken;
                ResultData.MarketItemId = IapData.MarketItemId;
                ResultData.ProductType = IapData.ProductType;
                ResultData.UserId = IapData.UserId;
                ResultData.PaymentId = IapData.PaymentId;
                ResultData.OriginalPaymentId = IapData.OriginalPaymentId;
                ResultData.PurchaseTime = IapData.PurchaseTime;
                ResultData.ExpiryTime = 0;
                ResultData.StoreCode = StoreCode;
                ResultData.Payload = IapData.GamebasePayload;
                ResultData.bIsPromotion = false;
                ResultData.bIsTestPurchase = false;
                            
                ResultArray.Add(ResultData);
            }
                        
            Callback.ExecuteIfBound(ResultArray, nullptr);
        });
}

void UGamebaseStandalonePurchaseAdapter::PreInitialize(const FGamebasePurchaseConfiguration& Configuration)
{
}

FIapPurchasableItemArray UGamebaseStandalonePurchaseAdapter::ConvertItemListPurchasable(const FString& JsonString)
{
    using namespace GpWindowsIapSdk;

    FIapPurchasableItemArray Result;
    
    TArray<FPurchasableDefaultData> IapItems = NHN::Json::ConvertJsonStringToJsonArray<FPurchasableDefaultData>(JsonString);
    for (const FPurchasableDefaultData& Item : IapItems)
    {
        FIapPurchasableItem ResultItem;
        ResultItem.ItemSeq = Item.ProductNo;
        ResultItem.Price = Item.OriginalPrice;
        ResultItem.Currency = Item.Currency;
        ResultItem.MarketItemId = Item.ProductId;
        ResultItem.ProductType = TEXT("CONSUMABLE");
        ResultItem.LocalizedPrice = FText::AsCurrency(Item.OriginalPrice, Item.Currency).ToString();
        ResultItem.LocalizedTitle = Item.ProductName;
        ResultItem.LocalizedDescription = Item.LocalizedDescription;
        ResultItem.bIsActive = Item.bPriceValid;
                
        Result.Add(ResultItem);
    }

    return Result;
}

TOptional<FString> UGamebaseStandalonePurchaseAdapter::GetStoreError() const
{
    return TOptional<FString>();
}

FString UGamebaseStandalonePurchaseAdapter::GetDisplayLanguage() const
{
    return SavedDisplayLanguage;
}

IGamebaseIapManager* UGamebaseStandalonePurchaseAdapter::GetIapManager() const
{
    return IapManager;
}

bool UGamebaseStandalonePurchaseAdapter::Tick(float DeltaTime)
{
    if (IapManager)
    {
        //TODO: EOSSDK를 위한 용도로 알고 있는데 이러한 업데이트는 게임에서 처리해야 한다고 생각하므로 제거 관련하여 논의 필요
        IapManager->Update();
    }
    
    return true;
}
