#include "GamebaseStandaloneAuthTwitter.h"

#include "GamebaseAuthProvider.h"
#include "Constants/GamebaseAuthProviderCredential.h"

namespace GamebaseAuthTwitterConstants
{
    constexpr int32 MaxCodeChallengeLength = 100;
    constexpr int32 MaxBase64Length = (MaxCodeChallengeLength / 4) * 3; // 75 bytes
    
    const FName AuthorizationProtocol("oauth2");
    
    FString GenerateRandomString(const int32 Length)
    {
        constexpr TCHAR AllowedChars[] = TEXT("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-._~");
        constexpr int32 AllowedCharCount = UE_ARRAY_COUNT(AllowedChars) - 1;

        FString RandomString;
        RandomString.Reserve(Length);

        const FRandomStream RandomStream(FPlatformTime::Cycles());

        for (int32 i = 0; i < Length; ++i)
        {
            const int32 Index = RandomStream.RandRange(0, AllowedCharCount - 1);
            RandomString.AppendChar(AllowedChars[Index]);
        }

        return RandomString;
    }

    FString EncodeBase64URLSafe(const FString& Input)
    {
        // Convert the input string to a byte array
        const FTCHARToUTF8 UTF8Converter(*Input);
        const uint8* ByteData = reinterpret_cast<const uint8*>(UTF8Converter.Get());
        int32 ByteLength = UTF8Converter.Length();

        // Twitter's Code Challenge has a maximum length of 100 characters, which translates to 75 bytes in Base64
        if (ByteLength > MaxBase64Length)
        {
            ByteLength = MaxBase64Length;
        }

        // Encode to Base64
        const FString Base64String = FBase64::Encode(ByteData, ByteLength);

        // Make the Base64 string URL-safe
        FString URLSafeString = Base64String.Replace(TEXT("+"), TEXT("-"))
                                           .Replace(TEXT("/"), TEXT("_"))
                                           .Replace(TEXT("="), TEXT(""));

        return URLSafeString;
    }
    
    FString GenerateCodeVerifier()
    {
        constexpr int32 MinLength = 43;
        constexpr int32 MaxLength = 128;

        const FRandomStream RandomStream(FPlatformTime::Cycles());
        const int32 Length = RandomStream.RandRange(MinLength, MaxLength);

        return GenerateRandomString(Length);
    }
    
    FString GenerateOAuthPKCE()
    {
        const FString CodeVerifier = GenerateCodeVerifier();
        const FString CodeChallenge = EncodeBase64URLSafe(CodeVerifier);

        return CodeChallenge;
    }

}

FGamebaseVariantMap UGamebaseStandaloneAuthTwitter::GetProviderSpecificQueries() const
{
    FGamebaseVariantMap Queries;
    Queries.Add(GamebaseAuthMemberWebLoginParameter::AuthorizationProtocol, GamebaseAuthTwitterConstants::AuthorizationProtocol.ToString());
    Queries.Add(GamebaseAuthMemberWebLoginParameter::CodeChallenge, GamebaseAuthTwitterConstants::GenerateOAuthPKCE());
    
    return Queries;
}

FGamebaseVariantMap UGamebaseStandaloneAuthTwitter::GetProviderSpecificCredential() const
{
    FGamebaseVariantMap Credentials;
    Credentials.Add(GamebaseAuthProviderCredential::AuthorizationProtocol, GamebaseAuthTwitterConstants::AuthorizationProtocol.ToString());
    
    return Credentials;
}

FString UGamebaseStandaloneAuthTwitter::GetUniqueKey() const
{
    return GamebaseAuthProvider::Twitter;
}
