#include "GamebaseStandaloneAuthSteam.h"

#include "GamebaseAuthProvider.h"
#include "GamebaseStandaloneAuthSteamConstants.h"
#include "Constants/GamebaseAuthProviderCredential.h"
#include "Constants/GamebaseErrorCode.h"

#if WITH_GAMEBASE_STEAMWORKS_SHARED

#pragma warning(push)
#pragma warning(disable : 4996)
#include "steam/steam_api.h"
#pragma warning(pop)

using FGamebaseSteamAuthTicketResult = FGamebaseInternalResult<FString>;

class FGamebaseSteamWebApiAuthTicketRequest
{
public:
    using FResultCallback = TFunction<void(const FGamebaseSteamAuthTicketResult&)>;

    FGamebaseSteamWebApiAuthTicketRequest(const char* Identity, const FResultCallback& Callback)
        : OnGetTicketForWebApiResponseCallback(this, &FGamebaseSteamWebApiAuthTicketRequest::OnAuthTicketResponse), OnAuthTicketReceived(Callback)
    {
        const HAuthTicket AuthTicket = SteamUser()->GetAuthTicketForWebApi(Identity);
        if (AuthTicket == k_HAuthTicketInvalid)
        {
            OnAuthTicketReceived(FGamebaseSteamAuthTicketResult(
                FGamebaseError(GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR, TEXT("Failed to obtain an Auth Ticket from Steamworks."), GamebaseAuthSteam::Domain)));
        }
    }

private:
    STEAM_CALLBACK(FGamebaseSteamWebApiAuthTicketRequest, OnAuthTicketResponse, GetTicketForWebApiResponse_t, OnGetTicketForWebApiResponseCallback);

    FResultCallback OnAuthTicketReceived;
};

void FGamebaseSteamWebApiAuthTicketRequest::OnAuthTicketResponse(GetTicketForWebApiResponse_t* CallbackData)
{
    if (CallbackData->m_eResult == k_EResultOK)
    {
        const FString ResultToken = BytesToHex(CallbackData->m_rgubTicket, CallbackData->m_cubTicket);
        OnAuthTicketReceived(FGamebaseSteamAuthTicketResult(ResultToken));
    }
    else
    {
        OnAuthTicketReceived(FGamebaseSteamAuthTicketResult(
            FGamebaseError(GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR, TEXT("Failed to retrieve Auth Ticket"), GamebaseAuthSteam::Domain)));
    }
}

#endif

namespace GamebaseAuthSteamConstants
{
    const FName AuthorizationProtocol("steamworks");
}

void UGamebaseStandaloneAuthSteam::Initialize(const FGamebaseAuthConfiguration& Configuration)
{
    Super::Initialize(Configuration);
}

void UGamebaseStandaloneAuthSteam::Login(const FGamebaseVariantMap& AdditionalInfo, const FLoginCallback& Callback)
{
    bool bIsProcess = false;
    
#if WITH_GAMEBASE_STEAMWORKS_SHARED
    bIsProcess = true;
    
    SteamErrMsg SteamErrorMessage;
    const ESteamAPIInitResult InitResult = SteamAPI_InitEx(&SteamErrorMessage);
    if (InitResult != k_ESteamAPIInitResult_OK)
    {
        const FString ErrorMessage = FString::Printf(TEXT("Steamworks SDK initialization failed. (ErrorCode: %d, ErrorMessage: %s)"), static_cast<int32>(InitResult), UTF8_TO_TCHAR(SteamErrorMessage));
        
        Callback(FGamebaseAuthCredentialsResult(FGamebaseAuthCredentialsResult(
            FGamebaseError(GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED, ErrorMessage, GamebaseAuthSteam::Domain))));
        return;
    }

    if (SteamUser() != nullptr && SteamUser()->BLoggedOn())
    {
        const char* Identity = "";
        Event = MakeShared<FGamebaseSteamWebApiAuthTicketRequest>(Identity,
            [this, Callback](const FGamebaseSteamAuthTicketResult& Result)
            {
                if (Result.IsError())
                {
                    Callback(FGamebaseAuthCredentialsResult(FGamebaseAuthCredentialsResult(Result.GetErrorValue())));
                    Event.Reset();
                    return;
                }
                
                FGamebaseAuthCredentials Credentials;
                Credentials.Add(GamebaseAuthProviderCredential::AccessToken, Result.GetOkValue());
                Credentials.Add(GamebaseAuthProviderCredential::AuthorizationProtocol, GamebaseAuthSteamConstants::AuthorizationProtocol.ToString());

                Callback(FGamebaseAuthCredentialsResult(Credentials));
                Event.Reset();
            });
    }
    else
    {
        Callback(FGamebaseAuthCredentialsResult(FGamebaseAuthCredentialsResult(
            FGamebaseError(GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED, "Not logged into Steam.", GamebaseAuthSteam::Domain))));
        return;
    }
#endif
    
    if (bIsProcess == false)
    {
        Callback(FGamebaseAuthCredentialsResult(FGamebaseAuthCredentialsResult(
            FGamebaseError(GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED, "This is an environment where Steam Login is not available.", GamebaseAuthSteam::Domain))));
    }
}

void UGamebaseStandaloneAuthSteam::Logout(const FLogoutCallback& Callback)
{
}

void UGamebaseStandaloneAuthSteam::Withdraw(const FWithdrawCallback& Callback)
{
}

FString UGamebaseStandaloneAuthSteam::GetProviderName() const
{
    return {};
}

FString UGamebaseStandaloneAuthSteam::GetUserID() const
{
    return {};
}

FString UGamebaseStandaloneAuthSteam::GetAccessToken() const
{
    return {};
}

FGamebaseAuthProviderProfilePtr UGamebaseStandaloneAuthSteam::GetProfile() const
{
    return {};
}

FString UGamebaseStandaloneAuthSteam::GetUniqueKey() const
{
    return GamebaseAuthProvider::Steam;
}
