#include "GamebaseEOSAuthHandler.h"

#include "GamebaseErrorCode.h"
#include "GamebaseSharedEOSDefines.h"
#include "Utils/GamebaseTaskFlowRunner.h"

using FLoginFlowRunner = TGamebaseStopOnSuccessRunner<FGamebaseAuthEpicLoginData>;
using FLogoutFlowRunner = TGamebaseStopOnSuccessRunner<FGamebaseAuthEpicLogoutData>;

void FGamebaseEOSAuthHandler::Login(FLoginFunc&& Callback)
{
    bool bEosPersistentAuth = true;

#if PLATFORM_WINDOWS
    if (!GConfig->GetBool(TEXT("/Script/GamebaseEditor.GamebaseWindowsPlatformSettings"), TEXT("bEosPersistentAuth"), bEosPersistentAuth, GEngineIni))
    {
        bEosPersistentAuth = false;
    }
#endif

    TArray<FLoginFlowRunner::FTaskStep> Steps = {
        AsyncLoginWithExistingSession(),
        AsyncLoginWithExchangeCode(),
        bEosPersistentAuth ? AsyncLoginWithPersistentAuth() : nullptr,
        AsyncLoginWithAccountPortal()
    };

    Steps.RemoveAll([](const FLoginFlowRunner::FTaskStep& Step) { return Step == nullptr; });

    const TSharedPtr<FLoginFlowRunner> Runner = MakeShared<FLoginFlowRunner>(
        MoveTemp(Steps),
        [Callback = MoveTemp(Callback)](const FGamebaseInternalResult<FGamebaseAuthEpicLoginData>& Result, const TArray<FGamebaseError>& FailedResults)
        {
            if (Result.IsError() && FailedResults.Num() > 0)
            {
                Callback(FGamebaseInternalResult<FGamebaseAuthEpicLoginData>(FailedResults.Last()));
            }
            else
            {
                Callback(Result);
            }
        },
        [](const TArray<FGamebaseError>& FailedResults) -> FGamebaseError
        {
            return FGamebaseError(
                GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED,
                TEXT("All steps failed"),
                GamebaseSharedEOS::Auth::Domain
            );
        }
    );

    Runner->Run();
}

void FGamebaseEOSAuthHandler::Logout()
{
    TArray<FLogoutFlowRunner::FTaskStep> Steps = {
        AsyncLogout()
    };

    const TSharedPtr<FLogoutFlowRunner> Runner = MakeShared<FLogoutFlowRunner>(
        MoveTemp(Steps),
        [](const FGamebaseInternalResult<FGamebaseAuthEpicLogoutData>& Result, const TArray<FGamebaseError>& FailedResults)
        {
        },
        [](const TArray<FGamebaseError>& FailedResults) -> FGamebaseError
        {
            return FGamebaseError(
                GamebaseErrorCode::AUTH_LOGOUT_FAILED,
                TEXT("Logout failed"),
                GamebaseSharedEOS::Auth::Domain
            );
        }
    );

    Runner->Run();
}

bool FGamebaseEOSAuthHandler::IsLoggedIn() const
{
    return false;
}
