﻿#include "GpLoggerSettingLoader.h"

#include "GpLoggerSettings.h"
#include "HttpModule.h"
#include "Interfaces/IHttpResponse.h"

namespace GpLogger
{
    namespace ServerUrl
    {
        const FString SettingUrl = TEXT("https://setting-logncrash.cloud.toast.com/v2");

        FString GetSettingUrl(const FString& Appkey, const EGpLoggerServiceZone ServiceZone)
        {
            FString ZoneString;
            switch (ServiceZone)
            {
                case EGpLoggerServiceZone::Alpha:
                    ZoneString = TEXT("alpha-");
                break;
                case EGpLoggerServiceZone::Beta:
                    ZoneString = TEXT("beta-");
                break;
                default: ;
            }

            return FString::Printf(TEXT("https://%ssetting-logncrash.cloud.toast.com/v2/%s/logsconf"), *ZoneString, *Appkey);
        }
    }
}

using namespace GpLogger;

void FGpLoggerDefaultSettingLoader::Update(FUpdateCompleteFunc&& Func)
{
    FGpLoggerSettings DefaultSettings;
    Func(DefaultSettings);
}

void FGpLoggerConsoleSettingLoader::Update(FUpdateCompleteFunc&& Func)
{
    Request().Next([Func](const FGpLoggerSettingResult& Result)
    {
        FGpLoggerSettings ConsoleSettings;
        ConsoleSettings.FromJson(Result.Message);
        Func(ConsoleSettings);
    });
}

TFuture<FGpLoggerSettingResult> FGpLoggerConsoleSettingLoader::Request()
{
    const TSharedRef<IHttpRequest, ESPMode::ThreadSafe> HttpRequestRef = FHttpModule::Get().CreateRequest();
    
    HttpRequestRef->SetURL(ServerUrl::GetSettingUrl(Appkey, ServiceZone));
    HttpRequestRef->SetHeader(TEXT("Content-Type"), TEXT("application/json"));
    HttpRequestRef->SetVerb(TEXT("GET"));
    
    TSharedPtr<IGpLoggerSettingLoader> SharedThis = AsShared();
    
    TSharedPtr<TPromise<FGpLoggerSettingResult>> PromisePtr = MakeShared<TPromise<FGpLoggerSettingResult>>();
    HttpRequestRef->OnProcessRequestComplete().BindLambda([PromisePtr](const FHttpRequestPtr& Request, const FHttpResponsePtr& Response, const bool bSucceeded)
    {
        if (!bSucceeded || !Response.IsValid())
        {
            PromisePtr->SetValue(FGpLoggerSettingResult{ EGpLoggerSettingResult::Failed, FString() });
            return;
        }

        if (Response->GetContentLength() <= 0)
        {
            PromisePtr->SetValue(FGpLoggerSettingResult{ EGpLoggerSettingResult::Failed, FString() });
            return;
        }

        PromisePtr->SetValue(FGpLoggerSettingResult{ EGpLoggerSettingResult::Success, Response->GetContentAsString() });
    });

    if (!HttpRequestRef->ProcessRequest())
    {
        return MakeFulfilledPromise<FGpLoggerSettingResult>(FGpLoggerSettingResult{ EGpLoggerSettingResult::Failed, FString() }).GetFuture();
    }
    
    return PromisePtr->GetFuture();
}
