#include "GpLoggerLogTransfer.h"

#include "HttpModule.h"
#include "Interfaces/IHttpResponse.h"
#include "GpLoggerDefines.h"
#include "GpLoggerHelper.h"
#include "Utils/GpLoggerJsonMacros.h"

namespace GpLogTransfer
{
    constexpr uint32 MaxSendSize = 2097152;
    constexpr float TransferIntervalTime = 0.1f;
}

struct FGpLoggerBulkApiResponse final : FGpLoggerJsonSerializable
{
    struct FHeader final : FGpLoggerJsonSerializable
    {
        bool bIsSuccessful;
        int32 ResultCode;
        FString ResultMessage;
        
        BEGIN_GPLOGGER_JSON_SERIALIZER
            GPLOGGER_JSON_SERIALIZE("isSuccessful", bIsSuccessful);
            GPLOGGER_JSON_SERIALIZE("resultCode", ResultCode);
            GPLOGGER_JSON_SERIALIZE("resultMessage", ResultMessage);
        END_GPLOGGER_JSON_SERIALIZER
    };

    struct FBody final : FGpLoggerJsonSerializable
    {
        struct FData final : FGpLoggerJsonSerializable
        {
            struct FResult final : FGpLoggerJsonSerializable
            {
                bool bIsSuccessful;
                FString ResultMessage;
                
                BEGIN_GPLOGGER_JSON_SERIALIZER
                    GPLOGGER_JSON_SERIALIZE("isSuccessful", bIsSuccessful);
                    GPLOGGER_JSON_SERIALIZE("resultMessage", ResultMessage);
                END_GPLOGGER_JSON_SERIALIZER
            };
            
            int32 Total;
            int32 Errors;
            TArray<FResult> ResultList;
                
            BEGIN_GPLOGGER_JSON_SERIALIZER
                GPLOGGER_JSON_SERIALIZE("total", Total);
                GPLOGGER_JSON_SERIALIZE("errors", Errors);
                GPLOGGER_JSON_SERIALIZE_ARRAY_SERIALIZABLE("resultList", ResultList, FResult);
            END_GPLOGGER_JSON_SERIALIZER
        };

        FData Data;
                
        BEGIN_GPLOGGER_JSON_SERIALIZER
            GPLOGGER_JSON_SERIALIZE_OBJECT_SERIALIZABLE("data", Data);
        END_GPLOGGER_JSON_SERIALIZER
    };

    FHeader Header;
    FBody Body;
    
    BEGIN_GPLOGGER_JSON_SERIALIZER
        GPLOGGER_JSON_SERIALIZE_OBJECT_SERIALIZABLE("header", Header);
        GPLOGGER_JSON_SERIALIZE_OBJECT_SERIALIZABLE("body", Body);
    END_GPLOGGER_JSON_SERIALIZER
};


FGpLoggerLogTransfer::FGpLoggerLogTransfer()
{
    RunnableThread.Reset(FRunnableThread::Create(this, *FString::Printf(TEXT("GPLoggerTransferThread"))));
}

FGpLoggerLogTransfer::~FGpLoggerLogTransfer()
{
    Stop();
    
    if (RunnableThread.IsValid())
    {
        RunnableThread->WaitForCompletion();
        RunnableThread.Reset();
    }
}

void FGpLoggerLogTransfer::Enqueue(const FGpBulkLogDataRef& BulkLog)
{
    FScopeLock Lock(&DataGuard);
    SendQueue.Enqueue(BulkLog);
}

bool FGpLoggerLogTransfer::Init()
{
    return FRunnable::Init();
}

uint32 FGpLoggerLogTransfer::Run()
{
    GPLOGGER_LOG_DEBUG("starting up.");

    while (ShouldStop() == false)
    {
        ProcessTransfer();
        FPlatformProcess::Sleep(GpLogTransfer::TransferIntervalTime);
    }

    ProcessTransfer();
    
    GPLOGGER_LOG_DEBUG("returning");
    
    return 0;
}

void FGpLoggerLogTransfer::Stop()
{
    StopTaskCounter.Increment();
}

void FGpLoggerLogTransfer::Exit()
{
}

void FGpLoggerLogTransfer::DirectTransfer()
{
    ProcessTransfer();
}

void FGpLoggerLogTransfer::ProcessTransfer()
{
    TArray<FGpBulkLogDataPtr> TransferDataList;
    {
        FScopeLock Lock(&DataGuard);
        FGpBulkLogDataPtr DataItem;
        while (SendQueue.Dequeue(DataItem))
        {
            TransferDataList.Add(DataItem);
        }
    }
        
    if (TransferDataList.Num() > 0)
    {
        for (const auto& BulkLogData : TransferDataList)
        {
            ProcessTransferBulk(BulkLogData.ToSharedRef());
        }
    }
}

void FGpLoggerLogTransfer::ProcessTransferBulk(const FGpBulkLogDataRef& BulkLogData)
{
    const TOptional<FString> ContentJsonString = BulkLogData->GetJsonString();
    if (ContentJsonString.IsSet() == false)
    {
        FGpLoggerTransferResult TransferResult;
        TransferResult.ServiceData = BulkLogData->GetServiceData();
        TransferResult.BulkData = BulkLogData;
        TransferResult.ErrorMessage = TEXT("Failed to serialize BulkLogData to JSON string.");
        TransferResult.bIsRetry = false;

        if (OnTransferDelegate.ExecuteIfBound(TransferResult) == false)
        {
            GPLOGGER_LOG_ERROR("Failed to deliver Log Transfer results");
        }

        return;
    }
    
    const TSharedRef<IHttpRequest, ESPMode::ThreadSafe> HttpRequest = FHttpModule::Get().CreateRequest();
    HttpRequest->SetURL(GpLoggerServerUrl::GetCollectorUrl(BulkLogData->GetServiceZone()));
    HttpRequest->SetHeader(TEXT("Content-Type"), TEXT("application/json"));
    HttpRequest->SetVerb(TEXT("POST"));
    HttpRequest->SetContentAsString(*ContentJsonString);
    
    TWeakPtr<FGpLoggerLogTransfer> WeakThisPtr;
    if (DoesSharedInstanceExist() == true)
    {
        WeakThisPtr = AsShared();    
    }
    
    HttpRequest->OnProcessRequestComplete().BindLambda([WeakThisPtr, BulkLogData](const FHttpRequestPtr& Request, const FHttpResponsePtr& Response, const bool bSucceeded)
    {
        const FString ResponseContent = Response->GetContentAsString();

        FGpLoggerTransferResult TransferResult;
        TransferResult.ServiceData = BulkLogData->GetServiceData();
        TransferResult.BulkData = BulkLogData;
        
        if (bSucceeded && Response.IsValid())
        {
            if (ResponseContent.IsEmpty())
            {
                GPLOGGER_LOG_WARNING("The request to get console settings failed. (ResponseCode= %d)", Response->GetResponseCode());
                TransferResult.ErrorMessage = TEXT("Not response content");
            }
            else
            {
                GPLOGGER_LOG_DEBUG("Get response content - %s", *Response->GetContentAsString());
                
                FGpLoggerBulkApiResponse ResponseData;
                ResponseData.FromJson(Response->GetContentAsString());

                if (ResponseData.Header.bIsSuccessful)
                {
                    check(ResponseData.Body.Data.ResultList.Num() == BulkLogData->Num());
                    
                    for (int i = 0; i < ResponseData.Body.Data.ResultList.Num(); i++)
                    {
                        const auto ResponseResult = ResponseData.Body.Data.ResultList[i];
                        if (ResponseResult.bIsSuccessful)
                        {
                            TransferResult.SuccessLogs.Emplace(BulkLogData->Get(i));
                        }
                        else
                        {
                            TransferResult.FailedLogs.Emplace(BulkLogData->Get(i));
                            TransferResult.FailedLogErrors.Emplace(ResponseResult.ResultMessage);
                        }
                    }
                }
                else
                {
                    TransferResult.ErrorMessage = FString::Printf(TEXT("ResultCode= %d, ResultMessage= %s"),
                        ResponseData.Header.ResultCode, *ResponseData.Header.ResultMessage);
                    GPLOGGER_LOG_WARNING("ResultCode= %d, ResultMessage= %s", ResponseData.Header.ResultCode, *ResponseData.Header.ResultMessage);
                }
            }
        }
        else
        {
            TransferResult.ErrorMessage = TEXT("The request to log send failed.");
            GPLOGGER_LOG_WARNING("The request to log send failed. (RequestCode= %s)", EHttpRequestStatus::ToString(Request->GetStatus()));
        }
        
        if (const TSharedPtr<FGpLoggerLogTransfer> SharedThis = WeakThisPtr.Pin())
        {
            if (SharedThis->OnTransferDelegate.ExecuteIfBound(TransferResult) == false)
            {
                GPLOGGER_LOG_ERROR("Failed to deliver Log Transfer results");
            }
        }
    });
    
    GPLOGGER_LOG_DEBUG("log transfer - %s", *ContentJsonString.Get(TEXT("Failed json serialization")));
    HttpRequest->ProcessRequest();
}
