#include "GpLoggerLogSender.h"

#include "GpBaseLogger.h"
#include "GpLoggerBackupStorage.h"
#include "GpLoggerDefines.h"
#include "GpLoggerEngineSubsystem.h"
#include "GpLoggerHelper.h"
#include "GpLoggerLogCollector.h"
#include "GpLoggerLogTransfer.h"

void UGpLoggerLogSender::Initialize()
{
    if (LogCollector.IsValid() == false)
    {
        LogCollector = MakeShared<FGpLoggerLogCollector>();
        LogCollector->GetBulkCreatedDelegate().BindUObject(this, &UGpLoggerLogSender::OnBulkCreated);
    }
    
    if (LogTransfer.IsValid() == false)
    {
        LogTransfer = MakeShared<FGpLoggerLogTransfer>();
        LogTransfer->GetTransferDelegate().BindUObject(this, &UGpLoggerLogSender::OnTransferResult);
    }

    if (LogBackupStorage.IsValid() == false)
    {
        FString SessionId;
        if (const UGpLoggerEngineSubsystem* CoreSubsystem = GEngine->GetEngineSubsystem<UGpLoggerEngineSubsystem>())
        {
            SessionId = CoreSubsystem->GetSessionId();
        }
        
        LogBackupStorage = MakeShared<FGpLoggerBackupStorage>(SessionId);
    }

    FCoreDelegates::OnExit.AddUObject(this, &UGpLoggerLogSender::DirectSend);
}

void UGpLoggerLogSender::Enqueue(const FGpLoggerLogData& LogData)
{
    if (LogCollector.IsValid() == false)
    {
        return;
    }
    
    LogCollector->Enqueue(LogData);
}

void UGpLoggerLogSender::DirectSend()
{
    if (LogCollector.IsValid() == false || LogTransfer.IsValid() == false)
    {
        return;
    }

    LogCollector->DirectCollect();
    LogTransfer->DirectTransfer();
}

void UGpLoggerLogSender::DirectSend(const FGpLoggerLogData& LogData)
{
    if (LogCollector.IsValid() == false || LogTransfer.IsValid() == false)
    {
        return;
    }

    LogCollector->DirectCollect();
    
    const FGpBulkLogDataRef BulkData = MakeShared<FGpLoggerBulkLogData, ESPMode::ThreadSafe>(LogData.GetServiceData(), LogData);
    LogTransfer->Enqueue(BulkData);
    LogTransfer->DirectTransfer();
}

void UGpLoggerLogSender::SetListener(const FGpLoggerServiceData& Service, const FEventParams& Listener)
{
    if (Events.Contains(Service))
    {
        Events[Service] = Listener;
    }
    else
    {
        Events.Add(Service, Listener);
    }
}

void UGpLoggerLogSender::OnBulkCreated(const FGpBulkLogDataRef& BulkLog) const
{
    if (LogTransfer.IsValid() == false)
    {
        return;
    }
    
    LogTransfer->Enqueue(BulkLog);
}

void UGpLoggerLogSender::OnTransferResult(const FGpLoggerTransferResult& Result)
{
    const FEventParams* EventListener = Events.Find(Result.ServiceData);
    if (EventListener == nullptr)
    {
        //GPLOGGER_LOG_DEBUG("No event listener");
        return;
    }
    
    FGpBulkLogDataPtr FailedBulkData = nullptr;
    
    if (Result.SuccessLogs.Num() > 0)
    {
        if (FUObjectThreadContext::Get().IsRoutingPostLoad == false)
        {
            for (const auto& SuccessLog : Result.SuccessLogs)
            {
                if (EventListener->OnSuccessDelegate.ExecuteIfBound(GpLoggerHelper::ConvertLogEntry(SuccessLog)) == false)
                {
                    GPLOGGER_LOG_DEBUG("OnSuccessDelegate was not called.");
                }
            }
        }
    
        if (LogBackupStorage.IsValid())
        {
            const bool bIsDeleted = LogBackupStorage->DeleteData(Result.ServiceData, *Result.BulkData);
            if (bIsDeleted == false)
            {
                GPLOGGER_LOG_DEBUG("Failed to delete data from backup storage.");
            }
        }
        
        if (Result.FailedLogs.Num() > 0)
        {
            //TODO: 새로운 벌크를 만드는?
            //FailedBulkData = MakeShared<FGpLoggerBulkLogData, ESPMode::ThreadSafe>(Result.ServiceData, Result.FailedLogs);
        }
    }
    else
    {
        FailedBulkData = Result.BulkData;
    }
    
    if (FailedBulkData.IsValid())
    {
        if (Result.bIsRetry && LogBackupStorage.IsValid() && LogBackupStorage->SaveData(Result.ServiceData, *FailedBulkData))
        {
            if (FUObjectThreadContext::Get().IsRoutingPostLoad == false)
            {
                for (const auto& LogData : FailedBulkData->GetLogs())
                {
                    if (EventListener->OnSaveDelegate.ExecuteIfBound(GpLoggerHelper::ConvertLogEntry(LogData)) == false)
                    {
                        GPLOGGER_LOG_DEBUG("OnSaveDelegate was not called.");
                    }
                }
            }
        }
        else
        {
            if (FUObjectThreadContext::Get().IsRoutingPostLoad == false)
            {
                for (int i = 0; i < FailedBulkData->Num(); i++)
                {
                    FString ErrorMsg;
                    if (Result.FailedLogErrors.IsValidIndex(i) && !Result.FailedLogErrors[i].IsEmpty())
                    {
                        ErrorMsg = Result.FailedLogErrors[i];
                    }
                    else
                    {
                        ErrorMsg = Result.ErrorMessage;
                    }
                    
                    if (EventListener->OnErrorDelegate.ExecuteIfBound(GpLoggerHelper::ConvertLogEntry(FailedBulkData->Get(i)), ErrorMsg) == false)
                    {
                        GPLOGGER_LOG_DEBUG("OnErrorDelegate was not called.");
                    }
                }
            }
        }
    }

    if (LogBackupStorage.IsValid())
    {
        const FGpBulkLogDataPtr LoadData = LogBackupStorage->LoadData(Result.ServiceData);
        if (LoadData.IsValid())
        {
            OnBulkCreated(LoadData.ToSharedRef());
        }
    }
}