#include "GpLoggerLogCollector.h"

#include "GpLoggerDefines.h"
#include "GpLoggerLogInfo.h"

namespace GpLogReceiver
{
    constexpr int32 QueueCapacity = 2048;
    constexpr float CheckReceiverTime = 0.1f;
}

FGpLoggerLogCollector::FGpLoggerLogCollector()
{
    RunnableThread.Reset(FRunnableThread::Create(this, *FString::Printf(TEXT("GPLoggerCollectorThread"))));
}

FGpLoggerLogCollector::~FGpLoggerLogCollector()
{
    Stop();
    
    if (RunnableThread.IsValid())
    {
        RunnableThread->WaitForCompletion();
        RunnableThread.Reset();
    }
}

void FGpLoggerLogCollector::Enqueue(const FGpLoggerLogData& LogData)
{
    FScopeLock Lock(&QueueLock);

    if (!Logs.Contains(LogData.GetServiceData()))
    {
        Logs.Emplace(LogData.GetServiceData(), MakeShared<FCollectQueue>());
    }

    Logs[LogData.GetServiceData()]->Enqueue(LogData);
}

void FGpLoggerLogCollector::DirectCollect()
{
    Collect();
}

bool FGpLoggerLogCollector::Init()
{
    return FRunnable::Init();
}

uint32 FGpLoggerLogCollector::Run()
{
    while (ShouldStop() == false)
    {
        Collect();
        FPlatformProcess::Sleep(GpLogReceiver::CheckReceiverTime);
    }
    
    Collect();
    
    return 0;
}

void FGpLoggerLogCollector::Stop()
{
    StopTaskCounter.Increment();
}

void FGpLoggerLogCollector::Exit()
{
}

void FGpLoggerLogCollector::Collect()
{
    FServiceCollectMap LocalLogs;
    {
        FScopeLock Lock(&QueueLock);
        if (Logs.Num() == 0)
        {
            return;
        }

        LocalLogs = MoveTemp(Logs);
        Logs.Reset();
    }

    for (auto& LogPair : LocalLogs)
    {
        TArray<FGpLoggerLogData> LogArray;
        FGpLoggerLogData LogData;
        while (LogPair.Value->Dequeue(LogData))
        {
            LogArray.Add(LogData);
        }

        const FGpBulkLogDataRef BulkLog = MakeShared<FGpLoggerBulkLogData, ESPMode::ThreadSafe>(LogPair.Key, LogArray);
        if (OnBulkCreated.ExecuteIfBound(BulkLog) == true)
        {
            GPLOGGER_LOG_DEBUG("log collector - %s", *BulkLog->GetJsonString().Get(TEXT("Failed json serialization")));
        }
        else
        {
            GPLOGGER_LOG_ERROR("No bulk created callback");
        }
    }
}
