﻿#include "GpLoggerFilter.h"

#include "GpLoggerLogInfo.h"
#include "GpLoggerInternalTypes.h"

using namespace GpLogger;

bool FGpLoggerNormalFilter::IsFiltered(const FGpLoggerLogData& LogItem)
{
    // 비활성화 - Normal이면 보내지 않는다. 필터 확인 후 거른다.
    // 활성화 - Normal이면 보내야 한다. 필터 거칠 필요가 없다.
    if (bIsEnabled)
    {
        return true;
    }

    return LogItem.GetLogType().Equals(LogType::Normal) == false;
}

bool FGpLoggerSessionFilter::IsFiltered(const FGpLoggerLogData& LogItem)
{
    if (bIsEnabled)
    {
        return true;
    }

    return LogItem.GetLogType().Equals(LogType::Session) == false;
}

bool FGpLoggerCrashFilter::IsFiltered(const FGpLoggerLogData& LogItem)
{
    if (bIsEnabled)
    {
        return true;
    }

    return LogItem.GetLogType().Equals(LogType::Crash) == false;
}

bool FGpLoggerLogLevelFilter::IsFiltered(const FGpLoggerLogData& LogItem)
{
    if (bIsEnabled == false)
    {
        return true;
    }
    
    return static_cast<uint8>(LogItem.GetLogLevel()) >= static_cast<uint8>(LogLevel);
}

bool FGpLoggerLogTypeFilter::IsFiltered(const FGpLoggerLogData& LogItem)
{
    if (bIsEnabled == false)
    {
        return true;
    }

    for (FString Type : LogType)
    {
        if (Type.Equals(LogItem.GetLogType()))
        {
            return false;
        }
    }

    return true;
}

bool FGpLoggerDuplicateFilter::IsFiltered(const FGpLoggerLogData& LogItem)
{
    if (bIsEnabled == false)
    {
        return true;
    }
    
    const auto CurrentTime = FDateTime::UtcNow() - FDateTime(1970, 1, 1);
    const int64 CurrentTimeMS = CurrentTime.GetTotalMilliseconds();
    
    for (auto It = DuplicateInfoMap.CreateIterator(); It; ++It)
    {
        if (It.Value() < CurrentTimeMS - ExpireTime * 1000)
        {
            It.RemoveCurrent();
        }
    }

    const TOptional<FString> Key = LogItem.GetDuplicateJsonString();
    if (Key.IsSet() == false)
    {
        // If the key is not set, we cannot filter duplicates.
        return true;
    }

    {   
        const uint32 KeyHash = FCrc::StrCrc32(*Key.GetValue());
        FScopeLock Lock(&CriticalSection);
        if (DuplicateInfoMap.Contains(KeyHash))
        {
            return false;
        }

        const int64 CreateTime = LogItem.GetCreateTime();
        DuplicateInfoMap.Add(KeyHash, CreateTime);

        static constexpr int32 DefaultDuplicateFilterCapacity = 2048;
        if (DuplicateInfoMap.Num() >= DefaultDuplicateFilterCapacity)
        {
            DuplicateInfoMap.Remove(DuplicateInfoMap.CreateConstIterator().Key());
        }
    }
    
    return true;
}