﻿#include "GpLoggerBackupStorage.h"

#include "Kismet/GameplayStatics.h"

namespace GpFile
{
    FString HashProjectKey(const FString& ProjectKey)
    {
        FSHAHash Hash;
        FSHA1::HashBuffer(TCHAR_TO_ANSI(*ProjectKey), ProjectKey.Len(), Hash.Hash);
        return Hash.ToString();
    }

    FString MakeOldSaveSlotName(const FString& HashedAppkey)
    {
        return FString::Printf(TEXT("GpLogger_%s"), *HashedAppkey);
    }
    
    FString MakeSaveSlotName(const FString& HashedAppkey)
    {
        return FString::Printf(TEXT("GpLogger-%s"), *HashedAppkey);
    }
}

FGpBulkLogDataPtr FGpLoggerBackupStorage::LoadData(const FGpLoggerServiceData& ServiceData)
{
    DeleteOldData(ServiceData);
    const FString HashedAppkey = GpFile::HashProjectKey(ServiceData.Appkey);
    if (BackupLogs.Contains(HashedAppkey))
    {
        UGpLoggerLogSave* SaveLogObject = BackupLogs.FindRef(HashedAppkey);
        for (auto& SaveData : SaveLogObject->BulkLogs)
        {
            if (SaveData.Logs.Num() > 0 && SaveData.SessionId != SessionId)
            {
                SaveData.SessionId = SessionId;
                return MakeShared<FGpLoggerBulkLogData, ESPMode::ThreadSafe>(ServiceData, SaveData.Logs, SaveData.Uuid);
            }
        }
        return nullptr;
    }
    
    const FString SlotName = GpFile::MakeSaveSlotName(HashedAppkey);
    if (!UGameplayStatics::DoesSaveGameExist(SlotName, 0))
    {
        return nullptr;
    }
    
    UGpLoggerLogSave* SaveLogObject = Cast<UGpLoggerLogSave>(UGameplayStatics::LoadGameFromSlot(SlotName, 0));

    if (SaveLogObject)
    {
        BackupLogs.FindOrAdd(HashedAppkey, SaveLogObject);
        SaveLogObject->AddToRoot();
        for (auto& SaveData : SaveLogObject->BulkLogs)
        {
            if (SaveData.Logs.Num() > 0 && SaveData.SessionId != SessionId)
            {
                SaveData.SessionId = SessionId;
                return MakeShared<FGpLoggerBulkLogData, ESPMode::ThreadSafe>(ServiceData, SaveData.Logs, SaveData.Uuid);
            }
        }
    }

    return nullptr;
}

bool FGpLoggerBackupStorage::SaveData(const FGpLoggerServiceData& ServiceData, const FGpLoggerBulkLogData& Info)
{
    FString HashedAppkey = GpFile::HashProjectKey(ServiceData.Appkey);

    UGpLoggerLogSave* SaveLogObject = BackupLogs.FindRef(HashedAppkey);
    if (SaveLogObject)
    {
        bool bIsNewData = true;
        for (int i = 0; i < SaveLogObject->BulkLogs.Num(); i++)
        {
            auto& SaveLog = SaveLogObject->BulkLogs[i];
            if (SaveLog.Uuid == Info.GetUuid())
            {
                if (SaveLog.SessionId == SessionId)
                {
                    return true;
                }
                
                bIsNewData = false;
                SaveLog.SessionId = SessionId;
                break;
            }
        }

        if (bIsNewData)
        {
            constexpr int32 MaxBulkLogsCount = 50;
            
            FGpSaveBulkData SaveData;
            SaveData.Uuid = FGuid::NewGuid().ToString();
            SaveData.SessionId = SessionId;
            SaveData.Logs.Append(Info.GetLogs());

            SaveLogObject->BulkLogs.Add(SaveData);

            while (SaveLogObject->BulkLogs.Num() > MaxBulkLogsCount)
            {
                SaveLogObject->BulkLogs.RemoveAt(0);
            }
        }
    }
    else
    {
        SaveLogObject = Cast<UGpLoggerLogSave>(UGameplayStatics::CreateSaveGameObject(UGpLoggerLogSave::StaticClass()));

        SaveLogObject->Appkey = HashedAppkey;
        
        FGpSaveBulkData SaveData;
        SaveData.Uuid = FGuid::NewGuid().ToString();
        SaveData.SessionId = SessionId;
        SaveData.Logs.Append(Info.GetLogs());

        SaveLogObject->BulkLogs.Add(SaveData);
        SaveLogObject->AddToRoot();
        BackupLogs.FindOrAdd(HashedAppkey, SaveLogObject);
    }

    return UGameplayStatics::SaveGameToSlot(SaveLogObject, GpFile::MakeSaveSlotName(HashedAppkey), 0);
}

bool FGpLoggerBackupStorage::DeleteData(const FGpLoggerServiceData& ServiceData, const FGpLoggerBulkLogData& Info)
{
    const FString HashedAppkey = GpFile::HashProjectKey(ServiceData.Appkey);

    if (Info.GetUuid().IsEmpty())
    {
        return true;
    }

    UGpLoggerLogSave* SaveLogObject = BackupLogs.FindRef(HashedAppkey);
    if (SaveLogObject == nullptr)
    {
        return true;
    }
    
    const FString SlotName = GpFile::MakeOldSaveSlotName(HashedAppkey);

    for (int i = 0; i < SaveLogObject->BulkLogs.Num(); i++)
    {
        const auto SaveLog = SaveLogObject->BulkLogs[i];
        if (SaveLog.Uuid == Info.GetUuid())
        {
            SaveLogObject->BulkLogs.RemoveAt(i);
            break;
        }
    }

    if (SaveLogObject->BulkLogs.Num() == 0)
    {
        SaveLogObject->RemoveFromRoot();
        SaveLogObject = nullptr;
        
        const FString SaveSlotName = GpFile::MakeSaveSlotName(HashedAppkey);
        const bool bSlotExists = UGameplayStatics::DoesSaveGameExist(SaveSlotName, 0);
        bool bDeleted = true;
        
        if (bSlotExists)
        {
            bDeleted = UGameplayStatics::DeleteGameInSlot(SaveSlotName, 0);
            
            if (UGameplayStatics::DoesSaveGameExist(SaveSlotName, 0))
            {
                bDeleted = false;
            }
        }
        
        if (bDeleted)
        {
            BackupLogs.Remove(HashedAppkey);
        }
        
        return bDeleted;
    }
    
    return UGameplayStatics::SaveGameToSlot(SaveLogObject, GpFile::MakeSaveSlotName(HashedAppkey), 0);
}

void FGpLoggerBackupStorage::DeleteOldData(const FGpLoggerServiceData& ServiceData)
{
    static bool bDeleted = false;
    if (bDeleted)
    {
        return;
    }

    const FString HashedAppkey = GpFile::HashProjectKey(ServiceData.Appkey);
    const FString SlotName = GpFile::MakeOldSaveSlotName(HashedAppkey);
    if (!UGameplayStatics::DoesSaveGameExist(SlotName, 0))
    {
        bDeleted = true;
        return;
    }

    UGameplayStatics::DeleteGameInSlot(SlotName, 0);
    bDeleted = true;
}
