﻿#include "GpStandaloneLogger.h"

#include "GenericPlatform/GenericPlatformCrashContext.h"
#include "GpLoggerInternalCrashTypes.h"
#include "GpLoggerDefines.h"
#include "GpLoggerEngineSubsystem.h"
#include "GpLoggerInternalTypes.h"
#include "GpLoggerInfo.h"
#include "GpLoggerStandaloneSubsystem.h"
#include "Interfaces/IPluginManager.h"
#include "Internal/GpLoggerFilter.h"
#include "Internal/GpLoggerHelper.h"
#include "Internal/GpLoggerSettingLoader.h"
#include "Internal/GpLoggerLogSender.h"
#include "Internal/GpLoggerSettings.h"
#include "Utils/GpLoggerUtils.h"

namespace GpLogger
{
    void PrepareDefaultInfo(FGpLoggerLogData& LogData)
    {
        LogData.SetUserField(LogField::ProjectVersion, GpLoggerInfo::GetProjectVersion());
        LogData.SetUserField(LogField::SdkVersion, GpLoggerInfo::GetPluginVersion());
        
        if (const UGpLoggerEngineSubsystem* CoreSubsystem = GEngine->GetEngineSubsystem<UGpLoggerEngineSubsystem>())
        {
            LogData.SetUserField(LogField::DeviceID, CoreSubsystem->GetDeviceId());
            LogData.SetUserField(LogField::PlatformName, CoreSubsystem->GetPlatform());
            LogData.SetUserField(LogField::LaunchedID, CoreSubsystem->GetLaunchedId());
        }
    }

    void PreparePublicInfo(FGpLoggerLogData& LogData)
    {
        PrepareDefaultInfo(LogData);
        
        if (const UGpLoggerEngineSubsystem* CoreSubsystem = GEngine->GetEngineSubsystem<UGpLoggerEngineSubsystem>())
        {
            LogData.SetUserField(LogField::SessionID, CoreSubsystem->GetSessionId());
        }
    }

    FString ExtractCrashGUIDRoot(const FString& CrashConfigFilePath)
    {
        FString FolderPath = FPaths::GetPath(CrashConfigFilePath);
        FString CrashGUIDRoot = FPaths::GetCleanFilename(FolderPath);
    
        return CrashGUIDRoot;
    }
    
    FString GetLatestCrashFolder(const FString& CrashGUIDRoot)
    {
        const FString GUIDPattern = CrashGUIDRoot + TEXT("_");
        
        const FString BaseCrashFolder = FPaths::Combine(FPaths::ProjectSavedDir(), TEXT("Crashes"));
        const FString FullCrashFolder = FPaths::ConvertRelativePathToFull(BaseCrashFolder);
        
        IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
        int32 MaxIndex = -1;
        FString LatestCrashFolder;

        PlatformFile.IterateDirectory(*BaseCrashFolder, [&LatestCrashFolder, &GUIDPattern, &MaxIndex](const TCHAR* FileOrDirectory, const bool bIsDirectory) {
            if (bIsDirectory)
            {
                const FString FolderName = FPaths::GetCleanFilename(FileOrDirectory);
                
                if (FolderName.StartsWith(GUIDPattern))
                {
                    int32 UnderscoreIndex;
                    if (FolderName.FindLastChar(TEXT('_'), UnderscoreIndex))
                    {
                        
                        const FString IndexStr = FolderName.Mid(UnderscoreIndex + 1);
                        const int32 CurrentIndex = FCString::Atoi(*IndexStr);
                        
                        if (CurrentIndex > MaxIndex)
                        {
                            MaxIndex = CurrentIndex;
                            LatestCrashFolder = FileOrDirectory;
                        }
                    }
                }
            }
            return true;
        });
        
        return LatestCrashFolder;
    }
    
    FString GetDumpPath()
    {
        const FString CrashGUIDRoot = ExtractCrashGUIDRoot(FGenericCrashContext::GetCrashConfigFilePath());
        const FString LatestCrashFolder = GetLatestCrashFolder(CrashGUIDRoot);
        
        if (!LatestCrashFolder.IsEmpty())
        {
            return FPaths::Combine(FPaths::ConvertRelativePathToFull(LatestCrashFolder), FGenericCrashContext::UE4MinidumpName);
        }

        return FString();
    }
}

using namespace GpLogger;

void UGpStandaloneLogger::Initialize(const FInitializeParams& Params)
{
    ServiceData.Appkey = Params.Appkey;
    ServiceData.ServiceZone = Params.ServiceZone;

    const auto MakeSettingsFunc = [Params]() -> TSharedPtr<IGpLoggerSettingLoader>
    {
        switch (Params.SettingType)
        {
            case EGpLoggerSettingType::Default:     return MakeShared<FGpLoggerDefaultSettingLoader>(Params.Appkey, Params.ServiceZone);
            case EGpLoggerSettingType::Console:     return MakeShared<FGpLoggerConsoleSettingLoader>(Params.Appkey, Params.ServiceZone);
            default:                                checkNoEntry();
        }
        
        return MakeShared<FGpLoggerDefaultSettingLoader>(Params.Appkey, Params.ServiceZone);
    };
    
    const TSharedPtr<IGpLoggerSettingLoader> SettingLoader = MakeSettingsFunc();
    SettingLoader->Update([this, bIsSendSessionLog = Params.bIsCrashReporter](const FGpLoggerSettings& Settings)
    {
        Filters.AddUnique(MakeShared<FGpLoggerNormalFilter>(Settings.Result.Log.bEnabled));
        Filters.AddUnique(MakeShared<FGpLoggerSessionFilter>(Settings.Result.Session.bEnabled));
        Filters.AddUnique(MakeShared<FGpLoggerCrashFilter>(Settings.Result.Crash.bEnabled));
        
        Filters.AddUnique(MakeShared<FGpLoggerLogLevelFilter>(Settings.Result.Filter.LogLevelFilter.bEnabled, GpLogger::StringToEnum<EGpLogLevel>(Settings.Result.Filter.LogLevelFilter.LogLevel)));
        Filters.AddUnique(MakeShared<FGpLoggerLogTypeFilter>(Settings.Result.Filter.LogTypeFilter.bEnabled, Settings.Result.Filter.LogTypeFilter.LogType));
        Filters.AddUnique(MakeShared<FGpLoggerDuplicateFilter>(Settings.Result.Filter.LogDuplicateFilter.bEnabled, Settings.Result.Filter.LogDuplicateFilter.ExpiredTime));

        if (bIsSendSessionLog)
        {
            SendSessionLog();
        }
    });
}

void UGpStandaloneLogger::Log(const FGpLoggerLogData& LogData, const bool bIsDirectSend)
{
    if (const auto Sender = UGpLoggerStandaloneSubsystem::GetLogSender())
    {
        auto IsFiltered = [&]() -> bool
        {
            for (int i = 0; i < Filters.Num(); i++)
            {
                IGpLoggerFilter* Filter = Filters[i].Get();
                if (Filter == nullptr)
                {
                    continue;
                }
                
                if (Filter->IsFiltered(LogData) == false)
                {
                    FGpLogFilter LogFilter;
                    LogFilter.Name = Filter->GetName();
                
                    OnFilterDelegate.ExecuteIfBound(GpLoggerHelper::ConvertLogEntry(LogData), LogFilter);
                    return false;
                }
            }
            
            return true;
        };
    
        if (IsFiltered())
        {
            if (bIsDirectSend)
            {
                void* OutputReadPipe = nullptr;
                void* OutputWritePipe = nullptr;
                
                const FString PluginDir = IPluginManager::Get().FindPlugin(TEXT("GPLogger"))->GetBaseDir();
                const FString RelativePath  = FPaths::Combine(PluginDir, TEXT("Binaries"), FPlatformProcess::GetBinariesSubdirectory());
                const FString AbsolutePath = FPaths::ConvertRelativePathToFull(RelativePath);
                const FString ExePath = FString::Printf(TEXT("%s/CrashUploader.exe"), *AbsolutePath);
                const FString CommandLineArgs = FString::Printf(TEXT("-Url=%s -Data=%s -DumpPath=%s"),
                        *GpLoggerServerUrl::GetCollectorUrl(ServiceData.ServiceZone), *FBase64::Encode(LogData.GetJsonString().Get({})), *GetDumpPath());
                
                GPLOGGER_LOG_DEBUG("Fatal error transfer - %s %s", *ExePath, *CommandLineArgs);
                //FPlatformProcess::CreateProc(*ExePath, *CommandLineArgs, true, false, false, nullptr, 0, nullptr, nullptr);
                
                FPlatformProcess::CreatePipe(OutputReadPipe, OutputWritePipe);
                FProcHandle Proc = FPlatformProcess::CreateProc(*ExePath, *CommandLineArgs, true, true,
                    true, nullptr, 2, nullptr, OutputWritePipe, nullptr);

                if (!Proc.IsValid())
                {
                    FPlatformProcess::ClosePipe(OutputReadPipe, OutputWritePipe);
                    GPLOGGER_LOG_DEBUG("Fatal error transfer failed");
                    return;
                }

                GPLOGGER_LOG_DEBUG("Fatal error transfer open");
                int32 RC;
                FPlatformProcess::WaitForProc(Proc);
                FPlatformProcess::GetProcReturnCode(Proc, &RC);
                const FString OutStdOut = FPlatformProcess::ReadPipe(OutputReadPipe);
                FPlatformProcess::ClosePipe(OutputReadPipe, OutputWritePipe);
                FPlatformProcess::CloseProc(Proc);
                GPLOGGER_LOG_DEBUG("Fatal error transfer result - %d %s", RC, *OutStdOut);
            }
            else
            {
                Sender->Enqueue(LogData);
            }
        }
    }
}

void UGpStandaloneLogger::SendSessionLog()
{
    FGpLoggerLogData LogData(ServiceData, LogType::Session, EGpLogLevel::None, TEXT("SESSION"));
    PreparePublicInfo(LogData);
    
    if (const UGpLoggerEngineSubsystem* CoreSubsystem = GEngine->GetEngineSubsystem<UGpLoggerEngineSubsystem>())
    {
        LogData.SetUserField(LogField::DeviceModel, CoreSubsystem->GetDeviceModel());
        LogData.SetUserField(LogField::CountryCode, CoreSubsystem->GetCountryCode());
    }

    LogData.SetUserFields(BaseLogFields);
    LogData.SetUserFields(SavedUserFields);
    
    Log(LogData);
}

void UGpStandaloneLogger::Log(const EGpLogLevel Level, const FString& Message, const TMap<FString, FString>& UserFields)
{
    FGpLoggerLogData LogData(ServiceData, LogType::Normal, Level, Message);
    PreparePublicInfo(LogData);
    
    LogData.SetUserFields(BaseLogFields);
    LogData.SetUserFields(SavedUserFields);
    
    for (auto& Field : UserFields)
    {
        LogData.SetUserField(LogField::ConvertField(Field.Key), Field.Value);
    }
    
    Log(LogData);
}

void UGpStandaloneLogger::SetUserField(const FString& Key, const FString& Value)
{
    if (Key.IsEmpty() || Value.IsEmpty())
    {
        GPLOGGER_LOG_WARNING("The key or value is empty. (Key=%s, Value=%s)", *Key, *Value);
        return;
    }
    
    SavedUserFields.Emplace(LogField::ConvertField(Key), Value);
}

void UGpStandaloneLogger::SetReserveField(const FString& Key, const FString& Value)
{
    if (Key.IsEmpty() || Value.IsEmpty())
    {
        GPLOGGER_LOG_WARNING("The key or value is empty. (Key=%s, Value=%s)", *Key, *Value);
        return;
    }
    
    BaseLogFields.Emplace(Key, Value);
}

void UGpStandaloneLogger::SetListener(const FGpLoggerEventListener& Listener)
{
    if (const auto Sender = UGpLoggerStandaloneSubsystem::GetLogSender())
    {
        UGpLoggerLogSender::FEventParams Params;
        Params.OnSuccessDelegate = Listener.OnSuccessDelegate;
        Params.OnErrorDelegate = Listener.OnErrorDelegate;
        Params.OnSaveDelegate = Listener.OnSaveDelegate;
        Sender->SetListener(ServiceData, Params);

        OnFilterDelegate = Listener.OnFilterDelegate;
    }
}

void UGpStandaloneLogger::Reporter(const FGpLoggerCrashInternalData& CrashData)
{
    FGpLoggerLogData LogData(ServiceData, LogType::GetTypeString(CrashData.LogType), CrashData.LogLevel, CrashData.Message);
    PreparePublicInfo(LogData);
    
    LogData.SetUserField(LogField::CrashDumpData, CrashData.DmpData);
    LogData.SetUserField(LogField::CrashStyle, TEXT("unreal"));
    LogData.SetUserField(LogField::CrashSymbol, TEXT("none"));
    LogData.SetUserField(LogField::LogSource, TEXT("CrashDump"));
    
    if (const UGpLoggerEngineSubsystem* CoreSubsystem = GEngine->GetEngineSubsystem<UGpLoggerEngineSubsystem>())
    {
        LogData.SetUserField(LogField::DeviceModel, CoreSubsystem->GetDeviceModel());
        LogData.SetUserField(LogField::CountryCode, CoreSubsystem->GetCountryCode());
    }

    LogData.SetUserFields(BaseLogFields);
    LogData.SetUserFields(SavedUserFields);
    LogData.SetUserFields(CrashData.UserFields);
    
    Log(LogData, true);
}
