﻿#pragma once

#include "CoreMinimal.h"
#include "GpLoggerLogInfo.generated.h"

enum class EGpLoggerServiceZone : uint8;
enum class EGpLogLevel : uint8;

USTRUCT()
struct GPLOGGERCORE_API FGpLoggerServiceData
{
    GENERATED_BODY()

    UPROPERTY()
    FString Appkey;

    UPROPERTY()
    EGpLoggerServiceZone ServiceZone;
    
    bool operator==(const FGpLoggerServiceData& Other) const
    {
        return Appkey == Other.Appkey;
    }
    
    friend FORCEINLINE uint32 GetTypeHash(const FGpLoggerServiceData& Data)
    {
        return GetTypeHash(Data.Appkey);
    }
};

USTRUCT()
struct GPLOGGERCORE_API FGpLoggerLogData
{
    GENERATED_BODY()

    FGpLoggerLogData() = default;
    FGpLoggerLogData(const FGpLoggerServiceData& ServiceData, const FString& Type, const EGpLogLevel Level, const FString& Message, const FString& TransactionId = FString());
    
    bool operator==(const FGpLoggerLogData& Other) const
    {
        return GetTransactionId() == Other.GetTransactionId();
    }
    
    void SetUserField(const FString& Field, const FString& Value);
    void SetUserFields(const TMap<FString, FString>& Fields);
    
    FString GetAppkey() const;
    EGpLogLevel GetLogLevel() const;
    FString GetLogType() const;
    FString GetLogMessage() const;
    FString GetLogSource() const;
    FString GetLogVersion() const;
    FString GetUserId() const;
    int64 GetCreateTime() const;
    FString GetTransactionId() const;

    FString Get(const FString& Field) const;
    TOptional<FString> GetJsonString() const;
    TOptional<FString> GetDuplicateJsonString() const;

    FGpLoggerServiceData GetServiceData() const;

    TArray<FString> GetFields() const;
    
    TSharedRef<FJsonObject> ToJsonObject() const;

private:
    UPROPERTY()
    TMap<FString, FString> LogFields;

    UPROPERTY()
    FGpLoggerServiceData ServiceData;
};

USTRUCT()
struct GPLOGGERCORE_API FGpLoggerBulkLogData
{
    GENERATED_BODY()

    FGpLoggerBulkLogData() = default;
    
    explicit FGpLoggerBulkLogData(const FGpLoggerServiceData& ServiceData, const TArray<FGpLoggerLogData>& Logs, const FString& Uuid = FString())
        : Uuid(Uuid), ServiceData(ServiceData), Logs(Logs)
    {
    }
    
    explicit FGpLoggerBulkLogData(const FGpLoggerServiceData& ServiceData, const FGpLoggerLogData& Log)
        : ServiceData(ServiceData), Logs(TArray<FGpLoggerLogData>{ Log })
    {
    }
    
    TOptional<FString> GetJsonString() const;

    TArray<FGpLoggerLogData> GetLogs() const
    {
        return Logs;
    }
    
    FGpLoggerLogData Get(const int32 Index) const
    {
        return Logs[Index];
    }

    FString GetUuid() const
    {
        return Uuid;
    }
    
    FGpLoggerServiceData GetServiceData() const
    {
        return  ServiceData;
    }
    
    FString GetAppkey() const
    {
        return ServiceData.Appkey;
    }
    
    EGpLoggerServiceZone GetServiceZone() const
    {
        return ServiceData.ServiceZone;
    }
    
    int32 Num() const
    {
        return Logs.Num();
    }
    
private:
    FString Uuid;
    FGpLoggerServiceData ServiceData;
    UPROPERTY()
    TArray<FGpLoggerLogData> Logs;
};

using FGpLogDataRef = TSharedRef<FGpLoggerLogData, ESPMode::ThreadSafe>;
using FGpBulkLogDataPtr = TSharedPtr<FGpLoggerBulkLogData, ESPMode::ThreadSafe>;
using FGpBulkLogDataRef = TSharedRef<FGpLoggerBulkLogData, ESPMode::ThreadSafe>;
