#pragma once

#include "CoreMinimal.h"

enum class EGpLoggerType : uint8
{
    Normal,
    Session,
    Crash,
    CrashFromInactivatedState,
    CrashFromUnreal,
    Handled,
    AppDetail
};

namespace GpLogger
{
    namespace LogType
    {
        const FString Normal = TEXT("NORMAL");
        const FString Session = TEXT("SESSION");
        const FString Crash = TEXT("CRASH");
        const FString CrashFromInactivatedState = TEXT("CRASH_FROM_INACTIVATED_STATE");
        const FString CrashFromUnreal = TEXT("CRASH_FROM_UNREAL");
        const FString Handled = TEXT("HANDLED");
        const FString AppDetail = TEXT("APP_DETAIL");

        static FString GetTypeString(const EGpLoggerType Type)
        {
            switch (Type)
            {
                case EGpLoggerType::Normal:                      return Normal;
                case EGpLoggerType::Session:                     return Session;
                case EGpLoggerType::Crash:                       return Crash;
                case EGpLoggerType::CrashFromInactivatedState:   return CrashFromInactivatedState;
                case EGpLoggerType::CrashFromUnreal:             return CrashFromUnreal;
                case EGpLoggerType::Handled:                     return Handled;
                case EGpLoggerType::AppDetail:                   return AppDetail;
                default:                                         checkNoEntry();
            }
            
            return FString();
        }
    }
    
    namespace LogVersion
    {
        static const FString Version(TEXT("v2"));
    }
    
    namespace LogField
    {
        // Project Info
        static const FString ProjectKey(TEXT("projectName"));
        static const FString ProjectVersion(TEXT("projectVersion"));

        // Log info
        static const FString LogVersion(TEXT("logVersion"));
        static const FString LogType(TEXT("logType"));
        static const FString LogSource(TEXT("logSource"));
        static const FString LogLevel(TEXT("logLevel"));
        static const FString LogMessage(TEXT("body"));
        static const FString LogSendTime(TEXT("sendTime"));
        static const FString LogCreateTime(TEXT("createTime"));
        static const FString LogBulkIndex(TEXT("lncBulkIndex"));
        static const FString LogTransactionID(TEXT("transactionID"));

        // Device info
        static const FString DeviceModel(TEXT("DeviceModel"));
        static const FString CarrierName(TEXT("Carrier"));
        static const FString CountryCode(TEXT("CountryCode"));
        static const FString PlatformName(TEXT("Platform"));

        // Network info
        static const FString NetworkType(TEXT("NetworkType"));

        // Identify info
        static const FString DeviceID(TEXT("DeviceID"));
        static const FString SessionID(TEXT("SessionID"));
        static const FString LaunchedID(TEXT("launchedID"));
        static const FString UserID(TEXT("UserID"));
        
        // Version info
        static const FString SdkVersion(TEXT("SdkVersion"));

        // Crash info
        static const FString CrashStyle(TEXT("CrashStyle"));
        static const FString CrashSymbol(TEXT("SymMethod"));
        static const FString CrashDumpData(TEXT("dmpData"));
        
        // These fields generated by symbolication in server.
        static const FString SinkVersion(TEXT("SinkVersion"));
        static const FString ErrorCode(TEXT("errorCode"));
        static const FString CrashMeta(TEXT("crashMeta"));
        static const FString SymbolicationResult(TEXT("SymResult"));
        static const FString ExceptionType(TEXT("ExceptionType"));
        static const FString Location(TEXT("Location"));
        static const FString IssueID(TEXT("lncIssueID"));

        static const TSet<FString> ReservedFields = {
            ProjectKey.ToLower(),
            ProjectVersion.ToLower(),
            LogVersion.ToLower(),
            LogType.ToLower(),
            LogSource.ToLower(),
            LogLevel.ToLower(),
            LogMessage.ToLower(),
            LogSendTime.ToLower(),
            LogCreateTime.ToLower(),
            LogBulkIndex.ToLower(),
            LogTransactionID.ToLower(),
            DeviceModel.ToLower(),
            CarrierName.ToLower(),
            CountryCode.ToLower(),
            PlatformName.ToLower(),
            NetworkType.ToLower(),
            DeviceID.ToLower(),
            SessionID.ToLower(),
            LaunchedID.ToLower(),
            UserID.ToLower(),
            SdkVersion.ToLower(),
            CrashStyle.ToLower(),
            CrashSymbol.ToLower(),
            CrashDumpData.ToLower(),
            SinkVersion.ToLower(),
            ErrorCode.ToLower(),
            CrashMeta.ToLower(),
            SymbolicationResult.ToLower(),
            ExceptionType.ToLower(),
            Location.ToLower(),
            IssueID.ToLower()
        };

        static FString ConvertField(const FString& Field)
        {
            const FString ConvertField = Field.Replace(TEXT(" "), TEXT("_"));

            if (ReservedFields.Contains(ConvertField.ToLower()))
            {
                return TEXT("reserved_") + ConvertField;
            }

            return ConvertField;
        }
    }
}