﻿#include "GpBridgeLogger.h"

#include "GpLoggerBridgeSubsystem.h"
#include "Internal/IGpLoggerBridge.h"
#include "Protocols/GpLoggerProtocolInstanceLogger.h"
#include "Protocols/GpLoggerProtocolLogger.h"
#include "Protocols/GpLoggerProtocolLoggerResponse.h"

using namespace GpLoggerProtocol;

UGpBridgeLogger* UGpBridgeLogger::SingletonInstance = nullptr;

namespace GpLoggerEventName
{
    static const FString SuccessMethod = TEXT("success");
    static const FString FilterMethod = TEXT("filter");
    static const FString SaveMethod = TEXT("save");
    static const FString ErrorMethod = TEXT("error");
}

UGpBridgeLogger::UGpBridgeLogger(const FObjectInitializer& ObjectInitializer)
    : Super(ObjectInitializer)
{
    //TODO: Let's think about UObject's singletons a bit more.
    // On mobile, only one should exist
    if (HasAnyFlags(RF_ClassDefaultObject
        | RF_NeedLoad
        | RF_NeedPostLoad
        | RF_NeedPostLoadSubobjects) == false)
    {
        ensureMsgf(SingletonInstance == nullptr, TEXT("Two UGpBridgeLogger should not be created. Only one GameInstance should exist in this module."));
        SingletonInstance = this;
    }
}

void UGpBridgeLogger::Initialize(const FInitializeParams& Params)
{
    if (const auto Bridge = UGpLoggerBridgeSubsystem::GetBridge())
    {
        Payload::FInitializePayload Payload;
        Payload.ProjectKey = Params.Appkey;
        Payload.ServiceZone = GpLogger::EnumToString(Params.ServiceZone).ToUpper();
        Payload.bEnableCrashReporter = Params.bIsCrashReporter;
        
        FInitializeRequest Request { Payload };
        Bridge->SendMessage(Request.ToJson(false));
    }
}

void UGpBridgeLogger::Log(const EGpLogLevel Level, const FString& Message, const TMap<FString, FString>& UserFields)
{
    if (const auto Bridge = UGpLoggerBridgeSubsystem::GetBridge())
    {
        Payload::FLogPayload Payload;
        Payload.Level = GpLogger::EnumToString(Level);
        Payload.Message = Message;
        Payload.UserFields = UserFields;
        
        FLogRequest Request { Payload };
        Bridge->SendMessage(Request.ToJson(false));
    }
}

void UGpBridgeLogger::SetUserField(const FString& Key, const FString& Value)
{
    if (const auto Bridge = UGpLoggerBridgeSubsystem::GetBridge())
    {
        Payload::FSetUserFieldPayload Payload;
        Payload.Key = Key;
        Payload.Value = Value;
        
        FSetUserFieldRequest Request { Payload };
        Bridge->SendMessage(Request.ToJson(false));
    }
}

void UGpBridgeLogger::SetListener(const FGpLoggerEventListener& Listener)
{
    if (const auto Bridge = UGpLoggerBridgeSubsystem::GetBridge())
    {
        OnSuccessDelegate = Listener.OnSuccessDelegate;
        OnSaveDelegate = Listener.OnSaveDelegate;
        OnFilterDelegate = Listener.OnFilterDelegate;
        OnErrorDelegate = Listener.OnErrorDelegate;
        
        FSetLoggerListenerRequest Request {
            GpLoggerEventName::SuccessMethod,
            GpLoggerEventName::FilterMethod,
            GpLoggerEventName::SaveMethod,
            GpLoggerEventName::ErrorMethod
        };
        Bridge->SendMessage(Request.ToJson(false));
    }
}

void UGpBridgeLogger::Reporter(const FGpLoggerCrashInternalData& CrashData)
{
    if (const auto Bridge = UGpLoggerBridgeSubsystem::GetBridge())
    {
        Payload::FExceptionPayload Payload;
        Payload.Level = GpLogger::EnumToString(CrashData.LogLevel).ToUpper();
        Payload.LogType = GpLogger::LogType::GetTypeString(CrashData.LogType);
        Payload.Message = CrashData.Message;
        Payload.DmpData = CrashData.DmpData;
        Payload.UserFields = CrashData.UserFields;
        
        FExceptionRequest Request { Payload };
        Bridge->SendMessage(Request.ToJson(false));
    }
}

void UGpBridgeLogger::ReceiveMessage(const FString& Method, const FString& JsonString) const
{
    FFunctionGraphTask::CreateAndDispatchWhenReady([this, Method, JsonString]()
    {
        if (Method.Equals(GpLoggerEventName::SuccessMethod))
        {
            SetLoggerListener::FSuccessResponse Response;
            Response.FromJson(JsonString);

            if (Response.Body.bIsSuccessful)
            {
                auto LogData = Response.Body.Log;
                FGpLogEntry LogEntry;
                LogEntry.LogType = LogData.Type;
                LogEntry.LogLevel = GpLogger::StringToEnum<EGpLogLevel>(LogData.Level);
                LogEntry.Message = LogData.Message;
                LogEntry.TransactionId = LogData.TransactionId;
                LogEntry.CreateTime = LogData.CreateTime;
                LogEntry.UserFields = LogData.UserFields;
                    
                OnSuccessDelegate.ExecuteIfBound(LogEntry);
            }
        }
        else if (Method.Equals(GpLoggerEventName::FilterMethod))
        {
            SetLoggerListener::FFilterResponse Response;
            Response.FromJson(JsonString);

            if (Response.Body.bIsSuccessful)
            {
                auto LogData = Response.Body.Log;
                FGpLogEntry LogEntry;
                LogEntry.LogType = LogData.Type;
                LogEntry.LogLevel = GpLogger::StringToEnum<EGpLogLevel>(LogData.Level);
                LogEntry.Message = LogData.Message;
                LogEntry.TransactionId = LogData.TransactionId;
                LogEntry.CreateTime = LogData.CreateTime;
                LogEntry.UserFields = LogData.UserFields;
                    
                FGpLogFilter LogFilter;
                LogFilter.Name = Response.Body.Filter.Name;
                    
                OnFilterDelegate.ExecuteIfBound(LogEntry, LogFilter);
            }
        }
        else if (Method.Equals(GpLoggerEventName::SaveMethod))
        {
            SetLoggerListener::FSaveResponse Response;
            Response.FromJson(JsonString);

            if (Response.Body.bIsSuccessful)
            {
                auto LogData = Response.Body.Log;
                FGpLogEntry LogEntry;
                LogEntry.LogType = LogData.Type;
                LogEntry.LogLevel = GpLogger::StringToEnum<EGpLogLevel>(LogData.Level);
                LogEntry.Message = LogData.Message;
                LogEntry.TransactionId = LogData.TransactionId;
                LogEntry.CreateTime = LogData.CreateTime;
                LogEntry.UserFields = LogData.UserFields;
                    
                OnSaveDelegate.ExecuteIfBound(LogEntry);
            }
        }
        else if (Method.Equals(GpLoggerEventName::ErrorMethod))
        {
            SetLoggerListener::FErrorResponse Response;
            Response.FromJson(JsonString);

            if (Response.Body.bIsSuccessful)
            {
                auto LogData = Response.Body.Log;
                FGpLogEntry LogEntry;
                LogEntry.LogType = LogData.Type;
                LogEntry.LogLevel = GpLogger::StringToEnum<EGpLogLevel>(LogData.Level);
                LogEntry.Message = LogData.Message;
                LogEntry.TransactionId = LogData.TransactionId;
                LogEntry.CreateTime = LogData.CreateTime;
                LogEntry.UserFields = LogData.UserFields;
                    
                OnErrorDelegate.ExecuteIfBound(LogEntry, Response.Body.ErrorMessage);
            }
        }
        else
        {
            checkNoEntry();
        }
    }
    , TStatId(), nullptr, ENamedThreads::GameThread);
}

void UGpBridgeInstanceLogger::Initialize(const FInitializeParams& Params)
{
    if (const auto Bridge = UGpLoggerBridgeSubsystem::GetBridge())
    {
        Payload::FInitializeInstancePayload Payload;
        Payload.ProjectKey = Appkey = Params.Appkey;
        Payload.ServiceZone = GpLogger::EnumToString(Params.ServiceZone).ToUpper();
        Payload.Setting = GpLogger::EnumToString(Params.SettingType).ToUpper();
        
        FInitializeInstanceRequest Request { Payload };
        Bridge->SendMessage(Request.ToJson(false));
    }
}

void UGpBridgeInstanceLogger::Log(const EGpLogLevel Level, const FString& Message, const TMap<FString, FString>& UserFields)
{
    if (const auto Bridge = UGpLoggerBridgeSubsystem::GetBridge())
    {
        Payload::FLogInstancePayload Payload;
        Payload.Type = TEXT("NORMAL");
        Payload.Level = GpLogger::EnumToString(Level).ToUpper();
        Payload.Message = Message;
        Payload.UserFields = UserFields;
        
        FLogInstanceRequest Request { Appkey, Payload };
        Bridge->SendMessage(Request.ToJson(false));
    }
}

void UGpBridgeInstanceLogger::SetUserField(const FString& Key, const FString& Value)
{
    if (const auto Bridge = UGpLoggerBridgeSubsystem::GetBridge())
    {
        Payload::FSetUserFieldPayload Payload;
        Payload.Key = Key;
        Payload.Value = Value;
        
        FSetUserFieldInstanceRequest Request { Appkey, Payload };
        Bridge->SendMessage(Request.ToJson(false));
    }
}

void UGpBridgeInstanceLogger::SetListener(const FGpLoggerEventListener& EventListener)
{
    checkNoEntry();
}

void UGpBridgeInstanceLogger::Reporter(const FGpLoggerCrashInternalData& CrashData)
{
    checkNoEntry();
}
