#include "NhnCefWindowsInvoker.h"

#include "NhnWebViewDefines.h"
#include "NhnWebViewPluginInfo.h"

void* GDLLHandle = nullptr;
void* GCEF3DllHandle = nullptr;
void* ElfHandle = nullptr;
void* D3DHandle = nullptr;
void* GLESHandle = nullptr;
void* EGLHandle = nullptr;

void* LoadDllCEF(const FString& Path)
{
    if (Path.IsEmpty())
    {
        return nullptr;
    }
    
    if (FPaths::FileExists(Path) == false)
    {
        NHNWEBVIEW_LOG_WARNING("No");
        return nullptr;
    }
    
    void* Handle = FPlatformProcess::GetDllHandle(*Path);
    if (!Handle)
    {
        const int32 ErrorNum = FPlatformMisc::GetLastError();
        TCHAR ErrorMsg[1024];
        FPlatformMisc::GetSystemErrorMessage(ErrorMsg, 1024, ErrorNum);
        NHNWEBVIEW_LOG_ERROR("Failed to get CEF3 DLL handle for %s: %s (%d)", *Path, ErrorMsg, ErrorNum);
    }
    
    return Handle;
}

FNhnCefWindowsInvoker::FNhnCefWindowsInvoker(): Functions()
{
    if (GDLLHandle == nullptr)
    {
        auto LoadDll = [](const FString& Path, bool& bInSuccess) -> void* {
            if (bInSuccess)
            {
                void* DllHandle = LoadDllCEF(Path);
                if (DllHandle == nullptr)
                {
                    bInSuccess = false;
                    NHNWEBVIEW_LOG_WARNING("Failed to load CefWebView DLL. (%s)", *Path);
                    DllHandle = nullptr;
                }

                return DllHandle;
            }

            return nullptr;
        };

        const FString Path = NhnWebViewPluginInfo::GetBinariesPath();
        NHNWEBVIEW_LOG_DEBUG("Import CefWebView DLLs (Path: %s)", *Path);
        FPlatformProcess::PushDllDirectory(*Path);

        bool bSuccess = true;

#define CEFWEBVIEW_DLL_LOAD_HANDLE(HandleName, DllName) \
        HandleName = LoadDll(FPaths::Combine(*Path, TEXT(DllName)), bSuccess);

        CEFWEBVIEW_DLL_LOAD_HANDLE(GDLLHandle, "nhncef.dll");
        CEFWEBVIEW_DLL_LOAD_HANDLE(GCEF3DllHandle, "nlibcef.dll");
        CEFWEBVIEW_DLL_LOAD_HANDLE(ElfHandle, "chrome_elf.dll");
        CEFWEBVIEW_DLL_LOAD_HANDLE(EGLHandle, "libGLESv2.dll");
        CEFWEBVIEW_DLL_LOAD_HANDLE(D3DHandle, "d3dcompiler_47.dll");

        FPlatformProcess::PopDllDirectory(*Path);

        check(bSuccess);

        LoadMethods();
    }
}

FNhnCefWindowsInvoker::~FNhnCefWindowsInvoker()
{
    // call CefShutdown : close web view thread, nhncefprocess.exe
    ExitCef();
    
    auto FreeDll = [](void** Handle){
        if (*Handle != nullptr)
        {
            FPlatformProcess::FreeDllHandle(*Handle);
            *Handle = nullptr;
        }
    };

    if (GDLLHandle)
    {
        FMemory::Memset(&Functions, 0, sizeof(NhnCefWindowsWrapper::FFunctionList));
    }
    
    FreeDll(&GDLLHandle);
    FreeDll(&GCEF3DllHandle);
    FreeDll(&ElfHandle);
    FreeDll(&D3DHandle);
    FreeDll(&GLESHandle);
    FreeDll(&EGLHandle);
}

void FNhnCefWindowsInvoker::LoadMethods()
{
    if (GDLLHandle == nullptr)
    {
        return;
    }

    void* LambdaDLLHandle = GDLLHandle;
    
    auto GetCefDllExport = [&LambdaDLLHandle](const TCHAR* FuncName, bool& bInSuccess) -> void*
    {
        if (bInSuccess)
        {
            void* FuncPtr = FPlatformProcess::GetDllExport(LambdaDLLHandle, FuncName);
            if (FuncPtr == nullptr)
            {
                bInSuccess = false;
                NHNWEBVIEW_LOG_ERROR("Failed to load nhncef sdk dll"); 
                FPlatformProcess::FreeDllHandle(LambdaDLLHandle);
                LambdaDLLHandle = nullptr;
            }
            return FuncPtr;
        }
        
        return nullptr;
    };

    using namespace NhnCefWindowsWrapper;
    
#define NHNCEF_DLL_EXPORT(Name) \
    Functions.##Name = static_cast<TCef##Name>(GetCefDllExport(TEXT(#Name), bSuccess));
    
    bool bSuccess = true;
    NHNCEF_DLL_EXPORT(Initialize);
    NHNCEF_DLL_EXPORT(CreateWeb);
    NHNCEF_DLL_EXPORT(RemoveWeb);
    NHNCEF_DLL_EXPORT(LoadWeb);
    NHNCEF_DLL_EXPORT(ResizeWeb);
    NHNCEF_DLL_EXPORT(ShowScrollbar);
    NHNCEF_DLL_EXPORT(UpdateWeb);
    NHNCEF_DLL_EXPORT(ExitCef);
    NHNCEF_DLL_EXPORT(InputWeb);
    NHNCEF_DLL_EXPORT(GoBackForwardHome);
    NHNCEF_DLL_EXPORT(CanGoBackForward);
    NHNCEF_DLL_EXPORT(ExecuteJavaScript);
    NHNCEF_DLL_EXPORT(ReserveInvalidRedirectUrlSchemes);
    NHNCEF_DLL_EXPORT(SetDownloadCompleteOption);
    NHNCEF_DLL_EXPORT(SetDebugEnable);
    NHNCEF_DLL_EXPORT(SetOnFocusedNodeChanged);
    NHNCEF_DLL_EXPORT(SetOnKeyEvent);
    NHNCEF_DLL_EXPORT(SetOnImeCompositionRangeChanged);
    NHNCEF_DLL_EXPORT(ImeSetComposition);
    NHNCEF_DLL_EXPORT(ImeCommitText);
    NHNCEF_DLL_EXPORT(ImeFinishComposingText);
    NHNCEF_DLL_EXPORT(ImeCancelComposition);
    NHNCEF_DLL_EXPORT(GetCompositionBoundIndex);
    NHNCEF_DLL_EXPORT(GetCompositionBoundSize);
    NHNCEF_DLL_EXPORT(GetCompositionBound);
    NHNCEF_DLL_EXPORT(SetPopupIcon);

    check(bSuccess);
}

bool FNhnCefWindowsInvoker::Initialize(const FString& Locale)
{
    return Functions.Initialize(TCHAR_TO_ANSI(*Locale));
}

void FNhnCefWindowsInvoker::CreateWeb(const int32 WebIndex, const int32 X, const int32 Y, const int32 Width, const int32 Height, void* Buffer, const int32 Option)
{
    Functions.CreateWeb(WebIndex, X, Y, Width, Height, Buffer, Option);
}

void FNhnCefWindowsInvoker::RemoveWeb(const int32 WebIndex)
{
    Functions.RemoveWeb(WebIndex);
}

void FNhnCefWindowsInvoker::LoadWeb(const int32 WebIndex, const FString URL, const bool bIsShowScrollbar)
{
    Functions.LoadWeb(WebIndex, TCHAR_TO_ANSI(*URL), bIsShowScrollbar);
}

void FNhnCefWindowsInvoker::ResizeWeb(const int32 WebIndex, void* Buffer, const int32 Width, const int32 Height)
{
    Functions.ResizeWeb(WebIndex, Buffer, Width, Height);
}

void FNhnCefWindowsInvoker::ShowScrollbar(const int32 WebIndex, const bool bIsShow)
{
    Functions.ShowScrollbar(WebIndex, bIsShow);
}

int32 FNhnCefWindowsInvoker::UpdateWeb(const int32 WebIndex, FString& URL, int32& AdditionalInfo)
{
    static char TempUrl[4096] = { 0, };
    FMemory::Memzero(TempUrl, sizeof(TempUrl));
    char* TempUrlPtr = TempUrl;
    char** TempUrlPtrPtr = &TempUrlPtr;

    int32 TempAdditionalInfo = 0;
    const int32 Ret = Functions.UpdateWeb(WebIndex, TempUrlPtrPtr, &TempAdditionalInfo);

    URL = FString(UTF8_TO_TCHAR(*TempUrlPtrPtr));
    AdditionalInfo = TempAdditionalInfo;
    
    return Ret;
}

void FNhnCefWindowsInvoker::ExitCef()
{
    Functions.ExitCef();
}

void FNhnCefWindowsInvoker::InputWeb(const int32 WebIndex, const int32 Flags, const int32 X, const int32 Y)
{
    Functions.InputWeb(WebIndex, Flags, X, Y);
}

void FNhnCefWindowsInvoker::GoBackForwardHome(const int32 WebIndex, const int32 Direction)
{
    Functions.GoBackForwardHome(WebIndex, Direction);
}

bool FNhnCefWindowsInvoker::CanGoBackForward(const int32 WebIndex, const int32 Direction)
{
    return Functions.CanGoBackForward(WebIndex, Direction);
}

void FNhnCefWindowsInvoker::ExecuteJavaScript(const int32 WebIndex, const FString JavaScript)
{
    Functions.ExecuteJavaScript(WebIndex, TCHAR_TO_ANSI(*JavaScript));
}

void FNhnCefWindowsInvoker::ReserveInvalidRedirectUrlSchemes(const FString Schemes)
{
    Functions.ReserveInvalidRedirectUrlSchemes(TCHAR_TO_ANSI(*Schemes));
}

void FNhnCefWindowsInvoker::SetDownloadCompleteOption(int Option)
{
    Functions.SetDownloadCompleteOption(Option);
}

void FNhnCefWindowsInvoker::SetDebugEnable(const bool bEnable)
{
    Functions.SetDebugEnable(bEnable);
}

void FNhnCefWindowsInvoker::SetOnFocusedNodeChanged(const int32 WebIndex, const OnFocusedNodeChangedDelegate& Callback)
{
    Functions.SetOnFocusedNodeChanged(WebIndex, Callback);
}

void FNhnCefWindowsInvoker::SetOnKeyEvent(const int32 WebIndex, const OnKeyEventDelegate& Callback)
{
    Functions.SetOnKeyEvent(WebIndex, Callback);
}

void FNhnCefWindowsInvoker::SetOnImeCompositionRangeChanged(const int32 WebIndex, const OnImeCompositionRangeChangedDelegate& Callback)
{
    Functions.SetOnImeCompositionRangeChanged(WebIndex, Callback);
}

void FNhnCefWindowsInvoker::ImeSetComposition(const int32 WebIndex, const FString Text, const int32 BeginIndex, const int32 Length)
{
    Functions.ImeSetComposition(WebIndex, TCHAR_TO_WCHAR(*Text), BeginIndex, Length);
}

void FNhnCefWindowsInvoker::ImeCommitText(const int32 WebIndex, const FString Text)
{
    Functions.ImeCommitText(WebIndex, TCHAR_TO_WCHAR(*Text));
}

void FNhnCefWindowsInvoker::ImeFinishComposingText(const int32 WebIndex, const bool bKeepSelection)
{
    Functions.ImeFinishComposingText(WebIndex, bKeepSelection);
}

void FNhnCefWindowsInvoker::ImeCancelComposition(const int32 WebIndex)
{
    Functions.ImeCancelComposition(WebIndex);
}

int32 FNhnCefWindowsInvoker::GetCompositionBoundIndex(const int32 WebIndex, const int32 X, const int32 Y)
{
    return Functions.GetCompositionBoundIndex(WebIndex, X, Y);
}

int32 FNhnCefWindowsInvoker::GetCompositionBoundSize(const int32 WebIndex)
{
    return Functions.GetCompositionBoundSize(WebIndex);
}

void FNhnCefWindowsInvoker::GetCompositionBound(const int32 WebIndex, const int32 BoundIndex, int32& X, int32& Y, int32& Width, int32& Height)
{
    if (Functions.GetCompositionBound != nullptr)
    {
        int32 OutX = 0;
        int32 OutY = 0;
        int32 OutWidth = 0;
        int32 OutHeight = 0;
        Functions.GetCompositionBound(WebIndex, BoundIndex, &OutX, &OutY, &OutWidth, &OutHeight);

        X = OutX;
        Y = OutY;
        Width = OutWidth;
        Height = OutHeight;
    }
}

void FNhnCefWindowsInvoker::SetPopupIcon(const int32 WebIndex, const FString& popupIconPath)
{
    return Functions.SetPopupIcon(WebIndex, TCHAR_TO_ANSI(*popupIconPath));
}