#include "GamebaseWindowsBrowserLauncher.h"

#if PLATFORM_WINDOWS

#include "GamebaseDebugLogger.h"
#include "GamebaseTicker.h"
#include "GamebaseTimer.h"
#include "Windows/AllowWindowsPlatformTypes.h"

namespace GamebaseBrowserHelper
{
    enum class EBrowserType
    {
        None,
        Chrome,
        Edge,
        Firefox
    };

    FString GetDefaultBrowserProgId()
    {
        const FString RegPath = TEXT("Software\\Microsoft\\Windows\\Shell\\Associations\\UrlAssociations\\http\\UserChoice");
        const FString RegValueName = TEXT("ProgId");
            
        FString ProgId;
        FWindowsPlatformMisc::QueryRegKey(HKEY_CURRENT_USER, *RegPath, *RegValueName, ProgId);
        return ProgId;
    }
    
    FString ExtractExecutablePath(const FString& FullCommand)
    {
        FString TrimmedCommand = FullCommand.TrimStartAndEnd();

        if (TrimmedCommand.StartsWith(TEXT("\"")))
        {
            int32 EndQuoteIndex;
            if (TrimmedCommand.Mid(1).FindChar('"', EndQuoteIndex))
            {
                return TrimmedCommand.Mid(1, EndQuoteIndex);
            }
        }
            
        int32 SpaceIndex;
        if (TrimmedCommand.FindChar(' ', SpaceIndex))
        {
            return TrimmedCommand.Left(SpaceIndex);
        }
            
        return TrimmedCommand;
    }
    
    FString GetBrowserExecutablePath(const FString& ProgId)
    {
        if (!ProgId.IsEmpty())
        {
            const FString RegCommandPath = FString::Printf(TEXT("Software\\Classes\\%s\\shell\\open\\command"), *ProgId);
            
            FString BrowserPath;
            if (FWindowsPlatformMisc::QueryRegKey(HKEY_CURRENT_USER, *RegCommandPath, TEXT(""), BrowserPath) ||
                FWindowsPlatformMisc::QueryRegKey(HKEY_LOCAL_MACHINE, *RegCommandPath, TEXT(""), BrowserPath))
            {
                return ExtractExecutablePath(BrowserPath);
            }
        }
        
        return {};
    }

    TSet<HWND> GetAllTopLevelWindows()
    {
        TSet<HWND> Result;
        EnumWindows([](HWND hwnd, LPARAM lParam) -> BOOL {
            if (IsWindowVisible(hwnd)) {
                reinterpret_cast<TSet<HWND>*>(lParam)->Add(hwnd);
            }
            return TRUE;
        }, reinterpret_cast<LPARAM>(&Result));
        return Result;
    }

    FString GetWindowTitle(HWND hwnd)
    {
        wchar_t title[256];
        GetWindowText(hwnd, title, 256);
        return FString(title);
    }

    FString GetWindowClass(HWND hwnd)
    {
        wchar_t className[256];
        GetClassNameW(hwnd, className, 256);
        return FString(className);
    }

    bool IsChromeAppWindow(HWND hwnd, const TArray<FString>& TitleTags)
    {
        const FString ClassName = GetWindowClass(hwnd);
        const FString Title = GetWindowTitle(hwnd);
        if (ClassName == TEXT("Chrome_WidgetWin_1"))
        {
            for (const FString& Tag : TitleTags)
            {
                if (Title.Contains(Tag))
                {
                    return true;
                }
            }
        }
        return false;
    }
}

#include "Windows/HideWindowsPlatformTypes.h"

void FGamebaseWindowsBrowserLauncher::Launch(
    const FLaunchParams& Params,
    const FLaunchCallback& LaunchCallback,
    const FShouldCloseCallback& ShouldCloseCallback)
{
    if (!Params.MonitoringTag.IsSet())
    {   
        FGamebaseGenericBrowserLauncher::Launch(Params, LaunchCallback, ShouldCloseCallback);
        return;
    }
    
    if (Params.Url.IsEmpty())
    {
        LaunchCallback(EGamebaseBrowserLaunchResult::Failed, TEXT("The URL is empty."));
        return;
    }

    using namespace GamebaseBrowserHelper;
    
    const FString BrowserProgId = GetDefaultBrowserProgId();
    const FString BrowserExePath = GetBrowserExecutablePath(BrowserProgId);

    if (BrowserProgId.IsEmpty() || BrowserExePath.IsEmpty())
    {
        GAMEBASE_LOG_GLOBAL_DEBUG("The Default Browser information was not found in the registry.");
        FGamebaseGenericBrowserLauncher::Launch(Params, LaunchCallback, ShouldCloseCallback);
        return;
    }
    
    FString ExeParams;
    if (BrowserExePath.Contains(TEXT("chrome.exe"), ESearchCase::IgnoreCase) || BrowserExePath.Contains(TEXT("msedge.exe"), ESearchCase::IgnoreCase))
    {
        ExeParams = FString::Printf(TEXT("--app=\"%s\""), *Params.Url);
    }
    else
    {
        GAMEBASE_LOG_GLOBAL_DEBUG("Unsupported browser: %s", *BrowserProgId);
        FGamebaseGenericBrowserLauncher::Launch(Params, LaunchCallback, ShouldCloseCallback);
        return;
    }
    
    TSet<HWND> BeforeWindows = GetAllTopLevelWindows();
    
    uint32 ProcessID = 0;
    FProcHandle ProcHandle = FPlatformProcess::CreateProc(*BrowserExePath, *ExeParams, true, false, false, &ProcessID, 0, nullptr, nullptr);
    if (!ProcHandle.IsValid())
    {
        FGamebaseGenericBrowserLauncher::Launch(Params, LaunchCallback, ShouldCloseCallback);
        return;
    }
    
    AsyncTask(ENamedThreads::AnyBackgroundThreadNormalTask, [BeforeWindows, Params, LaunchCallback, ShouldCloseCallback] {
        constexpr float RetryInterval = 1.0f;
        int MaxRetries = 10;
        if(Params.Timeout.IsSet() && Params.Timeout.GetValue() > 0)
        {
            MaxRetries = Params.Timeout.GetValue() / RetryInterval;    
        }
        int RetryCount = 0;

        const TArray<FString> TitleTags = Params.MonitoringTag.GetValue();

        HWND TrackedHwnd = nullptr;

        FPlatformProcess::Sleep(0.01f);
        while (RetryCount < MaxRetries)
        {
            TSet<HWND> AfterWindows = GetAllTopLevelWindows();
            for (const HWND WindowHwnd : AfterWindows)
            {
                if (!BeforeWindows.Contains(WindowHwnd) && IsChromeAppWindow(WindowHwnd, TitleTags))
                {
                    TrackedHwnd = WindowHwnd;
                    break;
                }
            }

            if (TrackedHwnd != nullptr)
            {
                break;
            }

            RetryCount++;
            FPlatformProcess::Sleep(RetryInterval);
        }

        AsyncTask(ENamedThreads::GameThread, [TrackedHwnd, Params, LaunchCallback, ShouldCloseCallback] {
            if (TrackedHwnd == nullptr)
            {
                if (ShouldCloseCallback && ShouldCloseCallback())
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("Received a window close event before the window could be found.");
                    return;
                }
            
                GAMEBASE_LOG_GLOBAL_DEBUG("Failed to find the browser window with tag.");
                LaunchCallback(EGamebaseBrowserLaunchResult::Failed, TEXT("External browser window not found"));
                return;
            }
        
            if (ShouldCloseCallback && ShouldCloseCallback())
            {
                PostMessageW(TrackedHwnd, WM_CLOSE, 0, 0);
                return;
            }
            
            constexpr float CloseCheckInterval = 1.0f;
            
            const auto TickerHandle = GamebaseTicker::AddTicker(FTickerDelegate::CreateLambda([TrackedHwnd, LaunchCallback, ShouldCloseCallback](float DeltaTime) -> bool {
                if (ShouldCloseCallback && ShouldCloseCallback())
                {
                    PostMessageW(TrackedHwnd, WM_CLOSE, 0, 0);
                    LaunchCallback(EGamebaseBrowserLaunchResult::Cancelled, {});
                    return false;
                }
                
                if (!IsWindow(TrackedHwnd))
                {
                    LaunchCallback(EGamebaseBrowserLaunchResult::Closed, TEXT("Browser app window closed"));
                    return false;
                }
                return true;
            }), CloseCheckInterval);
            
            if (Params.Timeout.IsSet()) 
            {
                GamebaseTimer::AddTimer([TrackedHwnd, Params, LaunchCallback, ShouldCloseCallback, TickerHandle] {
                    if (!TickerHandle.IsValid() || (ShouldCloseCallback && ShouldCloseCallback()))
                    {
                        return;
                    }
                    
                    GamebaseTicker::RemoveTicker(TickerHandle);
                    
                    if (!IsWindow(TrackedHwnd))
                    {
                        LaunchCallback(EGamebaseBrowserLaunchResult::Closed, TEXT("Browser app window closed"));
                    }
                    else
                    {
                        LaunchCallback(EGamebaseBrowserLaunchResult::Timeout, TEXT("External browser wait time has timed out"));
                    }
                }, Params.Timeout.GetValue());
            }
        });
    });
}

#endif