#include "GamebaseWebViewEngineSystem.h"

#if WITH_GAMEBASE_WEBVIEW_ENGINE_PLUGIN

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorCode.h"
#include "GamebaseWebBrowserManager.h"
#include "GamebaseWebBrowserTypes.h"
#include "WebView/GamebaseWebViewConstants.h"
#include "WebView/GamebaseWebViewOpenParams.h"

FGamebaseWebViewEngineSystem::FGamebaseWebViewEngineSystem(const TWeakObjectPtr<UGameInstance>& GameInstance)
    : FGamebaseWebViewSystem(GameInstance)
    , BrowserManager(GamebaseWebBrowserManager::Create())
{
}

FGamebaseWebViewEngineSystem::~FGamebaseWebViewEngineSystem()
{
}

void FGamebaseWebViewEngineSystem::Open(const FGamebaseWebViewOpenParams& Params)
{
    if (BrowserManager.IsValid() == false)
    {
        Params.CloseCallback(FGamebaseErrorResult(FGamebaseError(
            GamebaseErrorCode::WEBVIEW_UNKNOWN_ERROR,
            TEXT("BrowserManager is invalid."),
            GamebaseWebView::Domain)));
        return;
    }
    
    if (OwningGameInstance.IsValid() == false)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("OwningGameInstance is invalid.");
        Params.CloseCallback(FGamebaseErrorResult(FGamebaseError(
            GamebaseErrorCode::UNREAL_GET_ENGINE_INFO_ERROR,
            TEXT("OwningGameInstance is invalid."),
            GamebaseWebView::Domain)));
        return;
    }
    
    UGameViewportClient* ViewportClient = OwningGameInstance->GetGameViewportClient();
    if (ViewportClient == nullptr)
    {
        Params.CloseCallback(FGamebaseErrorResult(FGamebaseError(
            GamebaseErrorCode::UNREAL_GET_ENGINE_INFO_ERROR,
            TEXT("Not found viewport"),
            GamebaseWebView::Domain)));
        return;
    }
    
    FGamebaseWebBrowserParams BrowserParams;
    BrowserParams.Url = Params.Url;
    BrowserParams.ViewportClient = ViewportClient;

    FGamebaseWebBrowserWidgetSettings WidgetSettings;
    WidgetSettings.BackgroundColor = Params.BackgroundColor;
    WidgetSettings.ViewportSize = Params.ViewportSize;

    switch (Params.Type)
    {
        case FGamebaseWebViewOpenParams::FloatingPopup:
            {
                WidgetSettings.ViewType = EGamebaseWebBrowserViewType::FloatingPopup;
                break;
            }
        case FGamebaseWebViewOpenParams::Window:
        default:
            {
                WidgetSettings.ViewType = EGamebaseWebBrowserViewType::Window;
                break;
            }
    }
    
    if (Params.Navigation.IsSet())
    {
        const auto& NavigationInfo = Params.Navigation.GetValue();
        FGamebaseWebBrowserWidgetSettings::FNavigation Navigation;
        Navigation.Title = NavigationInfo.Title;
        Navigation.BarColor = NavigationInfo.BarColor;
        Navigation.BarHeight = NavigationInfo.BarHeight;
        Navigation.TitleColor = NavigationInfo.TitleColor;
        Navigation.IconTintColor = NavigationInfo.IconTintColor;
        Navigation.bIsBackButton = NavigationInfo.bIsBackButton;
        Navigation.bIsForwardButton = NavigationInfo.bIsForwardButton;
        Navigation.bVisibleButtons = NavigationInfo.bVisibleButtons;

        WidgetSettings.Navigation = Navigation;
    }

    BrowserParams.SchemeList = Params.SchemeList;
    BrowserParams.WidgetSettings = MakeShared<FGamebaseWebBrowserWidgetSettings>(WidgetSettings);
    BrowserParams.ContextSettings = MakeShared<FGamebaseWebBrowserContextSettings>(FGuid::NewGuid().ToString());

    BrowserParams.CloseCallback = FGamebaseWebBrowserClose::CreateLambda([this, CloseCallback = Params.CloseCallback](const FGamebaseErrorResult& Result)
    {
        CloseCallback(Result);
    });
    
    BrowserParams.SchemeEventCallback = FGamebaseWebBrowserSchemeEvent::CreateLambda([this, EventCallback = Params.EventCallback](const FString& Url)
    {
        EventCallback(FGamebaseWebviewEventResult(Url));
    });

    BrowserManager->Open(BrowserParams);
}

void FGamebaseWebViewEngineSystem::Close()
{
    if (BrowserManager.IsValid() == false)
    {
        return;
    }

    BrowserManager->Close();
}

#endif