#include "GamebasePurchaseAdapterManager.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseInternalData.h"
#include "GamebasePurchaseDefines.h"
#include "GamebaseStandaloneSubsystem.h"

#if WITH_GAMEBASE_STANDALONE_PURCHASE
#include "Types/GamebasePurchaseConfiguration.h"
#include "GamebaseStandalonePurchaseAdapter.h"
#endif

FGamebasePurchaseAdapterManager::FGamebasePurchaseAdapterManager(const FGamebaseInternalDataPtr& InternalData)
    : InternalData(InternalData)
{
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    UGamebaseStandaloneSubsystem* Subsystem = UGameInstance::GetSubsystem<UGamebaseStandaloneSubsystem>(InternalData->GetGameInstance().Get());
    LoadedAdapters.Initialize(Subsystem);
#endif

}

FGamebasePurchaseAdapterManager::~FGamebasePurchaseAdapterManager()
{
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    Adapter.Reset();
    LoadedAdapters.Deinitialize();
#endif

}

bool FGamebasePurchaseAdapterManager::IsDisable() const
{
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    return false;
#else
    return true;
#endif
}

void FGamebasePurchaseAdapterManager::SetDebugMode(const bool bIsDebugMode)
{
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    if (Adapter.IsValid())
    {
        Adapter->SetDebugMode(bIsDebugMode);
    }
#endif
}

void FGamebasePurchaseAdapterManager::Initialize(
    const FGamebasePurchaseConfiguration& Configuration,
    const FInitializeFunc& Callback)
{
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    Adapter = LoadedAdapters.GetAdapter(Configuration.StoreCode);
    if (Adapter.IsValid())
    {
        Adapter->Initialize(Configuration, [this, Callback](const FGamebaseError* Error)
        {
            Adapter->SetDebugMode(FGamebaseDebugLogger::IsDebugLog);
            Callback(Error);
        });
    }
    else
    {
        Callback(&MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
            GamebaseErrorCode::PURCHASE_NOT_INITIALIZED,
            TEXT("The adapter matching the store code is not included."),
            GamebasePurchase::Domain).Get());
    }
#endif
}


void FGamebasePurchaseAdapterManager::SetGamebaseUserInfo(const FString& GamebaseUserId, const FString& GamebaseToken)
{
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    if (const auto Error = CheckApiAvailability())
    {
        return;
    }

    Adapter->SetGamebaseUserId(GamebaseUserId, GamebaseToken);
#endif
}

void FGamebasePurchaseAdapterManager::SetDisplayLanguage(const FString& LanguageCode)
{
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    if (const auto Error = CheckApiAvailability())
    {
        return;
    }

    Adapter->SetDisplayLanguage(LanguageCode);
#endif
}

void FGamebasePurchaseAdapterManager::RequestReprocessPurchased(const FString& GamebaseItemListJsonString, const FReprocessingFunc& Callback)
{
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    if (const auto Error = CheckApiAvailability())
    {
        return;
    }

    Adapter->RequestReprocessPurchased(GamebaseItemListJsonString, Callback);
#endif
}

void FGamebasePurchaseAdapterManager::RequestPurchase(int64 ItemSeq, const FPurchasableReceiptFunc& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebasePurchaseAdapterManager::RequestPurchase(const GamebasePurchase::FIapProductResponse& ProductData, const FPurchasableReceiptFunc& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        Callback(FGamebasePurchasableReceiptResult(*Error.Get()));
        return;
    }
    
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    Adapter->RequestPurchase(ProductData.MarketItemId, ProductData.GamebaseProductId, FIapPurchaseReceiptDelegate::CreateLambda(
        [&, ProductData, Callback](const TOptional<FIapPurchaseReceipt>& ResponseData, const FGamebaseError* Error)
        {
            if (Error != nullptr)
            {
                Callback(FGamebasePurchasableReceiptResult(*Error));
                return;
            }

            if (ResponseData.IsSet() == false)
            {
                Callback(FGamebasePurchasableReceiptResult(TOptional<FGamebasePurchasableReceipt>()));
                return;
            }
                    
            FGamebasePurchasableReceipt ResultData;
            ResultData.GamebaseProductId = ProductData.GamebaseProductId;
            ResultData.ItemSeq = ResponseData->ItemSeq;
            ResultData.Price = ResponseData->Price;
            ResultData.Currency = ResponseData->Currency;
            ResultData.PaymentSeq = ResponseData->PaymentSeq;
            ResultData.PurchaseToken = ResponseData->PurchaseToken;
            ResultData.MarketItemId = ProductData.MarketItemId;
            ResultData.ProductType = ResponseData->ProductType;
            ResultData.UserId = ResponseData->UserId;
            ResultData.PaymentId = ResponseData->PaymentId;
            ResultData.OriginalPaymentId = ResponseData->OriginalPaymentId;
            ResultData.PurchaseTime = ResponseData->PurchaseTime;
            ResultData.ExpiryTime = ResponseData->ExpiryTime;
            ResultData.StoreCode = ResponseData->StoreCode; 
            ResultData.bIsPromotion = ResponseData->bIsPromotion;
            ResultData.bIsTestPurchase = ResponseData->bIsTestPurchase;
            
            Callback(FGamebasePurchasableReceiptResult(ResultData));
        }));
#endif
}

void FGamebasePurchaseAdapterManager::RequestPurchase(const GamebasePurchase::FIapProductResponse& ProductData, const FString& Payload, const FPurchasableReceiptFunc& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        Callback(FGamebasePurchasableReceiptResult(*Error.Get()));
        return;
    }
    
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    Adapter->RequestPurchase(ProductData.MarketItemId, ProductData.GamebaseProductId, Payload, FIapPurchaseReceiptDelegate::CreateLambda(
        [&, ProductData, Payload, Callback](const TOptional<FIapPurchaseReceipt>& ResponseData, const FGamebaseError* Error)
        {
            if (Error != nullptr)
            {
                Callback(FGamebasePurchasableReceiptResult(*Error));
                return;
            }

            if (ResponseData.IsSet() == false)
            {
                Callback(FGamebasePurchasableReceiptResult(TOptional<FGamebasePurchasableReceipt>()));
                return;
            }
    
            FGamebasePurchasableReceipt ResultData;
            ResultData.GamebaseProductId = ProductData.GamebaseProductId;
            ResultData.ItemSeq = ResponseData->ItemSeq;
            ResultData.Price = ResponseData->Price;
            ResultData.Currency = ResponseData->Currency;
            ResultData.PaymentSeq = ResponseData->PaymentSeq;
            ResultData.PurchaseToken = ResponseData->PurchaseToken;
            ResultData.MarketItemId = ProductData.MarketItemId;
            ResultData.ProductType = ResponseData->ProductType;
            ResultData.UserId = ResponseData->UserId;
            ResultData.PaymentId = ResponseData->PaymentId;
            ResultData.OriginalPaymentId = ResponseData->OriginalPaymentId;
            ResultData.PurchaseTime = ResponseData->PurchaseTime;
            ResultData.ExpiryTime = ResponseData->ExpiryTime;
            ResultData.StoreCode = ResponseData->StoreCode; 
            ResultData.Payload = Payload;
            ResultData.bIsPromotion = ResponseData->bIsPromotion;
            ResultData.bIsTestPurchase = ResponseData->bIsTestPurchase;
            
            Callback(FGamebasePurchasableReceiptResult(ResultData));
        }));
#endif
}

void FGamebasePurchaseAdapterManager::RequestItemListOfNotConsumed(const FGamebasePurchasableConfiguration& Configuration, const FRequestGetProductWithMarketItemId& RequestFunc, const FPurchasableReceiptListFunc& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        Callback(FGamebasePurchasableReceiptListResult(*Error.Get()));
        return;
    }
    
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    const auto Delegate = FIapPurchaseReceiptListDelegate::CreateLambda([this, RequestFunc, Callback](const TOptional<FIapPurchaseReceiptArray>& IapResultArray, const FGamebaseError* Error)
    {
        if (Error != nullptr)
        {
            Callback(FGamebasePurchasableReceiptListResult(*Error));
            return;
        }

        if (IapResultArray.IsSet() == false)
        {
            Callback(FGamebasePurchasableReceiptListResult(TArray<FGamebasePurchasableReceipt>()));
            return;
        }
        
        const auto IapResultArrayData = IapResultArray.GetValue();
        
        TSharedRef<TPromise<TArray<FGamebasePurchasableReceipt>>> SharedPromise = MakeShared<TPromise<TArray<FGamebasePurchasableReceipt>>>();
        TFuture<TArray<FGamebasePurchasableReceipt>> Future = SharedPromise->GetFuture();
        
        AsyncTask(ENamedThreads::AnyBackgroundThreadNormalTask, [this, IapResultArrayData, RequestFunc, SharedPromise]
        {
            const int32 TotalDataCount = IapResultArrayData.Num();
            const TSharedRef<TArray<FGamebasePurchasableReceipt>, ESPMode::ThreadSafe> ReceivedData = MakeShared<TArray<FGamebasePurchasableReceipt>, ESPMode::ThreadSafe>();
            
            for (const auto& ResponseData : IapResultArrayData)
            {
                RequestFunc(ResponseData.MarketItemId, [this, ResponseData, TotalDataCount, ReceivedData, SharedPromise](const TArray<FGamebaseIapProductWithMarketItemIdResponse>& Result, const FGamebaseError* RequestError)
                {
                    if (RequestError != nullptr)
                    {
                        GAMEBASE_LOG_WARNING("not found item");
                        ReceivedData->Add(FGamebasePurchasableReceipt());
                
                        if (ReceivedData->Num() == TotalDataCount)
                        {
                            SharedPromise->SetValue(*ReceivedData);
                        }
                        return;
                    }

                    using namespace GamebasePurchase;
                
                    FGamebasePayloadResponse GamebasePayload;
                    GamebasePayload.FromJson(ResponseData.Payload);
                
                    const FIapProductWithMarketItemIdResponse* SearchItemPtr = Result.FindByPredicate(
                        [GamebasePayload](const FIapProductWithMarketItemIdResponse& Data)
                        {
                            return Data.GamebaseProductId == GamebasePayload.GamebaseProductId;
                        });
                    if (SearchItemPtr == nullptr)
                    {
                        GAMEBASE_LOG_DEBUG("GamebasePayload.Id: %s, not searched item", *GamebasePayload.StoreUserId);
                        SearchItemPtr = &Result[0];
                    }
                
                    GAMEBASE_LOG_DEBUG("GamebasePayload.Id: %s, SearchItem: %s", *GamebasePayload.StoreUserId, *SearchItemPtr->ToJson(false));
                
                    FGamebasePurchasableReceipt ResultData;
                    ResultData.GamebaseProductId = SearchItemPtr->GamebaseProductId;
                    ResultData.ItemSeq = ResponseData.ItemSeq;
                    ResultData.Price = ResponseData.Price;
                    ResultData.Currency = ResponseData.Currency;
                    ResultData.PaymentSeq = ResponseData.PaymentSeq;
                    ResultData.PurchaseToken = ResponseData.PurchaseToken;
                    ResultData.MarketItemId = ResponseData.MarketItemId;
                    ResultData.ProductType = ResponseData.ProductType;
                    ResultData.UserId = ResponseData.UserId;
                    ResultData.PaymentId = ResponseData.PaymentId;
                    ResultData.OriginalPaymentId = ResponseData.OriginalPaymentId;
                    ResultData.PurchaseTime = ResponseData.PurchaseTime;
                    ResultData.ExpiryTime = ResponseData.ExpiryTime;
                    ResultData.StoreCode = ResponseData.StoreCode;
                    ResultData.bIsPromotion = ResponseData.bIsPromotion;
                    ResultData.bIsTestPurchase = ResponseData.bIsTestPurchase;
                    
                    ReceivedData->Add(ResultData);

                    if (ReceivedData->Num() == TotalDataCount)
                    {
                        SharedPromise->SetValue(*ReceivedData);
                    }
                });
                
                FPlatformProcess::Sleep(0.01f);
            }
        });

        AsyncTask(ENamedThreads::AnyBackgroundThreadNormalTask, [Future = MoveTemp(Future), Callback]() mutable {
            const auto& Responses = Future.Get();

            AsyncTask(ENamedThreads::GameThread, [Responses, Callback]() {
                Callback(FGamebasePurchasableReceiptListResult(Responses));
            });
        });
    });
    
    Adapter->RequestItemListOfNotConsumed(Configuration, Delegate);
#endif
}

void FGamebasePurchaseAdapterManager::RequestActivatedPurchases(const FPurchasableReceiptListFunc& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebasePurchaseAdapterManager::RequestActivatedPurchases(const FGamebasePurchasableConfiguration& Configuration, const FPurchasableReceiptListFunc& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebasePurchaseAdapterManager::RequestItemListPurchasable(const TArray<GamebasePurchase::FIapProductWithMarketItemIdResponse>& Products, const FPurchasableItemListFunc& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        Callback(FGamebasePurchasableItemListResult(*Error.Get()));
        return;
    }

#if WITH_GAMEBASE_STANDALONE_PURCHASE
    Adapter->SetDisplayLanguage(InternalData.Get()->GetDisplayLanguageCode());
    Adapter->RequestItemListPurchasable( FIapPurchasableItemListDelegate::CreateLambda(
    [&, Callback, Products](const TOptional<FIapPurchasableItemArray>& IapList, const FGamebaseError* Error)
    {
        if (Error != nullptr)
        {
            Callback(FGamebasePurchasableItemListResult(*Error));
            return;
        }
        
        if (IapList.IsSet() == false)
        {
            Callback(FGamebasePurchasableItemListResult(TArray<FGamebasePurchasableItem>()));
            return;
        }

        const auto IapListValue = IapList.GetValue();

        const auto ResultArray = MakeShareable(new TArray<FGamebasePurchasableItem>());

        for (const auto& Product : Products)
        {
            if (const auto FindIapProduct = IapListValue.FindByPredicate([Product](const FIapPurchasableItem& EventInfo) { return EventInfo.MarketItemId == Product.MarketItemId; }))
            {
                FGamebasePurchasableItem ResultData;
                ResultData.GamebaseProductId = Product.GamebaseProductId;
                ResultData.ItemSeq = FindIapProduct->ItemSeq;
                ResultData.Price = FindIapProduct->Price;
                ResultData.Currency = FindIapProduct->Currency;
                ResultData.ItemName = Product.GamebaseProductName;
                ResultData.MarketItemId = FindIapProduct->MarketItemId;
                ResultData.ProductType = FindIapProduct->ProductType;
                ResultData.LocalizedPrice = FindIapProduct->LocalizedPrice;
                ResultData.LocalizedTitle = FindIapProduct->LocalizedTitle;
                ResultData.LocalizedDescription = FindIapProduct->LocalizedDescription;
                ResultData.bIsActive = Product.bIsActive;

                ResultArray.Object->Add(ResultData);

                SavedProductList.Emplace(ResultData.MarketItemId, ResultData);
            }
        }
        
        Callback(FGamebasePurchasableItemListResult(*ResultArray.Object));
    }));
#endif
}

void FGamebasePurchaseAdapterManager::RequestItemListAtIAPConsole(const TArray<GamebasePurchase::FIapProductWithMarketItemIdResponse>& Products, const TArray<GamebasePurchase::FIapProductWithMarketItemIdResponse>& InvalidProducts, const FPurchasableItemListFunc& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        Callback(FGamebasePurchasableItemListResult(*Error.Get()));
        return;
    }

#if WITH_GAMEBASE_STANDALONE_PURCHASE
    Adapter->SetDisplayLanguage(InternalData.Get()->GetDisplayLanguageCode());
    Adapter->RequestItemListPurchasable(FIapPurchasableItemListDelegate::CreateLambda(
        [&, Callback, Products, InvalidProducts](const TOptional<FIapPurchasableItemArray>& IapList, const FGamebaseError* Error)
    {
        if (Error != nullptr)
        {
            Callback(FGamebasePurchasableItemListResult(*Error));
            return;
        }
        
        if (IapList.IsSet() == false)
        {
            Callback(FGamebasePurchasableItemListResult(TArray<FGamebasePurchasableItem>()));
            return;
        }
            
        auto ResultArray = MakeShareable(new TArray<FGamebasePurchasableItem>());

        for (const auto& Product : Products)
        {
            if (const auto FindIapProduct = IapList->FindByPredicate([Product](const FIapPurchasableItem& EventInfo) { return EventInfo.MarketItemId == Product.MarketItemId; }))
            {
                FGamebasePurchasableItem ResultData;
                ResultData.GamebaseProductId = Product.GamebaseProductId;
                ResultData.ItemSeq = FindIapProduct->ItemSeq;
                ResultData.Price = FindIapProduct->Price;
                ResultData.Currency = FindIapProduct->Currency;
                ResultData.ItemName = Product.GamebaseProductName;
                ResultData.MarketItemId = FindIapProduct->MarketItemId;
                ResultData.ProductType = FindIapProduct->ProductType;
                ResultData.LocalizedPrice = FindIapProduct->LocalizedPrice;
                ResultData.LocalizedTitle = FindIapProduct->LocalizedTitle;
                ResultData.LocalizedDescription = FindIapProduct->LocalizedDescription;
                ResultData.bIsActive = Product.bIsActive;
                
                ResultArray.Object->Add(ResultData);

                SavedProductList.Emplace(ResultData.MarketItemId, ResultData);
            }
        }
        
        for (const auto Product : InvalidProducts)
        {
            if (const auto FindIapProduct = IapList->FindByPredicate([Product](const FIapPurchasableItem& EventInfo) { return EventInfo.MarketItemId == Product.MarketItemId; }))
            {
                FGamebasePurchasableItem ResultData;
                ResultData.GamebaseProductId = Product.GamebaseProductId;
                ResultData.ItemSeq = FindIapProduct->ItemSeq;
                ResultData.Price = FindIapProduct->Price;
                ResultData.Currency = FindIapProduct->Currency;
                ResultData.ItemName = Product.GamebaseProductName;
                ResultData.MarketItemId = FindIapProduct->MarketItemId;
                ResultData.ProductType = FindIapProduct->ProductType;
                ResultData.LocalizedPrice = FindIapProduct->LocalizedPrice;
                ResultData.LocalizedTitle = FindIapProduct->LocalizedTitle;
                ResultData.LocalizedDescription = FindIapProduct->LocalizedDescription;
                ResultData.bIsActive = Product.bIsActive;
                
                ResultArray.Object->Add(ResultData);

                SavedProductList.Emplace(ResultData.MarketItemId, ResultData);
            }
        }
        
        Callback(FGamebasePurchasableItemListResult(*ResultArray.Object));
    }));
#endif
}

void FGamebasePurchaseAdapterManager::RequestSubscriptionsStatus(const FGamebasePurchasableConfiguration& Configuration, const FGamebasePurchasableSubscriptionStatusDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebasePurchaseAdapterManager::SetPromotionIAPHandler(const FPurchasableReceiptFunc& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

FGamebaseErrorPtr FGamebasePurchaseAdapterManager::CheckApiAvailability() const
{
#if WITH_GAMEBASE_STANDALONE_PURCHASE
    if (Adapter.IsValid() == false)
    {
        return MakeShared<FGamebaseError, ESPMode::ThreadSafe>(GamebaseErrorCode::PURCHASE_NOT_INITIALIZED, TEXT("No adapter was created for purchase"), GamebasePurchase::Domain);
    }
#endif

    return nullptr;
}

