﻿#include "GamebaseImageNoticePreferences.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseImageNoticeConstants.h"
#include "GamebaseJsonConverter.h"
#include "GamebaseSaveInfo.h"

void FGamebaseImageNoticePreferences::Initialize()
{
    const TOptional<FString> StoreData = GamebaseSaveInfo::GetStoreValue(AppId, GamebaseImageNotice::Preference::Key);
    if (StoreData.IsSet() == false)
    {
        return;
    }
    
    TSharedPtr<FJsonObject> Object;
    const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(StoreData.GetValue());
    if (!FJsonSerializer::Deserialize(Reader, Object) || !Object.IsValid())
    {
        return;
    }

    for (const auto& Info : Object->Values)
    {
        int64 Value;
        Info.Value->TryGetNumber(Value);

        NextPopupTimeMap.FindOrAdd(FCString::Atoi64(*Info.Key), Value);
    }
}

void FGamebaseImageNoticePreferences::AddNextPopupTime(const int64 NoticeId, const int64 NextPopupTimeMs)
{
    if (NextPopupTimeMap.Contains(NoticeId))
    {
        NextPopupTimeMap.Remove(NoticeId);
    }
    
    NextPopupTimeMap.Add(NoticeId, NextPopupTimeMs);
    Save();
}

void FGamebaseImageNoticePreferences::UpdateNextTimeData(const int64 NoticeId)
{
    if (NextPopupTimeMap.Contains(NoticeId) == false)
    {
        return;
    }
    
    const FTimespan CurrentTime = FDateTime::UtcNow() - FDateTime(1970, 1, 1);
    const int64 CurrentEpochTime = CurrentTime.GetTotalMilliseconds();
    const int64 SavedEpochTime = NextPopupTimeMap.FindRef(NoticeId);
    
    if (SavedEpochTime > CurrentEpochTime)
    {
        const FDateTime ConvertDate = FDateTime(1970, 1, 1) + FTimespan::FromMilliseconds(SavedEpochTime);
        GAMEBASE_LOG_GLOBAL_DEBUG("Re-exposure time: %s", *ConvertDate.ToString());
        
        return;
    }
    
    NextPopupTimeMap.Remove(NoticeId);
    Save();
}

bool FGamebaseImageNoticePreferences::ShouldHideUntilTimeForNoticeId(const int64 NoticeId) const
{
    return NextPopupTimeMap.Contains(NoticeId);
}

void FGamebaseImageNoticePreferences::Save()
{
    const TSharedRef<FJsonObject> SaveData = MakeShared<FJsonObject>();
    for (const auto& Data : NextPopupTimeMap)
    {
        SaveData->SetNumberField(LexToString(Data.Key), Data.Value);
    }

    GamebaseSaveInfo::SetStoreValue(AppId, GamebaseImageNotice::Preference::Key, GamebaseJsonConverter::JsonObjectToJsonString(SaveData));
}
