﻿#pragma once

#include "GamebaseInternalData.h"
#include "GamebaseInternalDataPointer.h"
#include "GamebaseInternalResult.h"
#include "GamebaseTicker.h"
#include "WebSocket/GamebaseWebSocketRequest.h"
#include "WebSocket/GamebaseWebSocketResponse.h"
#include "Types/GamebaseError.h"

class FGamebaseStandaloneServerPush;
class IWebSocket;
struct IGamebaseWebSocketRequest;

using FGamebaseWebSocketResponseResult = FGamebaseInternalResult<FGamebaseWebSocketResponse>;

class FGamebaseWebSocket final
{
    using FResponseFunction = TFunction<void(const FGamebaseWebSocketResponseResult&)>;
    using FConnectFunction = TFunction<void(const TOptional<FGamebaseError>&)>;

    enum class EZoneType
    {
        Alpha,
        Beta,
        Real
    };
    
    enum class EDomainType
    {
        Main,
        Sub
    };
    
public:
    explicit FGamebaseWebSocket(const FGamebaseInternalDataPtr& InternalData);
    ~FGamebaseWebSocket();

    void Initialize();
    
    void Connect(const FConnectFunction& Callback);
    void Disconnect(bool bNotifyOnDisconnect = true);
    bool IsConnected() const;

    template <typename FParameterType = FGamebaseBaseParameter, typename FPayloadType = FGamebaseBasePayload>
    void Request(const FString& ProductId, const FString& ApiId, const FString& ApiVersion,
                 const FParameterType& Parameter, const FPayloadType& Payload, const FResponseFunction& Callback);

    template <typename FParameterType = FGamebaseBaseParameter>
    void Request(const FString& ProductId, const FString& ApiId, const FString& ApiVersion,
                 const FParameterType& Parameter, const FResponseFunction& Callback);
    
    DECLARE_MULTICAST_DELEGATE_OneParam(FReceiveServerPush, const FGamebaseWebSocketResponse&);
    FReceiveServerPush OnReceiveServerPush;
    
private:
    FString GetUrl();
    
    void Connect(EDomainType DomainType);
    
    void Request(const IGamebaseWebSocketRequest& Request, const FResponseFunction& Callback);
    void Send(const FString& TransactionId, const FString& Data);
    
    void OnWebsocketConnected();
    void OnWebsocketConnectionError(const FString& Error);
    void OnWebsocketClosed(int32 StatusCode, const FString& Reason, bool bWasClean);
    void OnWebsocketMessage(const FString& MessageString);
    
    void NotifyAllRequests(const FGamebaseError& Error);
    void NotifyDisconnectAllRequests();

    void SendResponseCallback(const FString& TransactionId, const FGamebaseWebSocketResponseResult& Result);
private:
    FGamebaseInternalDataPtr InternalData;
    TSharedPtr<FGamebaseStandaloneServerPush> ServerPush;
    
    EZoneType SelectZone = EZoneType::Real;
    EDomainType SelectDomain = EDomainType::Main;
    
    TSharedPtr<IWebSocket> WebSocket;
    TMap<FString, FResponseFunction> ResponseFunctions;
    TMap<FString, FGamebaseTickerDelegateHandle> RequestTimoutHandlers;
    FConnectFunction ConnectCallback;
    
    mutable FCriticalSection DataGuard;
};

template <typename FParameterType, typename FPayloadType>
void FGamebaseWebSocket::Request(
    const FString& ProductId,
    const FString& ApiId,
    const FString& ApiVersion,
    const FParameterType& Parameter,
    const FPayloadType& Payload,
    const FResponseFunction& Callback)
{
    TGamebaseWebSocketRequest<FParameterType, FPayloadType> RequestObject(
        ProductId, ApiId, ApiVersion, InternalData->GetAppId(), InternalData->GetAccessToken());
    RequestObject.Parameters = Parameter;
    RequestObject.Payload = Payload;
    
    Request(RequestObject, Callback);
}

template <typename FParameterType>
void FGamebaseWebSocket::Request(
    const FString& ProductId,
    const FString& ApiId,
    const FString& ApiVersion,
    const FParameterType& Parameter,
    const FResponseFunction& Callback)
{
    TGamebaseWebSocketRequest<FParameterType> RequestObject(
        ProductId, ApiId, ApiVersion, InternalData->GetAppId(), InternalData->GetAccessToken());
    RequestObject.Parameters = Parameter;
    
    Request(RequestObject, Callback);
}

