#include "GamebaseWebSocket.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorCode.h"
#include "GamebaseInternalReport.h"
#include "GamebaseJsonSecurity.h"
#include "GamebaseSystemUtils.h"
#include "GamebaseTimer.h"
#include "IWebSocket.h"
#include "WebSocketsModule.h"
#include "WebSocket/GamebaseWebSocketConstants.h"

FGamebaseWebSocket::FGamebaseWebSocket(const FGamebaseInternalDataPtr& InternalData)
    : InternalData(InternalData)
{
}

FGamebaseWebSocket::~FGamebaseWebSocket()
{
    Disconnect(false);
}

void FGamebaseWebSocket::Initialize()
{
    static auto GetZoneType = [](const FString& Zone) -> EZoneType
    {
        if (Zone.Equals(TEXT("alpha")))
        {
            return EZoneType::Alpha;
        }
        if (Zone.Equals(TEXT("beta")))
        {
            return EZoneType::Beta;
        }

        return EZoneType::Real;
    };

    SelectZone = GetZoneType(InternalData->GetZoneType());
    SelectDomain = EDomainType::Main;
}

void FGamebaseWebSocket::Connect(const FConnectFunction& Callback)
{
    if (IsConnected())
    {
        Callback(TOptional<FGamebaseError>());
        return;
    }
    
    if (ConnectCallback)
    {
        GAMEBASE_LOG_WARNING("WebSocket is already connecting.");
        return;
    }
    
    ConnectCallback = Callback;

    Connect(EDomainType::Main);
}

void FGamebaseWebSocket::Disconnect(const bool bNotifyOnDisconnect)
{
    if (WebSocket.IsValid())
    {
        WebSocket->OnConnected().RemoveAll(this);
        WebSocket->OnConnectionError().RemoveAll(this);
        WebSocket->OnClosed().RemoveAll(this);
        WebSocket->OnMessage().RemoveAll(this);

        WebSocket->Close();
        WebSocket = nullptr;

        GAMEBASE_LOG_DEBUG("Websocket disconnected.");
    }

    if (bNotifyOnDisconnect)
    {
        NotifyDisconnectAllRequests();
    }
}

bool FGamebaseWebSocket::IsConnected() const
{
    if (WebSocket.IsValid() == false)
    {
        return false;
    }

    return WebSocket->IsConnected();
}

FString FGamebaseWebSocket::GetUrl()
{
    using namespace GamebaseWebSocket;

    switch (SelectZone)
    {
    case EZoneType::Alpha:      return Urls::Alpha[static_cast<int32>(SelectDomain)];
    case EZoneType::Beta:       return Urls::Beta[static_cast<int32>(SelectDomain)];
    case EZoneType::Real:       return Urls::Real[static_cast<int32>(SelectDomain)];
    }

    checkNoEntry();
    return {};
};

void FGamebaseWebSocket::Connect(EDomainType DomainType)
{
    if (GamebaseSystemUtils::IsNetworkConnected() == false)
    {
        GAMEBASE_LOG_WARNING("Network is not available.");
        if (ConnectCallback)
        {
            ConnectCallback(FGamebaseError(GamebaseErrorCode::SOCKET_ERROR, "Network is not available", GamebaseWebSocket::Domain));
            ConnectCallback = nullptr;
        }
        return;
    }

    Disconnect(false);

    SelectDomain = DomainType;
    
    const FString ServerUrl = GetUrl();
    const FString ServerProtocol = TEXT("wss");

    GAMEBASE_LOG_DEBUG("ServerUrl : %s", *GamebaseJsonSecurity::MaskingJson(ServerUrl));
    WebSocket = FWebSocketsModule::Get().CreateWebSocket(ServerUrl, ServerProtocol);

    WebSocket->OnConnected().AddRaw(this, &FGamebaseWebSocket::OnWebsocketConnected);
    WebSocket->OnConnectionError().AddRaw(this, &FGamebaseWebSocket::OnWebsocketConnectionError);
    WebSocket->OnClosed().AddRaw(this, &FGamebaseWebSocket::OnWebsocketClosed);
    WebSocket->OnMessage().AddRaw(this, &FGamebaseWebSocket::OnWebsocketMessage);

    WebSocket->Connect();

    GAMEBASE_LOG_DEBUG("Websocket connect started.");
}

void FGamebaseWebSocket::Request(const IGamebaseWebSocketRequest& Request, const FResponseFunction& Callback)
{
    if(GamebaseSystemUtils::IsNetworkConnected() == false)
    {
        GAMEBASE_LOG_WARNING("Network is not available.");
        if (Callback)
        {
            Callback(FGamebaseWebSocketResponseResult(FGamebaseError(GamebaseErrorCode::SOCKET_ERROR, "Network is not available", GamebaseWebSocket::Domain)));
        }
        return;
    }
    
    FScopeLock Lock(&DataGuard);
    ResponseFunctions.Emplace(Request.GetTransactionId(), Callback);
    Send(Request.GetTransactionId(), Request.ToJson(false));
}

void FGamebaseWebSocket::Send(const FString& TransactionId, const FString& Data)
{
    Connect([this, TransactionId, Data](const TOptional<FGamebaseError>& Error) {
        if (Error.IsSet())
        {
            NotifyAllRequests(Error.GetValue());
            return;
        }

        GAMEBASE_LOG_DEBUG("%s", *GamebaseJsonSecurity::MaskingJson(Data));
        
        WebSocket->Send(Data);
    
        const auto TimeoutHandle = GamebaseTimer::AddTimer([this, TransactionId]
        {
            RequestTimoutHandlers.Remove(TransactionId);
            if (ResponseFunctions.Contains(TransactionId) == false)
            {
                GAMEBASE_LOG_ERROR("The transaction not found (ID: %s)", *TransactionId);
                return;
            }
                
            const auto Delegate = ResponseFunctions.FindRef(TransactionId);
            Delegate(FGamebaseWebSocketResponseResult(FGamebaseError(GamebaseErrorCode::SOCKET_RESPONSE_TIMEOUT, "socket response timeout", GamebaseWebSocket::Domain, TransactionId)));
            ResponseFunctions.Remove(TransactionId);
        }, GamebaseWebSocket::ResponseTimeout);
    
        RequestTimoutHandlers.FindOrAdd(TransactionId, TimeoutHandle);
    });
}

void FGamebaseWebSocket::OnWebsocketConnected()
{
    GAMEBASE_LOG_DEBUG("Websocket is connected.");

    if (SelectDomain == EDomainType::Sub)
    {
        GamebaseInternalReport::Network::ChangeDomainSuccess(*InternalData, GetUrl());
    }
    
    if (ConnectCallback)
    {
        ConnectCallback(TOptional<FGamebaseError>());
        ConnectCallback = nullptr;
    }
}

void FGamebaseWebSocket::OnWebsocketConnectionError(const FString& Error)
{
    GAMEBASE_LOG_WARNING("Websocket is error. (%s)", *Error);

    if (SelectDomain == EDomainType::Main)
    {
        Connect(EDomainType::Sub);
    }
    else
    {
        FGamebaseErrorPtr ConnectionError = MakeShared<FGamebaseError, ESPMode::ThreadSafe>(GamebaseErrorCode::SOCKET_ERROR, Error, GamebaseWebSocket::Domain);
        GamebaseInternalReport::Network::DomainConnectionFailed(*InternalData, ConnectionError.Get());
    
        if (ConnectCallback)
        {
            ConnectCallback(*ConnectionError.Get());
            ConnectCallback = nullptr;
        }

        NotifyAllRequests(*ConnectionError.Get());
    }
}

void FGamebaseWebSocket::OnWebsocketClosed(int32 StatusCode, const FString& Reason, bool bWasClean)
{
    GAMEBASE_LOG_DEBUG("Websocket is closed.");
    NotifyDisconnectAllRequests();
}

void FGamebaseWebSocket::OnWebsocketMessage(const FString& MessageString)
{
    GAMEBASE_LOG_DEBUG("%s", *GamebaseJsonSecurity::MaskingJson(MessageString));
    
    FGamebaseWebSocketResponse Response;
    if (Response.FromJson(MessageString) == false)
    {
        GAMEBASE_LOG_ERROR("Json parsing failed");
        return;
    }

    Response.OriginData = MessageString;

    if (Response.Header.ServerPush.IsSet())
    {
        OnReceiveServerPush.Broadcast(Response);
        return;
    }
    
    if (Response.Header.TransactionId.IsEmpty())
    {
        GAMEBASE_LOG_ERROR("The transaction is null");
        return;
    }

    const auto& TransactionId = Response.Header.TransactionId;
    SendResponseCallback(TransactionId, FGamebaseWebSocketResponseResult(Response));
}

void FGamebaseWebSocket::NotifyAllRequests(const FGamebaseError& Error)
{
    if (ResponseFunctions.Num() == 0)
    {
        return;
    }

    TArray<FString> Keys;
    ResponseFunctions.GetKeys(Keys);
    
    for (const auto& Key : Keys)
    {
        SendResponseCallback(Key, FGamebaseWebSocketResponseResult(Error));
    }
    
    ResponseFunctions.Empty();
}

void FGamebaseWebSocket::NotifyDisconnectAllRequests()
{
    NotifyAllRequests(FGamebaseError(GamebaseErrorCode::SOCKET_ERROR, "Socket disconnected", GamebaseWebSocket::Domain));
}

void FGamebaseWebSocket::SendResponseCallback(const FString& TransactionId, const FGamebaseWebSocketResponseResult& Result)
{
    if (RequestTimoutHandlers.Contains(TransactionId))
    {
        GamebaseTimer::RemoveTimer(RequestTimoutHandlers[TransactionId]);
        RequestTimoutHandlers.Remove(TransactionId);
    }
    
    FResponseFunction Delegate;
    if (ResponseFunctions.RemoveAndCopyValue(TransactionId, Delegate))
    {
        GAMEBASE_LOG_DEBUG("The transaction has been deleted (ID: %s)", *TransactionId);
        Delegate(Result);
    }
    else
    {
        GAMEBASE_LOG_ERROR("The transaction not found (ID: %s)", *TransactionId);
    }
}