#include "GamebaseStandalonePurchase.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseInternalReport.h"
#include "GamebaseStandaloneSubsystem.h"
#include "GamebaseWebSocket.h"
#include "Purchase/GamebasePurchaseAdapterManager.h"
#include "Purchase/GamebasePurchaseDefines.h"
#include "Purchase/GamebasePurchaseInternalStatus.h"
#include "Purchase/GamebasePurchaseRequest.h"
#include "Server/GamebaseServerInfo.h"

FGamebaseStandalonePurchase::FGamebaseStandalonePurchase(const FGamebaseWebSocketPtr& WebSocket, const FGamebaseInternalDataPtr& InternalData)
    : FGamebaseInternalModule(WebSocket, InternalData)
    , AdapterManager(nullptr)
    , InternalStatus(MakeUnique<FGamebasePurchaseInternalStatus>())
{
}

FGamebaseStandalonePurchase::~FGamebaseStandalonePurchase()
{
    AdapterManager.Reset();
}

void FGamebaseStandalonePurchase::Initialize(
    const FGamebasePurchaseConfiguration& Configuration)
{
    if (CachedConfiguration.IsSet())
    {
        GAMEBASE_LOG_DEBUG("Not handled since the IAP SDK only receives the initial initialization settings.");
        return;
    }

    if (!Configuration.IsValid())
    {
        GAMEBASE_LOG_ERROR("Invalid configuration");
        return;
    }
    
    CachedConfiguration.Set(Configuration);
    
    AdapterManager = MakeUnique<FGamebasePurchaseAdapterManager>(InternalData);
    AdapterManager->Initialize(*CachedConfiguration, [this](const FGamebaseError* Error)
    {
        if (Error != nullptr)
        {
            InternalStatus->SetError(Error);
            GamebaseInternalReport::Purchase::InitFailed(*InternalData, *CachedConfiguration, Error->Message);
            return;
        }
        
        GAMEBASE_LOG_DEBUG("Purchase adapter initialized successfully.");
    });
}

void FGamebaseStandalonePurchase::SetDebugMode(const bool bIsDebugMode)
{
    if (AdapterManager.IsValid())
    {
        AdapterManager->SetDebugMode(bIsDebugMode);
    }
}

void FGamebaseStandalonePurchase::RequestPurchase(int64 ItemSeq, const FGamebasePurchasableReceiptDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandalonePurchase::RequestPurchase(const FString& GamebaseProductId, const FGamebasePurchasableReceiptDelegate& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        Callback.ExecuteIfBound(nullptr, Error.Get());
        return;
    }
    
    if (GamebaseProductId.IsEmpty())
    {
        Callback.ExecuteIfBound(nullptr, &MakeShared<FGamebaseError, ESPMode::ThreadSafe>(GamebaseErrorCode::INVALID_PARAMETER, TEXT("GamebaseProductId is empty"), GamebasePurchase::Domain).Get());
        return;
    }
    
    RequestGetProductWithGamebaseProductId(GamebaseProductId, FGetProductWithGamebaseProductIdDelegate::CreateLambda(
        [this, GamebaseProductId, Callback](const GamebasePurchase::FIapProductResponse* Product, const FGamebaseError* Error)
        {
            if (Error != nullptr)
            {
                Callback.ExecuteIfBound(nullptr, Error);
                return;
            }
            
            AdapterManager->RequestPurchase(*Product, [this, GamebaseProductId, Callback](const FGamebasePurchasableReceiptResult& Result)
            {
                OnPurchaseCompleted(GamebaseProductId, Result.GetOkOrDefault(TOptional<FGamebasePurchasableReceipt>()), Result.TryGetErrorValue(), Callback);
            });
        }));
}

void FGamebaseStandalonePurchase::RequestPurchase(const FString& GamebaseProductId, const FString& Payload, const FGamebasePurchasableReceiptDelegate& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        Callback.ExecuteIfBound(nullptr, Error.Get());
        return;
    }
    
    if (GamebaseProductId.IsEmpty())
    {
        Callback.ExecuteIfBound(nullptr, &MakeShared<FGamebaseError, ESPMode::ThreadSafe>(GamebaseErrorCode::INVALID_PARAMETER, TEXT("GamebaseProductId is empty"), GamebasePurchase::Domain).Get());
        return;
    }
    
    RequestGetProductWithGamebaseProductId(GamebaseProductId, FGetProductWithGamebaseProductIdDelegate::CreateLambda(
        [this, GamebaseProductId, Payload, Callback](const GamebasePurchase::FIapProductResponse* Product, const FGamebaseError* Error)
        {
            if (Error != nullptr)
            {
                Callback.ExecuteIfBound(nullptr, Error);
                return;
            }

            AdapterManager->RequestPurchase(*Product, Payload, [this, GamebaseProductId, Callback](const FGamebasePurchasableReceiptResult& Result)
            {
                OnPurchaseCompleted(GamebaseProductId, Result.GetOkOrDefault(TOptional<FGamebasePurchasableReceipt>()), Result.TryGetErrorValue(), Callback);
            });
        }));
}

void FGamebaseStandalonePurchase::RequestItemListOfNotConsumed(const FGamebasePurchasableReceiptListDelegate& Callback)
{
    RequestItemListOfNotConsumed(FGamebasePurchasableConfiguration(), Callback);
}

void FGamebaseStandalonePurchase::RequestItemListOfNotConsumed(
    const FGamebasePurchasableConfiguration& Configuration,
    const FGamebasePurchasableReceiptListDelegate& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        const TSharedPtr<TArray<FGamebasePurchasableReceipt>> ResultList = MakeShared<TArray<FGamebasePurchasableReceipt>>();
        Callback.ExecuteIfBound(ResultList.Get(), Error.Get());
        return;
    }
    
    const auto RequestFunc = [this](const FString& MarketItemId, const FGetProductsWithMarketItemIdDelegate& RequestCallback)
    {
        RequestGetProductsWithMarketItemId(MarketItemId, RequestCallback);
    };
    
    AdapterManager->RequestItemListOfNotConsumed(Configuration, RequestFunc,[&, Callback](const FGamebasePurchasableReceiptListResult& Result)
    {
        if (Result.IsError())
        {
            const TSharedPtr<TArray<FGamebasePurchasableReceipt>> ResultList = MakeShared<TArray<FGamebasePurchasableReceipt>>();
            Callback.ExecuteIfBound(ResultList.Get(), &Result.GetErrorValue());
            GamebaseInternalReport::Purchase::ConsumableListFailed(*InternalData, CachedConfiguration->IapAppkey, Result.TryGetErrorValue());
            return;
        }
        
        if (Result.GetOkValue().Num() == 0)
        {
            const TSharedPtr<TArray<FGamebasePurchasableReceipt>> ResultList = MakeShared<TArray<FGamebasePurchasableReceipt>>();
            Callback.ExecuteIfBound(ResultList.Get(), nullptr);
            return;
        }
        
        Callback.ExecuteIfBound(&Result.GetOkValue(), nullptr);

        const TArray<FGamebasePurchasableReceipt> ReceiptList = Result.GetOkValue();
        const FString ReceiptListJsonString = GamebaseJsonUtils::ConvertJsonArrayToJsonString<FGamebasePurchasableReceipt>(ReceiptList);
        GamebaseInternalReport::Purchase::ConsumableListNotEmpty(*InternalData, CachedConfiguration->IapAppkey, ReceiptListJsonString);

        for (const auto& ResultData : ReceiptList)
        {
            if (ResultData.PaymentSeq.IsEmpty() || ResultData.PurchaseToken.IsEmpty() || ResultData.Price <= 0.0f)
            {
                GamebaseInternalReport::Purchase::PurchaseInvalidReceipt(*InternalData, ResultData);
            }
        }
    });
}

void FGamebaseStandalonePurchase::RequestItemListPurchasable(const FGamebasePurchasableItemListDelegate& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        const TSharedPtr<TArray<FGamebasePurchasableItem>> ResultList = MakeShared<TArray<FGamebasePurchasableItem>>();
        Callback.ExecuteIfBound(ResultList.Get(), Error.Get());
        return;
    }

    RequestGetGamebaseProducts(FGetGamebaseProductsDelegate::CreateLambda(
    [&, Callback](const TArray<GamebasePurchase::FIapProductWithMarketItemIdResponse>& Products, const TArray<GamebasePurchase::FIapProductWithMarketItemIdResponse>&, const FGamebaseError* Error)
    {
        if (Error != nullptr)
        {
            const TSharedPtr<TArray<FGamebasePurchasableItem>> ResultList = MakeShared<TArray<FGamebasePurchasableItem>>();
            Callback.ExecuteIfBound(ResultList.Get(), Error);
            GamebaseInternalReport::Purchase::ItemListFailed(*InternalData, CachedConfiguration->IapAppkey, Error);
            return;
        }

        AdapterManager->RequestItemListPurchasable(Products, [this, Callback](const FGamebasePurchasableItemListResult& Result)
        {
            if (Result.IsError())
            {
                const TSharedPtr<TArray<FGamebasePurchasableItem>> ResultList = MakeShared<TArray<FGamebasePurchasableItem>>();
                Callback.ExecuteIfBound(ResultList.Get(), &Result.GetErrorValue());
                GamebaseInternalReport::Purchase::ItemListFailed(*InternalData, CachedConfiguration->IapAppkey, Result.TryGetErrorValue());
                return;
            }
            
            Callback.ExecuteIfBound(&Result.GetOkValue(), nullptr);
        });
    }));
}

void FGamebaseStandalonePurchase::RequestItemListAtIAPConsole(const FGamebasePurchasableItemListDelegate& Callback)
{
    if (const auto Error = CheckApiAvailability())
    {
        const TSharedPtr<TArray<FGamebasePurchasableItem>> ResultList = MakeShared<TArray<FGamebasePurchasableItem>>();
        Callback.ExecuteIfBound(ResultList.Get(), Error.Get());
        return;
    }

    RequestGetGamebaseProducts(FGetGamebaseProductsDelegate::CreateLambda(
    [&, Callback](const TArray<GamebasePurchase::FIapProductWithMarketItemIdResponse>& Products, const TArray<GamebasePurchase::FIapProductWithMarketItemIdResponse>& InvalidProducts, const FGamebaseError* Error)
    {
        if (Error != nullptr)
        {
            const TSharedPtr<TArray<FGamebasePurchasableItem>> ResultList = MakeShared<TArray<FGamebasePurchasableItem>>();
            Callback.ExecuteIfBound(ResultList.Get(), Error);
            return;
        }
        
        AdapterManager->RequestItemListAtIAPConsole(Products, InvalidProducts, [Callback](const FGamebasePurchasableItemListResult& Result)
        {
            if (Result.IsError())
            {
                const TSharedPtr<TArray<FGamebasePurchasableItem>> ResultList = MakeShared<TArray<FGamebasePurchasableItem>>();
                Callback.ExecuteIfBound(ResultList.Get(), &Result.GetErrorValue());
                return;
            }

            Callback.ExecuteIfBound(&Result.GetOkValue(), nullptr);
        });
    }));
}

void FGamebaseStandalonePurchase::RequestActivatedPurchases(const FGamebasePurchasableReceiptListDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandalonePurchase::RequestActivatedPurchases(const FGamebasePurchasableConfiguration& Configuration, const FGamebasePurchasableReceiptListDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandalonePurchase::RequestSubscriptionsStatus(const FGamebasePurchasableConfiguration& Configuration, const FGamebasePurchasableSubscriptionStatusDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandalonePurchase::SetPromotionIAPHandler(const FGamebasePurchasableReceiptDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandalonePurchase::OnUpdateAuthToken(const TOptional<FGamebaseAuthToken>& AuthToken, const TOptional<FString>& ProviderName)
{
    if (AdapterManager->IsDisable())
    {
        GAMEBASE_LOG_DEBUG("No adapter was created for purchase");
        return;
    }

    RequestIapUserInfo(AuthToken);
}

void FGamebaseStandalonePurchase::RequestGetGamebaseProducts(const FGetGamebaseProductsDelegate& Callback)
{
    using namespace GamebaseServerInfo;
    
    GamebasePurchase::FGetGamebaseProductsParameters Parameters;
    Parameters.StoreCode = InternalData->GetStoreCode();

    WebSocket->Request(IapBackend::ProductId, IapBackend::Id::GetGamebaseProducts, ApiVersion, Parameters,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                Callback.ExecuteIfBound({}, {}, &Response.GetErrorValue());
                return;
            }
            
            const auto& ResponseSuccessData = Response.GetOkValue();

            if (ResponseSuccessData.Header.bIsSuccessful)
            {
                GamebasePurchase::FGetGamebaseProductsResponse ResponseVo;
                ResponseVo.FromJson(ResponseSuccessData.OriginData);

                Callback.ExecuteIfBound(ResponseVo.Products, ResponseVo.InvalidProducts, nullptr);
            }
            else
            {
                const FGamebaseErrorPtr RecvError = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebasePurchase::Domain);
                Callback.ExecuteIfBound({}, {}, RecvError.Get());
            }
        });
}

void FGamebaseStandalonePurchase::RequestGetProductWithGamebaseProductId(const FString& GamebaseProductId, const FGetProductWithGamebaseProductIdDelegate& Callback)
{
    using namespace GamebaseServerInfo;
    
    GamebasePurchase::FGetProductWithGamebaseProductIdParameters Parameters;
    Parameters.StoreCode = InternalData->GetStoreCode();
    Parameters.GamebaseProductId = GamebaseProductId;
    
    WebSocket->Request(IapBackend::ProductId, IapBackend::Id::GetProductWithGamebaseProductId, ApiVersion, Parameters,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                Callback.ExecuteIfBound({}, &Response.GetErrorValue());
                return;
            }
            
            const auto& ResponseSuccessData = Response.GetOkValue();

            if (ResponseSuccessData.Header.bIsSuccessful)
            {
                GamebasePurchase::FGetProductWithGamebaseProductIdResponse ResponseVo;
                ResponseVo.FromJson(ResponseSuccessData.OriginData);

                Callback.ExecuteIfBound(&ResponseVo.Product, nullptr);
            }
            else
            {
                const FGamebaseErrorPtr RecvError = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebasePurchase::Domain);
                Callback.ExecuteIfBound({}, RecvError.Get());
            }
        });
}

void FGamebaseStandalonePurchase::RequestGetProductsWithMarketItemId(const FString& MarketItemId, const FGetProductsWithMarketItemIdDelegate& Callback)
{
    using namespace GamebaseServerInfo;

    GamebasePurchase::FGetProductsWithMarketItemIdParameters Parameters;
    Parameters.StoreCode = InternalData->GetStoreCode();
    Parameters.MarketItemId = MarketItemId;
    
    WebSocket->Request(IapBackend::ProductId, IapBackend::Id::GetProductsWithMarketItemId, ApiVersion, Parameters,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                Callback({}, &Response.GetErrorValue());
                return;
            }
                
            const auto& ResponseSuccessData = Response.GetOkValue();

            if (ResponseSuccessData.Header.bIsSuccessful)
            {
                GamebasePurchase::FGetProductsWithMarketItemIdResponse ResponseVo;
                ResponseVo.FromJson(ResponseSuccessData.OriginData);
                
                Callback(ResponseVo.Products, nullptr);
            }
            else
            {
                Callback({}, GamebaseErrorUtil::NewError(ResponseSuccessData, GamebasePurchase::Domain).Get());
            }
        });
}

void FGamebaseStandalonePurchase::RequestIapUserInfo(const TOptional<FGamebaseAuthToken>& AuthToken)
{
    using namespace GamebaseServerInfo;
    
    if (const auto Error = CheckApiAvailability())
    {
        return;
    }
    
    GamebasePurchase::FGetGamebaseProductsParameters Parameters;
    Parameters.StoreCode = InternalData->GetStoreCode();

    WebSocket->Request(IapBackend::ProductId, IapBackend::Id::GetGamebaseProducts, ApiVersion, Parameters,
        [this, AuthToken](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                AdapterManager->SetGamebaseUserInfo(AuthToken.IsSet() ? AuthToken->Member.UserId : TEXT(""), FString());
                return;
            }
            
            const auto& ResponseSuccessData = Response.GetOkValue();

            if (ResponseSuccessData.Header.bIsSuccessful == false)
            {
                AdapterManager->SetGamebaseUserInfo(AuthToken.IsSet() ? AuthToken->Member.UserId : TEXT(""), FString());
                const FGamebaseErrorPtr RecvError = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebasePurchase::Domain);
                return;
            }

            if (AuthToken.IsSet())
            {
                AdapterManager->SetGamebaseUserInfo(AuthToken->Member.UserId, AuthToken->Token.AccessToken);
            }

            GamebasePurchase::FGetGamebaseProductsResponse ResponseVo;
            if (ResponseVo.FromJson(ResponseSuccessData.OriginData))
            {
                AdapterManager->RequestReprocessPurchased(ResponseVo.ToJson(false), [this](const FGamebaseError* Error)
                {
                    if (Error != nullptr)
                    {
                        InternalStatus->SetError(Error);
                    }
                    else
                    {
                        GAMEBASE_LOG_DEBUG("RequestReprocessPurchased success");
                    }
                });
            }
            else
            {
                GAMEBASE_LOG_WARNING("RequestReprocessPurchased failed");
            }
        });
}


void FGamebaseStandalonePurchase::OnPurchaseCompleted(
    const FString& GamebaseProductId,
    const TOptional<FGamebasePurchasableReceipt>& ResultData,
    const FGamebaseError* Error,
    const FGamebasePurchasableReceiptDelegate& Callback)
{
    if (Error != nullptr)
    {
        Callback.ExecuteIfBound(nullptr, Error);
    }
    else
    {
        if (ResultData.IsSet())
        {
            Callback.ExecuteIfBound(&ResultData.GetValue(), nullptr);
            OnFinishPurchase.Broadcast(ResultData.GetValue().PaymentSeq);
            
            if (ResultData->PaymentSeq.IsEmpty() || ResultData->PurchaseToken.IsEmpty() || ResultData->Price <= 0.0f)
            {
                GamebaseInternalReport::Purchase::PurchaseInvalidReceipt(*InternalData, *ResultData);
            }
        }
        else
        {
            Callback.ExecuteIfBound(nullptr, nullptr);
        }
    }

    GamebaseInternalReport::Purchase::Purchase(
        *InternalData,
        *CachedConfiguration,
        GamebaseProductId,
        ResultData.IsSet() ? &ResultData.GetValue() : nullptr,
        Error
    );
}

FGamebaseErrorPtr FGamebaseStandalonePurchase::CheckApiAvailability() const
{
    if (InternalData.IsValid() == false)
    {
        return GamebaseErrorUtil::NewError(GamebasePurchase::Domain, GamebaseErrorCode::NOT_INITIALIZED);
    }
    
    if (InternalData->IsLogin() == false)
    {
        return GamebaseErrorUtil::NewError(GamebasePurchase::Domain, GamebaseErrorCode::NOT_LOGGED_IN);
    }

	if (AdapterManager.IsValid() == false)
	{
        return GamebaseErrorUtil::NewError(GamebasePurchase::Domain, GamebaseErrorCode::PURCHASE_NOT_INITIALIZED);
	}
	
    if (InternalStatus.IsValid())
    {
        if (const FGamebaseErrorPtr InternalError = InternalStatus->GetError())
        {
            return InternalError;
        }
    }
    
    return nullptr;
}