﻿#include "GamebaseStandaloneIntrospect.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseStandaloneEventHandler.h"
#include "GamebaseWebSocket.h"

#include "Auth/GamebaseAuthExtras.h"

#include "Scheduler/GamebaseIntrospectRequest.h"
#include "Server/GamebaseServerInfo.h"

namespace GamebaseIntrospect
{
    const FName Domain("GamebaseIntrospect");
    
    constexpr int32 RetryCount = 2;
}

FGamebaseStandaloneIntrospect::FGamebaseStandaloneIntrospect(const FGamebaseWebSocketPtr& WebSocket, const FGamebaseInternalDataPtr& InternalData, const TSharedPtr<FGamebaseStandaloneEventHandler>& EventHandler)
    : FGamebaseInternalModule(WebSocket, InternalData)
    , FGamebaseScheduler(0)
    , EventHandler(EventHandler)
{
}

FGamebaseStandaloneIntrospect::~FGamebaseStandaloneIntrospect()
{
}

void FGamebaseStandaloneIntrospect::SetIntervalTime(const float InIntervalTime)
{
    if (InIntervalTime <= KINDA_SMALL_NUMBER)
    {
        GAMEBASE_LOG_WARNING("Introspect interval time null");
        return;
    }
    
    IntervalTime = InIntervalTime;
}

void FGamebaseStandaloneIntrospect::OnUpdateLaunchingInfo(const FGamebaseLaunchingInfo& LaunchingInfo)
{
    if (LaunchingInfo.Launching.TcgbClient.IsSet() && LaunchingInfo.Launching.TcgbClient.GetValue().Introspection.IsSet())
    {
        const auto& Introspection = LaunchingInfo.Launching.TcgbClient.GetValue().Introspection.GetValue();
        SetIntervalTime(Introspection.IntervalSeconds);
    }
}

void FGamebaseStandaloneIntrospect::OnUpdateAuthToken(const TOptional<FGamebaseAuthToken>& AuthToken, const TOptional<FString>& ProviderName)
{
    if (!AuthToken.IsSet())
    {
        StopScheduler();
        return;
    }

    bool bShouldUseIntrospection = true;
    if (AuthToken->Token.ExtraParams.IsSet())
    {
        const auto& ExtraParams = AuthToken->Token.ExtraParams.GetValue();
        if (ExtraParams.Contains(GamebaseAuthExtras::UseIntrospection))
        {
            bShouldUseIntrospection = ExtraParams.FindRef(GamebaseAuthExtras::UseIntrospection).GetValue<bool>();
        }
    }

    if (bShouldUseIntrospection)
    {
        StartScheduler();
    }
    else
    {
        StopScheduler();
    }
}

void FGamebaseStandaloneIntrospect::OnStartSchedule()
{
    if (IsGamebaseLogin() == false)
    {
        return;
    }

    if (RefreshIntervalTime <= KINDA_SMALL_NUMBER)
    {
        return;
    }

    GAMEBASE_LOG_DEBUG("Start introspect"); 

    SentIntervalTime = RefreshIntervalTime;
}

void FGamebaseStandaloneIntrospect::OnUpdateSchedule()
{
    GamebaseIntrospect::FParameters Parameters;
    if (InternalData->IsLogin())
    {
        Parameters.UserId = InternalData->GetUserId();
        Parameters.AccessToken = InternalData->GetAccessToken();
        Parameters.IdpCode = InternalData->GetAuthToken()->Token.SourceIdpCode;
    }
    else
    {
        Parameters.UserId = TEXT("0");
    }

    WebSocket->Request(GamebaseServerInfo::Gateway::ProductId, GamebaseServerInfo::Gateway::Id::IntrospectAccessToken, GamebaseServerInfo::ApiVersion, Parameters, [=](const FGamebaseWebSocketResponseResult& Response)
    {
        auto IntrospectError = Response.TryGetErrorValue();
        
        if (Response.IsOk())
        {
            const auto& ResponseSuccessData = Response.GetOkValue();
            if (ResponseSuccessData.Header.bIsSuccessful)
            {
                GAMEBASE_LOG_DEBUG("Send introspect succeeded");
                RetryCount = 0;
                SentIntervalTime = RefreshIntervalTime;
                return;
            }
            
            IntrospectError = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseIntrospect::Domain).Get();
        }

        if (IntrospectError == nullptr)
        {
            return;
        }

        switch (IntrospectError->Code)
        {
        case GamebaseErrorCode::SOCKET_ERROR:
        case GamebaseErrorCode::SOCKET_RESPONSE_TIMEOUT:
            {
                SentIntervalTime = RetryIntervalTime;
                RetryCount++;

                if (RetryCount <= GamebaseIntrospect::RetryCount)
                {
                    return;
                }
            }
            break;
        case GamebaseErrorCode::AUTH_INVALID_GAMEBASE_TOKEN:
            {
                WebSocket->Disconnect();
            }
            break;
        default:
            break;
        }

        RetryCount = 0;
        StopScheduler();
        
        FGamebaseEventObserverData EventData;
        EventData.Extras = IntrospectError->ToJson();
        
        FGamebaseEventMessage EventMessage;
        EventMessage.Category = GamebaseEventCategory::ObserverIntrospect;
        EventMessage.Data = EventData.ToJson();
        
        EventHandler->Notify(EventMessage);
    });
}