#include "GamebaseStandaloneHeartbeat.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseInternalReport.h"
#include "GamebaseStandaloneEventHandler.h"
#include "GamebaseSystemUtils.h"
#include "GamebaseWebSocket.h"
#include "Helper/GamebaseSystemPopupHelper.h"
#include "Scheduler/GamebaseHeartbeartRequest.h"
#include "Server/GamebaseServerInfo.h"

namespace GamebaseHeartbeat
{
    const FName Domain("GamebaseHeartbeat");
    constexpr int32 Interval = 120;
}

FGamebaseStandaloneHeartbeat::FGamebaseStandaloneHeartbeat(const FGamebaseWebSocketPtr& WebSocket, const FGamebaseInternalDataPtr& InternalData, const TSharedPtr<FGamebaseStandaloneEventHandler>& EventHandler)
    : FGamebaseInternalModule(WebSocket, InternalData)
    , FGamebaseScheduler(GamebaseHeartbeat::Interval)
    , EventHandler(EventHandler)
{
}

FGamebaseStandaloneHeartbeat::~FGamebaseStandaloneHeartbeat()
{
}

void FGamebaseStandaloneHeartbeat::OnUpdateSchedule()
{
    using namespace GamebaseServerInfo;
    
    GamebaseHeartbeat::FParameters Parameters;
    Parameters.AppId = InternalData->GetAppId();
    
    GamebaseHeartbeat::FPayload Payload;
    Payload.AppId = InternalData->GetAppId();
    Payload.UserId = IsGamebaseLogin() ? InternalData->GetUserId() : TEXT("0");
    Payload.ClientVersion = InternalData->GetAppVersion();
    Payload.StoreCode = InternalData->GetStoreCode();
    Payload.OSCode = GamebaseSystemUtils::GetPlatform();
    Payload.DeviceModel = GamebaseSystemUtils::GetDeviceModel();
    Payload.DeviceCountryCode = GamebaseSystemUtils::GetCountryCode();
    Payload.UsimCountryCode = GamebaseSystemUtils::GetSimCode();
    Payload.IdpCode = IdPCode;
    //Payload.ThirdIdPCode;
    
    WebSocket->Request(Presence::ProductId, Presence::Id::Heartbeat, ApiVersion, Parameters, Payload,[=](const FGamebaseWebSocketResponseResult& Response)
    {
        auto HeartbeatError = Response.TryGetErrorValue();
        
        if (Response.IsOk())
        {
            const auto& ResponseSuccessData = Response.GetOkValue();
            if (ResponseSuccessData.Header.bIsSuccessful)
            {
                GAMEBASE_LOG_DEBUG("Send heartbeat succeeded");
                return;
            }
            
            HeartbeatError = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseHeartbeat::Domain).Get();
        }

        if (HeartbeatError == nullptr)
        {
            return;
        }
        int32 ErrorCode = HeartbeatError->Code;
        switch (ErrorCode)
        {
            case GamebaseErrorCode::BANNED_MEMBER:
            case GamebaseErrorCode::INVALID_MEMBER:
                {
                    FGamebaseEventObserverData EventData;
                    EventData.Code = ErrorCode;
                    EventData.Message = HeartbeatError->Message;
                    EventData.Extras = HeartbeatError->ToJson();
                
                    FGamebaseEventMessage EventMessage;
                    EventMessage.Category = GamebaseEventCategory::ObserverHeartbeat;
                    EventMessage.Data = EventData.ToJson();

                    if (ErrorCode == GamebaseErrorCode::BANNED_MEMBER)
                    {
                        GamebaseInternalReport::Event::ObserverBannedMember(*InternalData, EventMessage.Data);
                        
                        if (InternalData->GetConfiguration().bEnablePopup && InternalData->GetConfiguration().bEnableBanPopup)
                        {
                            GamebaseSystemPopup::ShowHeartbeatBan(
                                *InternalData->GetDisplayLanguage(),
                                [=]
                                {
                                });
                        }
                    }
                    else if (ErrorCode == GamebaseErrorCode::INVALID_MEMBER)
                    {
                        GamebaseInternalReport::Event::ObserverInvalidMember(*InternalData, EventMessage.Data);
                    }
                    
                    EventHandler->Notify(EventMessage);
                }
                break;
            
            case GamebaseErrorCode::AUTH_INVALID_GAMEBASE_TOKEN:
                {
                    FGamebaseEventLoggedOutData EventData;
                    EventData.message = HeartbeatError->Message;
                    EventData.extras = HeartbeatError->ToJson();
                
                    FGamebaseEventMessage EventMessage;
                    EventMessage.Category = GamebaseEventCategory::LoggedOut;
                    EventMessage.Data = EventData.ToJson();

                    EventHandler->Notify(EventMessage);
                
                    GamebaseInternalReport::Event::LoggedOut(*InternalData, EventMessage.Data);
                }
                break;
            default:
                break;
        }
    });
}

void FGamebaseStandaloneHeartbeat::OnUpdateAuthToken(const TOptional<FGamebaseAuthToken>& AuthToken, const TOptional<FString>& ProviderName)
{
    IdPCode = ProviderName.Get(FString());
    
    if (AuthToken.IsSet())
    {
        StartScheduler();
    }
    else
    {
        StopScheduler();
    }
}
