﻿#include "GamebaseAuthMemberLoginWithWebView.h"

#include "GamebaseAuthConstants.h"
#include "GamebaseDebugLogger.h"
#include "GamebaseInternalData.h"
#include "GamebaseStandaloneSubsystem.h"
#include "GamebaseWebViewUtils.h"
#include "WebView/GamebaseWebViewManager.h"
#include "WebView/GamebaseWebViewOpenParams.h"

namespace GamebaseAuthMemberLoginWithWebView
{
    namespace Key
    {
        const FString Session = TEXT("session");
    }
        
    namespace Scheme
    {
        const FString AuthLogin = TEXT("gamebase://toast.gamebase/auth");
        const FString WebViewClose = TEXT("gamebase://dismiss");
    }
    
    const TArray<FString> DefaultSchemeList = {
        Scheme::AuthLogin,
        Scheme::WebViewClose
    };
}

FGamebaseAuthMemberLoginWithWebView::FGamebaseAuthMemberLoginWithWebView()
    : ResponseData(nullptr)
{
}

FGamebaseAuthMemberLoginWithWebView::~FGamebaseAuthMemberLoginWithWebView()
{
    ResponseData.Reset();
}

void FGamebaseAuthMemberLoginWithWebView::Login(const FGamebaseAuthConfiguration& Configuration, const FGamebaseVariantMap& Queries, const FResponseFunction& Callback)
{
    InternalData = Configuration.InternalData;

    const FString Url = GamebaseAuth::NeMemberLogin::GetUrl(Configuration, Queries, GamebaseAuthMemberLoginWithWebView::Scheme::AuthLogin);
    GAMEBASE_LOG_DEBUG("Login webview url: %s", *Url);

    FGamebaseWebViewOpenParams OpenParams;
    OpenParams.Url = Url;

    const auto& LoginWebViewInfoOptional = Configuration.LaunchingIdpInfo.LoginWebView;
    if (LoginWebViewInfoOptional.IsSet())
    {
        const auto& LoginWebViewInfo = LoginWebViewInfoOptional.GetValue();

        FGamebaseWebViewOpenParams::FNavigation Navigation;
        Navigation.Title = LoginWebViewInfo.Title;
        Navigation.TitleColor = Navigation.IconTintColor = FColor::FromHex(LoginWebViewInfo.TitleTextColor);
        Navigation.BarColor = FColor::FromHex(LoginWebViewInfo.TitleBgColor);
        Navigation.BarHeight = 41; // TODO: 값 확인

        OpenParams.Navigation = Navigation;
    }

    OpenParams.SchemeList = GamebaseAuthMemberLoginWithWebView::DefaultSchemeList;
    OpenParams.EventCallback = [this](const FGamebaseWebviewEventResult& EventResult)
    {
        // GAMEBASE_LOG_GLOBAL_DEBUG("A Scheme event was received from CefWebView. (%s)", *SchemeUrl);
        if (OnAuthWebViewSchemeEvent(EventResult.GetOkValue(), nullptr))
        {
            CloseWebView();
        }
    };

    OpenParams.CloseCallback = [this, Callback](const FGamebaseErrorResult& Error)
    {
        if (Error.IsError())
        {
            Callback(FGamebaseAuthLoginResult(Error.GetErrorValue()));
            return;
        }

        if (ResponseData.IsValid() == false)
        {
            Callback(FGamebaseAuthLoginResult(FGamebaseError(GamebaseErrorCode::AUTH_USER_CANCELED, "User canceled.", GamebaseAuth::Domain)));
            return;
        }

        if (ResponseData == nullptr)
        {
            Callback(FGamebaseAuthLoginResult(FGamebaseError(GamebaseErrorCode::AUTH_USER_CANCELED, "User canceled.", GamebaseAuth::Domain)));
            return;
        }

        Callback(*ResponseData);
    };

    if (const UGamebaseStandaloneSubsystem* Subsystem = UGameInstance::GetSubsystem<UGamebaseStandaloneSubsystem>(InternalData->GetGameInstance().Get()))
    {
        Subsystem->GetWebViewManager()->Open(OpenParams);
    }
}

void FGamebaseAuthMemberLoginWithWebView::CancelLogin(const FCancelLoginCallback& Callback)
{
    Callback(FGamebaseErrorResult(FGamebaseError(
        GamebaseErrorCode::AUTH_LOGIN_CANCEL_FAILED,
        "Login cancel is not supported in login with WebView",
        GamebaseAuth::Domain)));
}

bool FGamebaseAuthMemberLoginWithWebView::OnAuthWebViewSchemeEvent(const FString& Scheme, const FGamebaseError* Error)
{
    const TSharedPtr<GamebaseWebViewUtils::FSchemeInfo> SchemeInfo = GamebaseWebViewUtils::GetSchemeInfo(Scheme);
    if (!SchemeInfo.IsValid())
    {
        return false;
    }
            
    if (SchemeInfo->Scheme.Equals(GamebaseAuthMemberLoginWithWebView::Scheme::AuthLogin))
    {
        FGamebaseAuthMemberResponseData Response(SchemeInfo->ParameterMap);
        
        if (Response.IsValid())
        {
            ResponseData = MakeShared<FGamebaseAuthLoginResult>(Response);
        }
        else
        {
            ResponseData = MakeShared<FGamebaseAuthLoginResult>(FGamebaseError(GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                "Member WebView Login' one-time session or access token or one-time auth code is not exist.",  GamebaseAuth::Domain));
        }
        
        return true;
    }

    if (SchemeInfo->Scheme.Equals(GamebaseAuthMemberLoginWithWebView::Scheme::WebViewClose))
    {
        const auto CloseError = MakeShared<FGamebaseError, ESPMode::ThreadSafe>();
        CloseError->Code = GamebaseErrorCode::AUTH_USER_CANCELED;
        CloseError->Domain = GamebaseAuth::Domain.ToString();

        ResponseData = MakeShared<FGamebaseAuthLoginResult>(*CloseError);
        return true;
    }

    return false;
}

void FGamebaseAuthMemberLoginWithWebView::CloseWebView()
{   
    if (const UGamebaseStandaloneSubsystem* Subsystem = UGameInstance::GetSubsystem<UGamebaseStandaloneSubsystem>(InternalData->GetGameInstance().Get()))
    {
        Subsystem->GetWebViewManager()->Close();
    }
}
