#include "GamebaseAuthMemberLoginWithExternalBrowser.h"

#include "GamebaseAuthConstants.h"
#include "GamebaseDebugLogger.h"
#include "GamebaseErrorCode.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseTimer.h"
#include "Auth/GamebaseSessionTicketClient.h"
#include "Utils/BrowserLauncher/GamebaseBrowserLauncher.h"

namespace GamebaseAuth
{
    namespace LoginWebServer
    {
        constexpr float LoginTimeout = 180.0f;
        constexpr float SessionCheckInterval = 0.5f;
    }

    //TODO: Utils로 이동
    FString ExtractDomain(const FString& Url)
    {
        FString Result = Url;

        constexpr int32 ProtocolIdx = 0;
        if (Result.Find(TEXT("://"), ESearchCase::IgnoreCase, ESearchDir::FromStart, ProtocolIdx))
        {
            Result = Result.Mid(ProtocolIdx + 3);
        }

        int32 SlashIndex;
        if (Result.FindChar(TEXT('/'), SlashIndex))
        {
            Result = Result.Left(SlashIndex);
        }

        return Result;
    }
}

struct FGamebaseAuthMemberLoginWithExternalBrowser::FMonitoringState
{
    bool bWindowClosed = false;
    bool bCallbackCalled = false;
    
    FGamebaseTickerDelegateHandle SessionTickerHandle;
};

FGamebaseAuthMemberLoginWithExternalBrowser::~FGamebaseAuthMemberLoginWithExternalBrowser()
{
}

void FGamebaseAuthMemberLoginWithExternalBrowser::Login(const FGamebaseAuthConfiguration& Configuration, const FGamebaseVariantMap& Queries, const FResponseFunction& Callback)
{
    using namespace GamebaseAuth;

    if (MonitoringState.IsValid() || CurrentLoginCallback)
    {
        Callback(FGamebaseAuthLoginResult(FGamebaseError(
            GamebaseErrorCode::AUTH_ALREADY_IN_PROGRESS_ERROR,
            TEXT("Login is already in progress."),
            GamebaseAuth::Domain)));
        return;
    }
    
    SessionTicketClient = Configuration.SessionTicketClient;
    SessionTicketId = Configuration.SessionTicketId.IsSet() ? Configuration.SessionTicketId.GetValue() : TEXT("");

    if (!SessionTicketClient.IsValid())
    {
        Callback(FGamebaseAuthLoginResult(FGamebaseError(
            GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED,
            TEXT("SessionTicketClient is not available"),
            GamebaseAuth::Domain)));
        return;
    }
    
    MonitoringState = MakeShared<FMonitoringState>();
    CurrentLoginCallback = Callback;
    MonitoringState->bCallbackCalled = false;
    
    const FString Url = NeMemberLogin::GetUrl(Configuration, Queries);
    StartBrowserMonitoring(Url, Configuration, 
        [Callback, this](const FGamebaseAuthLoginResult& Response)
        {
            if (!MonitoringState->bCallbackCalled)
            {
                MonitoringState->bCallbackCalled = true;
                Callback(Response);
            }
        });
}

void FGamebaseAuthMemberLoginWithExternalBrowser::CancelLogin(const FCancelLoginCallback& Callback)
{
    if (!MonitoringState.IsValid() && !CurrentLoginCallback)
    {
        Callback(FGamebaseErrorResult(FGamebaseError(
            GamebaseErrorCode::AUTH_LOGIN_CANCEL_FAILED,
            TEXT("No external browser login in progress"),
            GamebaseAuth::Domain)));
        return;
    }

    if (MonitoringState.IsValid() && !SessionTicketId.IsEmpty() && SessionTicketClient.IsValid())
    {
        const auto PinnedSessionTicketClient = SessionTicketClient.Pin();
        if (PinnedSessionTicketClient.IsValid())
        {
            PinnedSessionTicketClient->RequestDeleteSessionTicket(SessionTicketId, [this](const FGamebaseSessionResult& Result)
                                                                  {
                if (Result.IsError())
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("SessionTicket delete failed: %s", *Result.GetErrorValue().Message);
                }
                else
                {
                    GAMEBASE_LOG_GLOBAL_DEBUG("SessionTicket deleted successfully (ticketId: %s)", *SessionTicketId);
                } });
        }
    }
    
    if (MonitoringState.IsValid())
    {
        if (MonitoringState->SessionTickerHandle.IsValid())
        {
            GamebaseTicker::RemoveTicker(MonitoringState->SessionTickerHandle);
            MonitoringState->SessionTickerHandle.Reset();
        }
        MonitoringState->bCallbackCalled = true;
    }

    if (CurrentLoginCallback)
    {
        CurrentLoginCallback(FGamebaseAuthLoginResult(FGamebaseError(
            GamebaseErrorCode::AUTH_USER_CANCELED,
            TEXT("Login canceled by user (CancelLogin)"),
            GamebaseAuth::Domain)));
    }

    MonitoringState.Reset();

    Callback(FGamebaseErrorResult());
}

void FGamebaseAuthMemberLoginWithExternalBrowser::StartBrowserMonitoring(const FString& Url, const FGamebaseAuthConfiguration& Configuration, const FResponseFunction& Callback)
{
    using namespace GamebaseAuth;

    const FSharedThisPtr SharedThisPtr = AsShared();
    FWeakThisPtr WeakThisPtr = SharedThisPtr;

    FGamebaseBrowserLauncher::FLaunchParams Params;
    Params.Url = Url;
    Params.Timeout = LoginWebServer::LoginTimeout;
    Params.MonitoringTag = {
        ExtractDomain(Configuration.GamebaseLoginUrl),
        TEXT("Google"),
        TEXT("Facebook"),
        TEXT("X"),
        TEXT("Apple"),
        TEXT("Line"),
        TEXT("Authentication")
    };

    GAMEBASE_LOG_GLOBAL_WARNING("LoginUrl: %s", *Url);

    FGamebaseBrowserLauncher::Launch(Params,
        [WeakThisPtr](const EGamebaseBrowserLaunchResult Result, const FString& Message)
        {
            const auto PinnedThis = WeakThisPtr.Pin();
            if (!PinnedThis.IsValid())
            {
                return;
            }

            GAMEBASE_LOG_GLOBAL_DEBUG("Browser launch result: %d, Message: %s", static_cast<int32>(Result), *Message);

            switch (Result)
            {
                case EGamebaseBrowserLaunchResult::Opened:
                    GAMEBASE_LOG_GLOBAL_DEBUG("Browser window opened successfully");
                    break;

                case EGamebaseBrowserLaunchResult::Failed:
                case EGamebaseBrowserLaunchResult::Timeout:             //TODO: Close랑 같이 처리해야 하지 않나?
                    GAMEBASE_LOG_GLOBAL_DEBUG("Browser app mode failed, fallback to session monitoring: %s", *Message);
                    PinnedThis->OnBrowserMonitoringFailed();
                    break;

                case EGamebaseBrowserLaunchResult::Closed:
                    GAMEBASE_LOG_GLOBAL_DEBUG("Browser window closed");
                    PinnedThis->OnBrowserMonitoringClosed();
                    break;

                case EGamebaseBrowserLaunchResult::Cancelled:
                    // ShouldCloseCallback에 의해 종료 당함 (본 객체가 없을 것으로 추정)
                    GAMEBASE_LOG_GLOBAL_DEBUG("Browser launch cancelled, stopping session monitoring");
                    break;
                default:
                    break;
            }
        },
        [WeakThisPtr]() -> bool
        {
            const auto PinnedThis = WeakThisPtr.Pin();
            if (!PinnedThis.IsValid())
            {
                return true;
            }
            
            if (PinnedThis->MonitoringState->bCallbackCalled)
            {
                return true;
            }
            
            return false;
        });

    GAMEBASE_LOG_GLOBAL_DEBUG("Starting session monitoring alongside window monitoring");

    StartSessionMonitoring(Configuration,
        [Callback](const FGamebaseAuthLoginResult& Response)
        {
            Callback(Response);
        });
}

void FGamebaseAuthMemberLoginWithExternalBrowser::StartSessionMonitoring(
    const FGamebaseAuthConfiguration& Configuration, 
    const FResponseFunction& Callback)
{
    using namespace GamebaseAuth;
    
    if (!SessionTicketClient.IsValid() || !Configuration.SessionTicketId.IsSet() || Configuration.SessionTicketId.GetValue().IsEmpty())
    {
        Callback(FGamebaseAuthLoginResult(FGamebaseError(
            GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED,
            TEXT("SessionTicket information is not available"),
            GamebaseAuth::Domain)));
        return;
    }

    auto PinnedSessionTicketClient = SessionTicketClient.Pin();
    if (!PinnedSessionTicketClient.IsValid())
    {
        Callback(FGamebaseAuthLoginResult(FGamebaseError(
            GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED,
            TEXT("SessionTicketClient is not valid"),
            GamebaseAuth::Domain)));
        return;
    }

    const FSharedThisPtr SharedThisPtr = AsShared();
    FWeakThisPtr WeakThisPtr = SharedThisPtr;

    auto SessionCheckAttempts = MakeShared<int32>(0);
    
    GAMEBASE_LOG_GLOBAL_DEBUG("Starting session monitoring (interval: %.1f seconds)", LoginWebServer::SessionCheckInterval);

    MonitoringState->SessionTickerHandle = GamebaseTicker::AddTicker(FTickerDelegate::CreateLambda([WeakThisPtr, Configuration, Callback, PinnedSessionTicketClient, SessionCheckAttempts](float) -> bool
    {
        const auto PinnedThis = WeakThisPtr.Pin();
        if (!PinnedThis.IsValid())
        {
            return false;
        }

        (*SessionCheckAttempts)++;
        GAMEBASE_LOG_GLOBAL_DEBUG("Session check attempt #%d", *SessionCheckAttempts);
        
        PinnedSessionTicketClient->RequestExchangeSession(Configuration.SessionTicketId.GetValue(),
            [WeakThisPtr, Callback](const FGamebaseSessionResult& Result)
            {
                const auto PinnedThis = WeakThisPtr.Pin();
                if (!PinnedThis.IsValid())
                {
                    return;
                }

                if (Result.IsError())
                {
                    const auto& Error = Result.GetErrorValue();
                    if (Error.Code == GamebaseErrorCode::AUTH_ALREADY_IN_PROGRESS_ERROR)
                    {
                        //Gateway_LoginInProgress는 정상적인 진행 상태
                        return;
                    }
                    
                    if (Error.Code == GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED)
                    {
                        PinnedThis->OnSessionMonitoringFailed();
                        Callback(FGamebaseAuthLoginResult(Error));
                    }
                    else
                    {
                        // 다른 일시적 에러들은 재시도
                        GAMEBASE_LOG_GLOBAL_DEBUG("Session exchange failed (will retry): ErrorCode=%d, Message=%s", 
                            Error.Code, *Error.Message);
                    }
                    return;
                }
                
                const FGamebaseAuthMemberResponseData ResponseData(Result.GetOkValue(), TOptional<FString>(), TOptional<FString>());
                Callback(FGamebaseAuthLoginResult(ResponseData));
            });

        return true;
    }), LoginWebServer::SessionCheckInterval);
    
    GamebaseTimer::AddTimer([WeakThisPtr, Callback] {
        const auto PinnedThis = WeakThisPtr.Pin();
        if (!PinnedThis.IsValid())
        {
            return;
        }

        PinnedThis->OnSessionMonitoringTimeout();
        
        Callback(FGamebaseAuthLoginResult(FGamebaseError(
            GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED,
            TEXT("Session check timeout - login may not be completed in browser"),
            GamebaseAuth::Domain)));
    }, LoginWebServer::LoginTimeout);
}

void FGamebaseAuthMemberLoginWithExternalBrowser::OnBrowserMonitoringFailed()
{
    if (MonitoringState->SessionTickerHandle.IsValid())
    {
        GamebaseTicker::RemoveTicker(MonitoringState->SessionTickerHandle);
    }
}

void FGamebaseAuthMemberLoginWithExternalBrowser::OnBrowserMonitoringClosed()
{
    MonitoringState->bWindowClosed = true;
    
    if (MonitoringState->bCallbackCalled == false)
    {
        GAMEBASE_LOG_GLOBAL_DEBUG("Browser window closed by user, treating as login cancellation");
        OnBrowserMonitoringFailed();
        
        MonitoringState->bCallbackCalled = true;
        CurrentLoginCallback(FGamebaseAuthLoginResult(FGamebaseError(
            GamebaseErrorCode::AUTH_USER_CANCELED,
            TEXT("User closed the browser window"),
            GamebaseAuth::Domain)));
    }
}

void FGamebaseAuthMemberLoginWithExternalBrowser::OnSessionMonitoringFailed()
{
    if (MonitoringState->SessionTickerHandle.IsValid())
    {
        GamebaseTicker::RemoveTicker(MonitoringState->SessionTickerHandle);
        MonitoringState->SessionTickerHandle.Reset();       //TODO: 필요한가?
    }
}

void FGamebaseAuthMemberLoginWithExternalBrowser::OnSessionMonitoringTimeout()
{
    if (MonitoringState->SessionTickerHandle.IsValid())
    {
        GamebaseTicker::RemoveTicker(MonitoringState->SessionTickerHandle);
        MonitoringState->SessionTickerHandle.Reset();       //TODO: 필요한가?
    }
}