#include "GamebaseAuthMemberLoginSystem.h"

FString GamebaseAuth::NeMemberLogin::GetUrl(
    const FGamebaseAuthConfiguration& Configuration,
    const FGamebaseVariantMap& Queries,
    const FString& SvcUrl)
{
    FString LoginUrl = FString::Printf(TEXT("%s?socialNetworkingServiceCode=%s"),
                                       *Configuration.GamebaseLoginUrl,
                                       *Configuration.ProviderName);

    if (Configuration.GbIdClientId.IsSet())
    {
        LoginUrl += FString::Printf(TEXT("&clientId=%s"), *Configuration.GbIdClientId.GetValue());
    }
    
    if (Configuration.SessionTicketId.IsSet() && !Configuration.SessionTicketId.GetValue().IsEmpty())
    {
        LoginUrl += FString::Printf(TEXT("&state=%s"), *Configuration.SessionTicketId.GetValue());
        LoginUrl += TEXT("&callbackType=INTERNAL");
    }
    
    if (SvcUrl.IsEmpty() == false)
    {
        LoginUrl += FString::Printf(TEXT("&svcUrl=%s"), *SvcUrl);
    }
    
    auto AppendQueryToUrl = [&LoginUrl](const FName& Key, const FGamebaseVariant& Value)
    {
        switch (Value.GetType())
        {
            case EGamebaseVariantType::Empty:
                break;
            case EGamebaseVariantType::Bool:
                LoginUrl += FString::Printf(TEXT("&%s=%s"), *Key.ToString(), Value.GetValue<bool>() ? TEXT("true") : TEXT("false"));
                break;
            case EGamebaseVariantType::Int32:
                LoginUrl += FString::Printf(TEXT("&%s=%d"), *Key.ToString(), Value.GetValue<int32>());
                break;
            case EGamebaseVariantType::Int64:
                LoginUrl += FString::Printf(TEXT("&%s=%lld"), *Key.ToString(), Value.GetValue<int64>());
                break;
            case EGamebaseVariantType::UInt32:
                LoginUrl += FString::Printf(TEXT("&%s=%u"), *Key.ToString(), Value.GetValue<uint32>());
                break;
            case EGamebaseVariantType::UInt64:
                LoginUrl += FString::Printf(TEXT("&%s=%llu"), *Key.ToString(), Value.GetValue<uint64>());
                break;
            case EGamebaseVariantType::Double:
                LoginUrl += FString::Printf(TEXT("&%s=%f"), *Key.ToString(), Value.GetValue<double>());
                break;
            case EGamebaseVariantType::Float:
                LoginUrl += FString::Printf(TEXT("&%s=%f"), *Key.ToString(), Value.GetValue<float>());
                break;
            case EGamebaseVariantType::String:
                LoginUrl += FString::Printf(TEXT("&%s=%s"), *Key.ToString(), *Value.GetValue<FString>());
                break;
            case EGamebaseVariantType::Name:
                LoginUrl += FString::Printf(TEXT("&%s=%s"), *Key.ToString(), *Value.GetValue<FName>().ToString());
                break;
            case EGamebaseVariantType::Array:
            case EGamebaseVariantType::Object:
                ensureMsgf(false, TEXT("Array/Object type cannot be converted to URL parameter: %s"), *Key.ToString());
                break;
            default:
                checkNoEntry();
                break;
        }
    };

    for (const auto& Query : Queries)
    {
        AppendQueryToUrl(Query.Key, Query.Value);
    }
    
    return LoginUrl;
}

FGamebaseAuthMemberResponseData::FGamebaseAuthMemberResponseData(
    const TOptional<FString>& Session,
    const TOptional<FString>& Token,
    const TOptional<FString>& Code): Session(Session), Token(Token), Code(Code)
{
}

FGamebaseAuthMemberResponseData::FGamebaseAuthMemberResponseData(
    const TMap<FString, FString>& Queries)
{
    using namespace GamebaseAuth;

    const auto FindQueryValue = [this, Queries](const FName& Key) -> TOptional<FString>
    {
        const FString KeyString = Key.ToString();
        return Queries.Contains(KeyString) ? Queries[KeyString] : TOptional<FString>();
    };

    Session = FindQueryValue(AuthMemberResponseKey::Session);
    Token = FindQueryValue(AuthMemberResponseKey::Token);
    Code = FindQueryValue(AuthMemberResponseKey::Code);
}

bool FGamebaseAuthMemberResponseData::IsValid() const
{
    return (Session.IsSet() && !Session->IsEmpty()) || (Token.IsSet() && !Token->IsEmpty()) || (Code.IsSet() && !Code->IsEmpty());
}
