#include "GamebaseAuthAdapterManager.h"

#include "GamebaseAuthConstants.h"
#include "GamebaseDebugLogger.h"
#include "GamebaseInternalData.h"
#include "GamebaseStandaloneAuthAdapter.h"
#include "GamebaseStandaloneAuthTypes.h"
#include "GamebaseStandaloneSubsystem.h"
#include "Constants/GamebaseErrorCode.h"

FGamebaseAuthAdapterManager::FGamebaseAuthAdapterManager(const FGamebaseInternalDataPtr& InternalData)
    : InternalData(InternalData)
{
}

FGamebaseAuthAdapterManager::~FGamebaseAuthAdapterManager()
{
}

void FGamebaseAuthAdapterManager::SetSessionTicketClient(const TWeakPtr<FGamebaseSessionTicketClient>& InSessionTicketClient)
{
    SessionTicketClient = InSessionTicketClient;
}

void FGamebaseAuthAdapterManager::Initialize(const FGamebaseLaunchingIdPMap& IdPMap)
{
    UGamebaseStandaloneSubsystem* Subsystem = UGameInstance::GetSubsystem<UGamebaseStandaloneSubsystem>(InternalData->GetGameInstance().Get());
    LoadedAdapters.Initialize(Subsystem);
    
    UpdateLaunchingIdpInfo(IdPMap);
}

void FGamebaseAuthAdapterManager::Login(const FString& ProviderName, const FLoginCallback& Callback)
{
    Login(ProviderName, {}, Callback);
}

void FGamebaseAuthAdapterManager::Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FLoginCallback& Callback)
{
    UGamebaseStandaloneAuthAdapter* Adapter = LoadedAdapters.GetAdapter(ProviderName);
    if (Adapter == nullptr)
    {
        Callback(FGamebaseAuthCredentialsResult(FGamebaseError(GamebaseErrorCode::AUTH_NOT_SUPPORTED_PROVIDER, TEXT("The adapter does not exist"), GamebaseAuth::Domain)));
        return;
    }

    Adapter->Login(AdditionalInfo, [Callback](const FGamebaseAuthCredentialsResult& Result)
    {
        if (Result.IsError())
        {
            Callback(FGamebaseAuthCredentialsResult(Result.GetErrorValue()));
            return;
        }
        
        Callback(FGamebaseAuthCredentialsResult(Result.GetOkValue()));
    });
}

void FGamebaseAuthAdapterManager::CancelLogin(const FString& ProviderName, const FCancelLoginCallback& Callback)
{
    UGamebaseStandaloneAuthAdapter* Adapter = LoadedAdapters.GetAdapter(ProviderName);
    if (Adapter == nullptr)
    {
        Callback(FGamebaseErrorResult(FGamebaseError(GamebaseErrorCode::AUTH_NOT_SUPPORTED_PROVIDER, TEXT("The adapter does not exist"), GamebaseAuth::Domain)));
        return;
    }
    
    Adapter->CancelLogin([Callback](const FGamebaseErrorResult& Result)
    {
        if (Result.IsError())
        {
            Callback(FGamebaseErrorResult(Result.GetErrorValue()));
            return;
        }
        
        Callback(FGamebaseErrorResult());
    });
}

void FGamebaseAuthAdapterManager::Logout(const FString& ProviderName, const FLogoutCallback& Callback)
{
    UGamebaseStandaloneAuthAdapter* Adapter = LoadedAdapters.GetAdapter(ProviderName);
    if (Adapter == nullptr)
    {
        Callback(FGamebaseErrorResult(FGamebaseError(GamebaseErrorCode::AUTH_NOT_SUPPORTED_PROVIDER, TEXT("The adapter does not exist"), GamebaseAuth::Domain)));
        return;
    }
    
    Adapter->Logout([Callback]
    {
        Callback(FGamebaseErrorResult());
    });
}

void FGamebaseAuthAdapterManager::LogoutAll(const FLogoutCallback& Callback)
{
    LoadedAdapters.ForEach([&](UGamebaseStandaloneAuthAdapter* Adapter)
    {
        Adapter->Logout([]
        {
        });
    });
    
    Callback(FGamebaseErrorResult());
}

void FGamebaseAuthAdapterManager::Withdraw(const FString& ProviderName, const FWithdrawCallback& Callback)
{
    UGamebaseStandaloneAuthAdapter* Adapter = LoadedAdapters.GetAdapter(ProviderName);
    if (Adapter == nullptr)
    {
        Callback(FGamebaseErrorResult(FGamebaseError(GamebaseErrorCode::AUTH_NOT_SUPPORTED_PROVIDER, TEXT("The adapter does not exist"), GamebaseAuth::Domain)));
        return;
    }

    Adapter->Withdraw([Callback]
    {
        Callback(FGamebaseErrorResult());
    });
}

void FGamebaseAuthAdapterManager::WithdrawAll(const FWithdrawCallback& Callback)
{
    LoadedAdapters.ForEach([&](UGamebaseStandaloneAuthAdapter* Adapter)
    {
        Adapter->Withdraw([]
        {
        });
    });
    
    Callback(FGamebaseErrorResult());
}

FString FGamebaseAuthAdapterManager::GetUserID(const FString& ProviderName)
{
    return FString();
}

FString FGamebaseAuthAdapterManager::GetAccessToken(const FString& ProviderName)
{
    return FString();
}

FGamebaseAuthProviderProfilePtr FGamebaseAuthAdapterManager::GetProfile(const FString& ProviderName)
{
    return {};
}

bool FGamebaseAuthAdapterManager::HasAdapter(const FString& ProviderName) const
{
    return LoadedAdapters.GetAdapter(ProviderName) != nullptr;
}

void FGamebaseAuthAdapterManager::OnUpdateLaunchingInfo(const FGamebaseLaunchingInfo& LaunchingInfo)
{
    GamebaseLoginUrl = LaunchingInfo.Launching.App.LoginUrls.GamebaseUrl;
    
    if (bIsInitialize == false)
    {
        Initialize(LaunchingInfo.Launching.App.Idp);
        bIsInitialize = true;
        return;
    }
    
    UpdateLaunchingIdpInfo(LaunchingInfo.Launching.App.Idp);
}

void FGamebaseAuthAdapterManager::UpdateLaunchingIdpInfo(const FGamebaseLaunchingIdPMap& IdPInfo)
{
    const auto GetGbidClientId = [](const FGamebaseLaunchingIdPMap& IdPMap) -> TOptional<FString>
    {
        const FString AuthSystem = TEXT("gbid");
        if (IdPMap.Contains(AuthSystem))
        {
            return IdPMap[AuthSystem].ClientId;
        }

        return TOptional<FString>();
    };
    
    const TOptional<FString> GbidClientId = GetGbidClientId(IdPInfo);

    const FString Uuid = InternalData->GetUuid();
    const FString GuestAccessToken = !Uuid.IsEmpty() ? FString::Printf(TEXT("GAMEBASE%s"), *InternalData->GetUuid()) : FString();

    for (const auto& IdP : IdPInfo)
    {
        const FString ProviderName = IdP.Key;
        UGamebaseStandaloneAuthAdapter* Adapter = LoadedAdapters.GetAdapter(ProviderName);
        if (Adapter)
        {
            FGamebaseAuthConfiguration Configuration;
            Configuration.LaunchingIdpInfo = IdP.Value;
            Configuration.ProviderName = ProviderName;
            Configuration.GbIdClientId = GbidClientId;
            Configuration.GamebaseLoginUrl = GamebaseLoginUrl;
            
            if (!GuestAccessToken.IsEmpty())
            {
                Configuration.GuestAccessToken = GuestAccessToken;
            }
            
            Configuration.InternalData = InternalData;
            Configuration.SessionTicketClient = SessionTicketClient;
            
            Adapter->Initialize(Configuration);
            
            GAMEBASE_LOG_DEBUG("%s adapter initialization is complete", *ProviderName);
        }
    }
}
