#include "GamebaseStandaloneProvider.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseInternalReport.h"
#include "GamebaseStandaloneAnalytics.h"
#include "GamebaseStandaloneAuth.h"
#include "GamebaseStandaloneCommunity.h"
#include "GamebaseStandaloneContact.h"
#include "GamebaseStandaloneEventHandler.h"
#include "GamebaseStandaloneHeartbeat.h"
#include "GamebaseStandaloneGameNotice.h"
#include "GamebaseStandaloneImageNotice.h"
#include "GamebaseStandaloneIntrospect.h"
#include "GamebaseStandaloneLaunching.h"
#include "GamebaseStandaloneNetwork.h"
#include "GamebaseStandalonePurchase.h"
#include "GamebaseStandalonePush.h"
#include "GamebaseStandaloneServerPush.h"
#include "GamebaseStandaloneSystemInfo.h"
#include "GamebaseStandaloneTerms.h"
#include "GamebaseStandaloneUtil.h"
#include "GamebaseStandaloneWebView.h"
#include "GamebaseWebSocket.h"
#include "GpLoggerSubsystem.h"
#include "IGamebaseDisplayLanguage.h"
#include "Types/GamebasePurchaseConfiguration.h"
#include "Utils/GamebaseHelper.h"

namespace GamebaseCore
{
    const FName Domain("GamebaseCore");
}

void UGamebaseStandaloneProvider::Initialize(UGameInstanceSubsystem* Subsystem)
{
    Super::Initialize(Subsystem);
}

void UGamebaseStandaloneProvider::Deinitialize()
{
    Super::Deinitialize();
    
    EventHandler.Reset();
    WebSocket.Reset();
    DisplayLanguage.Reset();
    Auth.Reset();
    Analytics.Reset();
    Contact.Reset();
    Launching.Reset();
    GameNotice.Reset();
    ImageNotice.Reset();
    Purchase.Reset();
    Terms.Reset();
    ServerPush.Reset();
    Heartbeat.Reset();
    Introspect.Reset();
    Network.Reset();
    Push.Reset();
    WebView.Reset();
    Community.Reset();
    SystemInfo.Reset();
    Util.Reset();
}

void UGamebaseStandaloneProvider::SetDebugMode(const bool bDebugMode)
{
    Super::SetDebugMode(bDebugMode);

    if (Purchase.IsValid())
    {
        Purchase->SetDebugMode(bDebugMode);
    }
}

void UGamebaseStandaloneProvider::InitializeGamebase(const FGamebaseConfiguration& Configuration, const FGamebaseLaunchingInfoDelegate& Callback)
{
    Super::InitializeGamebase(Configuration, Callback);
    
    DisplayLanguage->SetLanguageCode(Configuration.DisplayLanguageCode);
    
    Launching->RequestLaunchingInfo([this, Configuration, Callback](const FGamebaseLaunchingInfoResult& Result)
    {
        GamebaseInternalReport::Init::FailedMultipleTime(*InternalData, Configuration, Result.TryGetErrorValue());
        
        if (Result.IsError())
        {
            Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
            return;
        }
        
        OnInitializeGamebaseCompleted(Result.GetOkValue());
        Callback.ExecuteIfBound(&Result.GetOkValue(), nullptr);
    });
}

void UGamebaseStandaloneProvider::Login(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    Login(ProviderName, {}, Callback);
}

void UGamebaseStandaloneProvider::Login(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    if (InternalData.IsValid() == false)
    {
        Callback.ExecuteIfBound(nullptr, GamebaseErrorUtil::NewError(GamebaseCore::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }
    
    Launching->RequestGetLaunchingStatusIfNeedUpdate([=](const FGamebaseLaunchingStatusResult& Result)
    {
        if (Result.IsError())
        {
            Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
            return;
        }
        
        if (GamebaseHelper::IsLaunchingPlayable(Result.GetOkValue()) == false)
        {
            Callback.ExecuteIfBound(nullptr, GamebaseErrorUtil::NewError(GamebaseCore::Domain, GamebaseErrorCode::AUTH_NOT_PLAYABLE).Get());
            return;
        }
        
        Auth->Login(CredentialInfo, [=](const FGamebaseAuthTokenResult& AuthTokenResult)
        {
            GamebaseInternalReport::Auth::LoginWithCredential(*InternalData, CredentialInfo, AuthTokenResult.TryGetErrorValue());

            if (AuthTokenResult.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &AuthTokenResult.GetErrorValue());
            }
            else
            {
                // Gamebase-Client/1590 : 초기화 이후 점검이 걸리고 30초 안에 로그인을 할 경우,
                //                        최대 2분동안 점검 상태인 것을 모르고 게임에 정상 진입할 수 있기 때문에,
                //                        로그인 직후에 Launching Status를 갱신한다.
                Launching->RequestLaunchingStatus( [](const FGamebaseLaunchingStatusResult& Result){});
                Callback.ExecuteIfBound(&AuthTokenResult.GetOkValue(), nullptr);
            }
        });
    });
}

void UGamebaseStandaloneProvider::Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    if (InternalData.IsValid() == false)
    {
        Callback.ExecuteIfBound(nullptr, GamebaseErrorUtil::NewError(GamebaseCore::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }
    
    Launching->RequestGetLaunchingStatusIfNeedUpdate([=](const FGamebaseLaunchingStatusResult& LaunchingResult)
    {
        if (LaunchingResult.IsError())
        {
            Callback.ExecuteIfBound(nullptr, &LaunchingResult.GetErrorValue());
            return;
        }
        
        if (GamebaseHelper::IsLaunchingPlayable(LaunchingResult.GetOkValue()) == false)
        {
            Callback.ExecuteIfBound(nullptr, GamebaseErrorUtil::NewError(GamebaseCore::Domain, GamebaseErrorCode::AUTH_NOT_PLAYABLE).Get());
            return;
        }

        Auth->Login(ProviderName, AdditionalInfo, [=](const FGamebaseAuthTokenResult& AuthTokenResult)
        {
            GamebaseInternalReport::Auth::LoginWithProvider(*InternalData, ProviderName, AuthTokenResult.TryGetErrorValue());

            if (AuthTokenResult.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &AuthTokenResult.GetErrorValue());
            }
            else
            {
                // Gamebase-Client/1590 : 초기화 이후 점검이 걸리고 30초 안에 로그인을 할 경우,
                //                        최대 2분동안 점검 상태인 것을 모르고 게임에 정상 진입할 수 있기 때문에,
                //                        로그인 직후에 Launching Status를 갱신한다.
                Launching->RequestLaunchingStatus( [](const FGamebaseLaunchingStatusResult& Result){});
                Callback.ExecuteIfBound(&AuthTokenResult.GetOkValue(), nullptr);
            }
        });
    });
}

void UGamebaseStandaloneProvider::LoginForLastLoggedInProvider(const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::LoginForLastLoggedInProvider(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::CancelLoginWithExternalBrowser(const FGamebaseErrorDelegate& Callback)
{
    Auth->CancelLoginWithExternalBrowser(Callback);
}

void UGamebaseStandaloneProvider::AddMapping(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::AddMapping(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::AddMapping(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::ChangeLogin(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::AddMappingForcibly(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::AddMappingForcibly(const FGamebaseVariantMap& CredentialInfo, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::RemoveMapping(const FString& ProviderName, const FGamebaseErrorDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::Logout(const FGamebaseErrorDelegate& Callback)
{
    if (InternalData.IsValid() == false)
    {
        Callback.ExecuteIfBound(GamebaseErrorUtil::NewError(GamebaseCore::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }

    Auth->Logout(FGamebaseErrorDelegate::CreateLambda([Callback](const FGamebaseError* Error)
    {
        Callback.ExecuteIfBound(Error);
    }));
}

void UGamebaseStandaloneProvider::Withdraw(const FGamebaseErrorDelegate& Callback)
{
    if (InternalData.IsValid() == false)
    {
        Callback.ExecuteIfBound(GamebaseErrorUtil::NewError(GamebaseCore::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }

    Auth->Withdraw(FGamebaseErrorDelegate::CreateLambda([Callback](const FGamebaseError* Error)
    {
        Callback.ExecuteIfBound(Error);
    }));
}

void UGamebaseStandaloneProvider::WithdrawImmediately(const FGamebaseErrorDelegate& Callback)
{
    Withdraw(Callback);
}

void UGamebaseStandaloneProvider::RequestTemporaryWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback)
{
    if (InternalData.IsValid() == false)
    {
        Callback.ExecuteIfBound(nullptr, GamebaseErrorUtil::NewError(GamebaseCore::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }

    Auth->RequestTemporaryWithdrawal(FGamebaseTemporaryWithdrawalDelegate::CreateLambda([Callback](const FGamebaseTemporaryWithdrawalInfo* Data, const FGamebaseError* Error)
    {
        Callback.ExecuteIfBound(Data, Error);
    }));
}

void UGamebaseStandaloneProvider::CancelTemporaryWithdrawal(const FGamebaseErrorDelegate& Callback)
{
    if (InternalData.IsValid() == false)
    {
        Callback.ExecuteIfBound(GamebaseErrorUtil::NewError(GamebaseCore::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }

    Auth->CancelTemporaryWithdrawal(FGamebaseErrorDelegate::CreateLambda([Callback](const FGamebaseError* Error)
    {
        Callback.ExecuteIfBound(Error);
    }));
}

void UGamebaseStandaloneProvider::IssueTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::QueryTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::RenewTransferAccount(const FGamebaseTransferAccountRenewConfiguration& Configuration, const FGamebaseTransferAccountDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseStandaloneProvider::TransferAccountWithIdPLogin(const FString& AccountId, const FString& AccountPassword, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

TArray<FString> UGamebaseStandaloneProvider::GetAuthMappingList()
{
    GAMEBASE_NOT_SUPPORT_API();
    return {};
}

FString UGamebaseStandaloneProvider::GetAuthProviderUserID(const FString& ProviderName)
{
    GAMEBASE_NOT_SUPPORT_API();
    return {};
}

FString UGamebaseStandaloneProvider::GetAuthProviderAccessToken(const FString& ProviderName)
{
    GAMEBASE_NOT_SUPPORT_API();
    return {};
}

FGamebaseAuthProviderProfilePtr UGamebaseStandaloneProvider::GetAuthProviderProfile(const FString& ProviderName)
{
    GAMEBASE_NOT_SUPPORT_API();
    return {};
}

FGamebaseBanInfoPtr UGamebaseStandaloneProvider::GetBanInfo()
{
    if (InternalData.IsValid() == false)
    {
        GAMEBASE_LOG_WARNING("No initialize");
        return nullptr;
    }
    
    return Auth->GetBanInfo();
}

void UGamebaseStandaloneProvider::CreateInternalModule()
{
    if (WebSocket.IsValid())
    {
        return;
    }

    UGamebaseProvider::CreateInternalModule();
    WebSocket = MakeShared<FGamebaseWebSocket>(InternalData);
    EventHandler = MakeShared<FGamebaseStandaloneEventHandler>(WebSocket, InternalData);
    Analytics = MakeShared<FGamebaseStandaloneAnalytics>(WebSocket, InternalData);
    ShortTermTicket = MakeShared<FGamebaseStandaloneShortTermTicket>(WebSocket);
    Contact = MakeShared<FGamebaseStandaloneContact>(WebSocket, InternalData, ShortTermTicket);
    Auth = MakeShared<FGamebaseStandaloneAuth>(WebSocket, InternalData, Contact);
    GameNotice = MakeShared<FGamebaseStandaloneGameNotice>(WebSocket, InternalData);
    ImageNotice = MakeShared<FGamebaseStandaloneImageNotice>(WebSocket, InternalData);
    Launching = MakeShared<FGamebaseStandaloneLaunching>(WebSocket, InternalData, EventHandler);
    Push = MakeShared<FGamebaseStandalonePush>();
    Purchase = MakeShared<FGamebaseStandalonePurchase>(WebSocket, InternalData);
    Terms = MakeShared<FGamebaseStandaloneTerms>(WebSocket, InternalData);
    ServerPush = MakeShared<FGamebaseStandaloneServerPush>(WebSocket, InternalData, EventHandler);
    WebView = MakeShared<FGamebaseStandaloneWebView>(WebSocket, InternalData, EventHandler);
    Community = MakeShared<FGamebaseStandaloneCommunity>(WebSocket, InternalData);
    Util = MakeShared<FGamebaseStandaloneUtil>();
    Network = MakeShared<FGamebaseStandaloneNetwork>();
    Heartbeat = MakeShared<FGamebaseStandaloneHeartbeat>(WebSocket, InternalData, EventHandler);
    Introspect = MakeShared<FGamebaseStandaloneIntrospect>(WebSocket, InternalData, EventHandler);
    SystemInfo = MakeShared<FGamebaseStandaloneSystemInfo>(InternalData);
}

void UGamebaseStandaloneProvider::InitializeInternalModule()
{
    UGamebaseProvider::InitializeInternalModule();
    
    WebSocket->Initialize();
    
    WebSocket->OnReceiveServerPush.AddSP(ServerPush.ToSharedRef(), &FGamebaseStandaloneServerPush::OnReceiveServerPush);

    Launching->OnUpdateLaunchingInfo.AddSP(Auth.ToSharedRef(), &FGamebaseStandaloneAuth::OnUpdateLaunchingInfo);
    Launching->OnUpdateLaunchingInfo.AddSP(Introspect.ToSharedRef(), &FGamebaseStandaloneIntrospect::OnUpdateLaunchingInfo);
    
    Auth->OnUpdateAuthToken.AddSP(Analytics.ToSharedRef(), &FGamebaseStandaloneAnalytics::OnUpdateAuthToken);
    Auth->OnUpdateAuthToken.AddSP(Heartbeat.ToSharedRef(), &FGamebaseStandaloneHeartbeat::OnUpdateAuthToken);
    Auth->OnUpdateAuthToken.AddSP(Introspect.ToSharedRef(), &FGamebaseStandaloneIntrospect::OnUpdateAuthToken);
    Auth->OnUpdateAuthToken.AddSP(Launching.ToSharedRef(), &FGamebaseStandaloneLaunching::OnUpdateAuthToken);
    Auth->OnUpdateAuthToken.AddSP(Purchase.ToSharedRef(), &FGamebaseStandalonePurchase::OnUpdateAuthToken);
    Auth->OnUpdateAuthToken.AddSP(Terms.ToSharedRef(), &FGamebaseStandaloneTerms::OnUpdateAuthToken);
    Auth->OnUpdateAuthToken.AddUObject(this, &UGamebaseStandaloneProvider::OnUpdateAuthToken);

    Purchase->OnFinishPurchase.AddSP(Analytics.ToSharedRef(), &FGamebaseStandaloneAnalytics::OnFinishPurchase);

    ServerPush->OnCallLogout.BindLambda([&]
    {
        Auth->Logout(nullptr);
    });
    ServerPush->OnStopScheduler.AddSP(Heartbeat.ToSharedRef(), &FGamebaseStandaloneHeartbeat::StopScheduler);
    ServerPush->OnStopScheduler.AddSP(Introspect.ToSharedRef(), &FGamebaseStandaloneIntrospect::StopScheduler);
}

void UGamebaseStandaloneProvider::OnInitializeGamebaseCompleted(const FGamebaseLaunchingInfo& LaunchingInfo)
{
    Super::OnInitializeGamebaseCompleted(LaunchingInfo);

    FGamebasePurchaseConfiguration PurchaseConfiguration;
    PurchaseConfiguration.IapAppkey = LaunchingInfo.TcProduct.Iap.AppKey;
    PurchaseConfiguration.StoreCode = *InternalData->GetStoreCode();
    PurchaseConfiguration.ZoneType = InternalData->GetZoneType();
    PurchaseConfiguration.bIsLaunchingSandbox = LaunchingInfo.Launching.App.TypeCode.Equals(TEXT("SANDBOX"), ESearchCase::IgnoreCase);
    PurchaseConfiguration.ServerUrl = GamebasePurchase::GetHostUrl(InternalData->GetZoneType(), PurchaseConfiguration.bIsLaunchingSandbox); //LaunchingInfo.TcProduct.Iap.Url;

    const auto* FindStoreInfo = LaunchingInfo.TcIap.FindByPredicate([&](const auto& IapInfo)
    {
        return IapInfo.StoreCode.Equals(PurchaseConfiguration.StoreCode, ESearchCase::IgnoreCase);
    });

    if (FindStoreInfo != nullptr)
    {
        PurchaseConfiguration.StoreId = FindStoreInfo->Id;
        PurchaseConfiguration.StoreAppId = FindStoreInfo->StoreAppId;
    }
    else
    {
        GAMEBASE_LOG_ERROR("Failed to find store info for StoreCode: %s", *PurchaseConfiguration.StoreCode);
    }
    
    Purchase->Initialize(PurchaseConfiguration);
}

void UGamebaseStandaloneProvider::OnUpdateAuthToken(const TOptional<FGamebaseAuthToken>& AuthToken, const TOptional<FString>& ProviderName)
{
    FString UserId;
    if (AuthToken.IsSet() == true)
    {
        UserId = AuthToken.GetValue().Member.UserId;
    }
    
    if (UGpLoggerSubsystem* LoggerSubsystem = UGameInstance::GetSubsystem<UGpLoggerSubsystem>(GetWorld()->GetGameInstance()))
    {
        LoggerSubsystem->SetGameUserId(UserId);
    }
}

#define GAMEBASE_MODULE_GETTER_DEFINITION(Module) \
    IGamebase##Module* UGamebaseStandaloneProvider::Get##Module() const \
    { \
        return Module.Get(); \
    }

GAMEBASE_MODULE_GETTER_DEFINITION(Analytics);
GAMEBASE_MODULE_GETTER_DEFINITION(Contact);
GAMEBASE_MODULE_GETTER_DEFINITION(EventHandler);
GAMEBASE_MODULE_GETTER_DEFINITION(GameNotice)
GAMEBASE_MODULE_GETTER_DEFINITION(ImageNotice);
GAMEBASE_MODULE_GETTER_DEFINITION(Launching);
GAMEBASE_MODULE_GETTER_DEFINITION(Network);
GAMEBASE_MODULE_GETTER_DEFINITION(Purchase);
GAMEBASE_MODULE_GETTER_DEFINITION(Push);
GAMEBASE_MODULE_GETTER_DEFINITION(SystemInfo);
GAMEBASE_MODULE_GETTER_DEFINITION(Terms);
GAMEBASE_MODULE_GETTER_DEFINITION(Util);
GAMEBASE_MODULE_GETTER_DEFINITION(WebView);
GAMEBASE_MODULE_GETTER_DEFINITION(Community)

FGamebaseStandaloneServerPush* UGamebaseStandaloneProvider::GetServerPush() const
{
    return ServerPush.Get();
};
