#pragma once

#include "GamebaseBaseTypes.h"
#include "GamebaseDebugLogger.h"
#include "GamebaseValueObject.h"

#include "Serialization/JsonSerializerMacros.h"

#define BEGIN_GAMEBASE_JSON_SERIALIZER                                                              BEGIN_JSON_SERIALIZER
#define END_GAMEBASE_JSON_SERIALIZER                                                                END_JSON_SERIALIZER

#define GAMEBASE_JSON_SERIALIZE(JsonName, JsonValue)                                                JSON_SERIALIZE(JsonName, JsonValue)
#define GAMEBASE_JSON_SERIALIZE_ARRAY(JsonName, JsonArray)                                          JSON_SERIALIZE_ARRAY(JsonName, JsonArray) 
#define GAMEBASE_JSON_SERIALIZE_MAP(JsonName, JsonMap)                                              JSON_SERIALIZE_MAP(JsonName, JsonMap)
#define GAMEBASE_JSON_SERIALIZE_SIMPLE_COPY(JsonMap)                                                JSON_SERIALIZE_SIMPLECOPY(JsonMap)
#define GAMEBASE_JSON_SERIALIZE_SERIALIZABLE(JsonName, JsonValue)                                   JSON_SERIALIZE_SERIALIZABLE(JsonName, JsonValue) 
#define GAMEBASE_JSON_SERIALIZE_RAW_JSON_STRING(JsonName, JsonValue)                                JSON_SERIALIZE_RAW_JSON_STRING(JsonName, JsonValue)
#define GAMEBASE_JSON_SERIALIZE_ARRAY_SERIALIZABLE(JsonName, JsonArray, ElementType)                JSON_SERIALIZE_ARRAY_SERIALIZABLE(JsonName, JsonArray, ElementType)
#define GAMEBASE_JSON_SERIALIZE_MAP_SERIALIZABLE(JsonName, JsonMap, ElementType)                    JSON_SERIALIZE_MAP_SERIALIZABLE(JsonName, JsonMap, ElementType) 
#define GAMEBASE_JSON_SERIALIZE_OBJECT_SERIALIZABLE(JsonName, JsonSerializableObject)               JSON_SERIALIZE_OBJECT_SERIALIZABLE(JsonName, JsonSerializableObject)
#define GAMEBASE_JSON_SERIALIZE_DATETIME_UNIX_TIMESTAMP(JsonName, JsonDateTime)                     JSON_SERIALIZE_DATETIME_UNIX_TIMESTAMP(JsonName, JsonDateTime)

#define GAMEBASE_JSON_SERIALIZE_OPTIONAL(JsonName, JsonValue)                                       JSON_SERIALIZE_OPTIONAL(JsonName, JsonValue)
#define GAMEBASE_JSON_SERIALIZE_OPTIONAL_ARRAY_SERIALIZABLE(JsonName, OptionalJsonArray, ElementType)   JSON_SERIALIZE_OPTIONAL_ARRAY_SERIALIZABLE(JsonName, OptionalJsonArray, ElementType)
#define GAMEBASE_JSON_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE(JsonName, JsonSerializableObject)      JSON_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE(JsonName, JsonSerializableObject)


#define GAMEBASE_JSON_SERIALIZE_OBJECT_STRING_SERIALIZABLE(JsonName, JsonValue) \
        FString JsonTextValue = JsonValue.ToJson(false); \
        Serializer.Serialize(TEXT(JsonName), JsonTextValue)
    
#define GAMEBASE_JSON_SERIALIZE_STRING(JsonName, JsonValue) \
        if (!JsonValue.IsEmpty()) \
        { \
            Serializer.Serialize(TEXT(JsonName), JsonValue); \
        }

#define GAMEBASE_JSON_SERIALIZE_VARIANT_MAP(JsonName, VariantMap) \
    if (Serializer.IsLoading()) \
    { \
        GamebaseJsonUtils::DeserializeVariantMap(JsonName, VariantMap, Serializer); \
    } \
    else \
    { \
        GamebaseJsonUtils::SerializeVariantMap(JsonName, VariantMap, Serializer); \
    }

#define GAMEBASE_JSON_SERIALIZE_OPTIONAL_VARIANT_MAP(JsonName, OptionalVariantMap) \
    if (Serializer.IsLoading()) \
    { \
        if (OptionalVariantMap.IsSet()) \
        { \
            GamebaseJsonUtils::DeserializeVariantMap(JsonName, OptionalVariantMap.GetValue(), Serializer); \
        } \
    } \
    else \
    { \
        if (OptionalVariantMap.IsSet()) \
        { \
            GamebaseJsonUtils::SerializeVariantMap(JsonName, OptionalVariantMap.GetValue(), Serializer); \
        } \
    }

using FGamebaseJsonSerializable = FJsonSerializable;
using FGamebaseJsonSerializerWriter = FJsonSerializerWriter<>;

namespace GamebaseJsonUtils
{
    template<typename T, typename = FGamebaseJsonSerializable>
    TArray<T> ConvertJsonStringToJsonArray(const FString& JsonString)
    {
        TArray<T> ResultArray;

        FString ConvertJsonString(JsonString);
        ConvertJsonString.RemoveFromStart(FString(TEXT("[")));
        ConvertJsonString.RemoveFromEnd(FString(TEXT("]")));
        
        TArray<FString> ConvertJsonItemStringArray;
        ConvertJsonString.ParseIntoArray(ConvertJsonItemStringArray, TEXT("},{"), true);
        
        for (const FString& Data : ConvertJsonItemStringArray)
        {
            FString OriginData(Data);
            if (Data.StartsWith(TEXT("{")) == false)
            {
                OriginData = TEXT("{") + OriginData;
            }
            if (Data.EndsWith(TEXT("}")) == false)
            {
                OriginData += TEXT("}");
            }
                        
            T ResponseData;
            ResponseData.FromJson(OriginData);
            
            ResultArray.Emplace(ResponseData);
        }

        return ResultArray;
    }

    template<typename T = FGamebaseValueObject>
    FString ConvertJsonArrayToJsonString(const TArray<T>& JsonArray)
    {
        TArray<TSharedPtr<FJsonValue>> JsonValues;

        for (const T& Element : JsonArray)
        {
            TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
            TSharedRef<TJsonReader<>> JsonReader = TJsonReaderFactory<>::Create(Element.ToJson());

            if (FJsonSerializer::Deserialize(JsonReader, JsonObject) && JsonObject.IsValid())
            {
                JsonValues.Add(MakeShared<FJsonValueObject>(JsonObject));
            }
        }
        
        FString JsonString;
        const TSharedRef<TJsonWriter<TCHAR, TCondensedJsonPrintPolicy<TCHAR>>> Writer = TJsonWriterFactory<TCHAR, TCondensedJsonPrintPolicy<TCHAR>>::Create(&JsonString);
        FJsonSerializer::Serialize(JsonValues, Writer);

        return JsonString;
    }
    
    FGamebaseVariant ConvertJsonToVariant(const TSharedPtr<FJsonValue>& JsonValue);

    GAMEBASECORE_API void DeserializeVariantMap(const FString& JsonName, FGamebaseVariantMap& VariantMap, FJsonSerializerBase& Serializer);
    GAMEBASECORE_API void SerializeVariantMap(const FString& JsonName, FGamebaseVariantMap& VariantMap, FJsonSerializerBase& Serializer);

}