﻿#pragma once

#include "Misc/TVariant.h"
#include "Types/GamebaseError.h"

template <typename SuccessType, typename ErrorType>
class TGamebaseResult
{
private:
    // Helper wrapper types to distinguish Success and Error even when they are the same type
    template <typename T>
    struct TSuccessWrapper
    {
        T Value;
        
        TSuccessWrapper() = default;
        TSuccessWrapper(const T& InValue) : Value(InValue) {}
        TSuccessWrapper(T&& InValue) : Value(MoveTemp(InValue)) {}
        
        template <typename... Args>
        TSuccessWrapper(Args&&... Arguments) : Value(Forward<Args>(Arguments)...) {}
        
        const T& Get() const { return Value; }
        T& Get() { return Value; }
        
        const T* operator->() const { return &Value; }
        T* operator->() { return &Value; }
        
        const T& operator*() const { return Value; }
        T& operator*() { return Value; }
    };

    template <typename T>
    struct TErrorWrapper
    {
        T Value;
        
        TErrorWrapper() = default;
        TErrorWrapper(const T& InValue) : Value(InValue) {}
        TErrorWrapper(T&& InValue) : Value(MoveTemp(InValue)) {}
        
        template <typename... Args>
        TErrorWrapper(Args&&... Arguments) : Value(Forward<Args>(Arguments)...) {}
        
        const T& Get() const { return Value; }
        T& Get() { return Value; }
        
        const T* operator->() const { return &Value; }
        T* operator->() { return &Value; }
        
        const T& operator*() const { return Value; }
        T& operator*() { return Value; }
    };

    using FSuccessWrapperType = TSuccessWrapper<SuccessType>;
    using FErrorWrapperType = TErrorWrapper<ErrorType>;
    using FStorageType = TVariant<FSuccessWrapperType, FErrorWrapperType>;

public:
    explicit TGamebaseResult(const SuccessType& OkValue)
        : Storage(TInPlaceType<FSuccessWrapperType>(), OkValue)
    {
    }

    explicit TGamebaseResult(SuccessType&& OkValue)
        : Storage(TInPlaceType<FSuccessWrapperType>(), MoveTemp(OkValue))
    {
    }

    explicit TGamebaseResult(const ErrorType& ErrValue)
        : Storage(TInPlaceType<FErrorWrapperType>(), ErrValue)
    {
    }

    explicit TGamebaseResult(ErrorType&& ErrValue)
        : Storage(TInPlaceType<FErrorWrapperType>(), MoveTemp(ErrValue))
    {
    }
    
    TGamebaseResult(const TGamebaseResult& Other) = default;
    TGamebaseResult(TGamebaseResult&& Other) = default;

    TGamebaseResult& operator=(const TGamebaseResult& Other) = default;
    TGamebaseResult& operator=(TGamebaseResult&& Other) = default;

    virtual ~TGamebaseResult() = default;

    bool IsOk() const
    {
        return Storage.template IsType<FSuccessWrapperType>();
    }

    bool IsError() const
    {
        return Storage.template IsType<FErrorWrapperType>();
    }

    const SuccessType& GetOkValue() const
    {
        checkf(IsOk(), TEXT("It is an error to call GetOkValue() on a TResult that does not hold an ok value. Please either check IsOk() or use TryGetOkValue"));
        return Storage.template Get<FSuccessWrapperType>().Get();
    }

    SuccessType& GetOkValue()
    {
        checkf(IsOk(), TEXT("It is an error to call GetOkValue() on a TResult that does not hold an ok value. Please either check IsOk() or use TryGetOkValue"));
        return Storage.template Get<FSuccessWrapperType>().Get();
    }

    const ErrorType& GetErrorValue() const
    {
        checkf(IsError(), TEXT("It is an error to call GetErrorValue() on a TResult that does not hold an error value. Please either check IsError() or use TryGetErrorValue"));
        return Storage.template Get<FErrorWrapperType>().Get();
    }

    ErrorType& GetErrorValue()
    {
        checkf(IsError(), TEXT("It is an error to call GetErrorValue() on a TResult that does not hold an error value. Please either check IsError() or use TryGetErrorValue"));
        return Storage.template Get<FErrorWrapperType>().Get();
    }

    const SuccessType* TryGetOkValue() const
    {
        return const_cast<TGamebaseResult*>(this)->TryGetOkValue();
    }

    SuccessType* TryGetOkValue()
    {
        if (auto* Wrapper = Storage.template TryGet<FSuccessWrapperType>())
        {
            return &Wrapper->Get();
        }
        return nullptr;
    }

    const ErrorType* TryGetErrorValue() const
    {
        return const_cast<TGamebaseResult*>(this)->TryGetErrorValue();
    }

    ErrorType* TryGetErrorValue()
    {
        if (auto* Wrapper = Storage.template TryGet<FErrorWrapperType>())
        {
            return &Wrapper->Get();
        }
        return nullptr;
    }
    
    const SuccessType& GetOkOrDefault(const SuccessType& DefaultValue) const
    {
        return IsOk() ? GetOkValue() : DefaultValue;
    }
    
    // Monadic operations for better composability
    template <typename Func>
    auto Map(Func&& Function) const -> TGamebaseResult<decltype(Function(GetOkValue())), ErrorType>
    {
        using ReturnType = decltype(Function(GetOkValue()));
        if (IsOk())
        {
            return TGamebaseResult<ReturnType, ErrorType>::Success(Function(GetOkValue()));
        }
        return TGamebaseResult<ReturnType, ErrorType>::Failure(GetErrorValue());
    }
    
    template <typename Func>
    auto FlatMap(Func&& Function) const -> decltype(Function(GetOkValue()))
    {
        if (IsOk())
        {
            return Function(GetOkValue());
        }
        using ReturnType = decltype(Function(GetOkValue()));
        return ReturnType::Failure(GetErrorValue());
    }
    
    template <typename Func>
    TGamebaseResult OrElse(Func&& Function) const
    {
        if (IsError())
        {
            return Function(GetErrorValue());
        }
        return *this;
    }
    
    // Pattern matching style interface
    template <typename OkFunc, typename ErrFunc>
    auto Match(OkFunc&& OnOk, ErrFunc&& OnError) const -> decltype(OnOk(GetOkValue()))
    {
        if (IsOk())
        {
            return OnOk(GetOkValue());
        }
        return OnError(GetErrorValue());
    }
    
    // Convenient operators
    explicit operator bool() const noexcept
    {
        return IsOk();
    }
    
    // Pointer-like access (only when Ok)
    const SuccessType* operator->() const
    {
        checkf(IsOk(), TEXT("Cannot use operator-> on an error result"));
        return &GetOkValue();
    }
    
    SuccessType* operator->()
    {
        checkf(IsOk(), TEXT("Cannot use operator-> on an error result"));
        return &GetOkValue();
    }
    
    const SuccessType& operator*() const
    {
        return GetOkValue();
    }
    
    SuccessType& operator*()
    {
        return GetOkValue();
    }
    
    static TGamebaseResult Success(const SuccessType&Value)
    {
        return TGamebaseResult(Value);
    }
    
    static TGamebaseResult Success(SuccessType&& Value)
    {
        return TGamebaseResult(MoveTemp(Value));
    }

    static TGamebaseResult Failure(const ErrorType& ErrorValue)
    {
        return TGamebaseResult(ErrorValue);
    }
    
    static TGamebaseResult Failure(ErrorType&& ErrorValue)
    {
        return TGamebaseResult(MoveTemp(ErrorValue));
    }
    
    // Template factory for in-place construction
    template <typename... Args>
    static TGamebaseResult MakeSuccess(Args&&... Arguments)
    {
        return TGamebaseResult(SuccessType(Forward<Args>(Arguments)...));
    }
    
    template <typename... Args>
    static TGamebaseResult MakeFailure(Args&&... Arguments)
    {
        return TGamebaseResult(ErrorType(Forward<Args>(Arguments)...));
    }

protected:
    TGamebaseResult() = default;

private:
    FStorageType Storage;
};

template <typename OpType>
class FGamebaseInternalResult final : public TGamebaseResult<OpType, FGamebaseError>
{
public:
    using TGamebaseResult<OpType, FGamebaseError>::TGamebaseResult;
    
    // Convenient factory methods for FGamebaseInternalResult
    static FGamebaseInternalResult Success(const OpType& Value)
    {
        return FGamebaseInternalResult(Value);
    }
    
    static FGamebaseInternalResult Success(OpType&& Value)
    {
        return FGamebaseInternalResult(MoveTemp(Value));
    }
    
    static FGamebaseInternalResult Failure(const FGamebaseError& ErrorValue)
    {
        return FGamebaseInternalResult(ErrorValue);
    }
    
    static FGamebaseInternalResult Failure(FGamebaseError&& ErrorValue)
    {
        return FGamebaseInternalResult(MoveTemp(ErrorValue));
    }
    
    // Template factory for in-place construction
    template <typename... Args>
    static FGamebaseInternalResult MakeSuccess(Args&&... Arguments)
    {
        return FGamebaseInternalResult(OpType(Forward<Args>(Arguments)...));
    }
    
    static FGamebaseInternalResult MakeFailure(const FString& Code, const FString& Message)
    {
        // FGamebaseError 생성자에 맞게 수정 (Domain 파라미터 추가)
        return FGamebaseInternalResult(FGamebaseError(0, Message, FName(*Code)));
    }
};



template <typename ErrorType>
class TGamebaseErrorResult
{
public:
    explicit TGamebaseErrorResult()
        : Error()
    {
    }
    
    explicit TGamebaseErrorResult(const ErrorType& ErrValue)
        : Error(ErrValue)
    {
    }
    
    explicit TGamebaseErrorResult(ErrorType&& ErrValue)
        : Error(MoveTemp(ErrValue))
    {
    }
    
    TGamebaseErrorResult(const TGamebaseErrorResult& Other) = default;
    TGamebaseErrorResult(TGamebaseErrorResult&& Other) = default;

    TGamebaseErrorResult& operator=(const TGamebaseErrorResult& Other)
    {
        if (&Other != this)
        {
            Error = Other.Error;
        }
        return *this;
    }
    
    TGamebaseErrorResult& operator=(TGamebaseErrorResult&& Other)
    {
        if (&Other != this)
        {
            Error = MoveTemp(Other.Error);
        }
        return *this;
    }

    virtual ~TGamebaseErrorResult() = default;

    bool IsOk() const
    {
        return Error.IsSet() == false;
    }
    
    bool IsError() const
    {
        return Error.IsSet();
    }
    
    const ErrorType& GetErrorValue() const
    {
        checkf(IsError(), TEXT("It is an error to call GetErrorValue() on a TResult that does not hold an error value. Please either check IsError() or use TryGetErrorValue"));
        return Error.GetValue();
    }

    ErrorType& GetErrorValue()
    {
        checkf(IsError(), TEXT("It is an error to call GetErrorValue() on a TResult that does not hold an error value. Please either check IsError() or use TryGetErrorValue"));
        return Error.GetValue();
    }

    const ErrorType* TryGetErrorValue() const
    {
        return const_cast<TGamebaseErrorResult*>(this)->TryGetErrorValue();
    }
    
    ErrorType* TryGetErrorValue()
    {
        return Error.IsSet() ? &Error.GetValue() : nullptr;
    }

private:
    TOptional<ErrorType> Error;
};

class FGamebaseErrorResult final : public TGamebaseErrorResult<FGamebaseError>
{
public:
    using TGamebaseErrorResult::TGamebaseErrorResult;
};

template <typename T>
FString ToLogString(const FGamebaseInternalResult<T>& Result)
{
    if (Result.IsOk())
    {
        return ToLogString(Result.GetOkValue());
    }

    return ToLogString(Result.GetErrorValue());
}