﻿#pragma once

#include "GamebaseInternalDataPointer.h"
#include "GamebaseTypes.h"

class IGamebaseInternalData
{
public:
    virtual ~IGamebaseInternalData() = default;
    
    virtual TWeakObjectPtr<UGameInstance> GetGameInstance() const = 0;
    virtual FString GetInstanceName() const = 0;
    
    virtual void SetConfiguration(const FGamebaseConfiguration& NewConfiguration) = 0;
    virtual void SetDisplayLanguage(const FGamebaseDisplayLanguagePtr NewDisplayLanguage) = 0;
    
    virtual FString GetAppId() const = 0;
    virtual FString GetAppVersion() const = 0;
    virtual FString GetZoneType() const = 0;
    virtual FString GetStoreCode() const = 0;
    virtual bool IsSandbox() const = 0;

    virtual const FGamebaseConfiguration& GetConfiguration() const = 0;

    virtual bool IsInitialize() const = 0;
    virtual bool IsLogin() const = 0;
    virtual FString GetUserId() const = 0;
    virtual FString GetAccessToken() const = 0;
    
    virtual FGamebaseAuthToken* GetAuthToken() const = 0;
    virtual FGamebaseLaunchingInfo* GetLaunchingInfo() const = 0;
    virtual IGamebaseDisplayLanguage* GetDisplayLanguage() const = 0;

    //TODO: GetLaunchingInfo 정리 필요
    virtual FGamebaseLaunchingInfoPtr GetLaunchingInfoPtr() const = 0;

    virtual FString GetUuid() const = 0;
    virtual FString GetDisplayLanguageCode() const = 0;

    virtual void SetLaunchingInfo(const FGamebaseLaunchingInfoPtr& NewLaunchingInfo) = 0;
    virtual void SetAuthToken(const FGamebaseAuthTokenPtr& NewAuthToken) = 0;
};

class FGamebaseInternalData final : public IGamebaseInternalData
{
public:
    explicit FGamebaseInternalData(const TWeakObjectPtr<UGameInstance>& GameInstance, const FString& InstanceName, const FGamebaseConfiguration& Configuration)
        : GameInstance(GameInstance), InstanceName(InstanceName), Configuration(Configuration)
    {
    }

    FORCEINLINE virtual void SetConfiguration(const FGamebaseConfiguration& NewConfiguration) override
    {
        Configuration = NewConfiguration;
    }
    
    virtual void SetDisplayLanguage(const FGamebaseDisplayLanguagePtr NewDisplayLanguage) override
    {
        DisplayLanguage = NewDisplayLanguage;
    }
    
    FORCEINLINE virtual FString GetAppId() const override
    {
        return Configuration.AppID;
    }

    FORCEINLINE virtual FString GetAppVersion() const override
    {
        return Configuration.AppVersion;
    }
    
    FORCEINLINE virtual FString GetZoneType() const override
    {
        return Configuration.ZoneType;
    }
    
    FORCEINLINE virtual FString GetStoreCode() const override
    {
        return Configuration.StoreCode;
    }
    
    FORCEINLINE virtual bool IsSandbox() const override
    {
        return bIsSandbox;
    }
    
    FORCEINLINE virtual const FGamebaseConfiguration& GetConfiguration() const override
    {
        return Configuration;
    }

    virtual bool IsInitialize() const override
    {
        return LaunchingInfo.IsValid();
    }

    virtual bool IsLogin() const override
    {
        return AuthToken.IsValid();
    }

    virtual FString GetUserId() const override
    {
        return AuthToken.IsValid() ? AuthToken->Member.UserId : FString();
    }
    
    virtual FString GetAccessToken() const override
    {
        return AuthToken.IsValid() ? AuthToken->Token.AccessToken : FString();
    }
    
    virtual FGamebaseLaunchingInfo* GetLaunchingInfo() const override
    {
        return LaunchingInfo.Get();
    }

    virtual FGamebaseLaunchingInfoPtr GetLaunchingInfoPtr() const override
    {
        return LaunchingInfo;
    }

    virtual IGamebaseDisplayLanguage* GetDisplayLanguage() const override
    {
        return DisplayLanguage.Get();
    }
    
    virtual FGamebaseAuthToken* GetAuthToken() const override
    {
        return AuthToken.Get();
    }

    virtual void SetLaunchingInfo(const FGamebaseLaunchingInfoPtr& NewLaunchingInfo) override;

    virtual void SetAuthToken(const FGamebaseAuthTokenPtr& NewAuthToken) override
    {
        AuthToken = NewAuthToken;
    }

    virtual FString GetUuid() const override;

    virtual FString GetDisplayLanguageCode() const override;

    virtual FString GetInstanceName() const override
    {
        return InstanceName;
    }

    virtual TWeakObjectPtr<UGameInstance> GetGameInstance() const override
    {
        return GameInstance;
    }

private:
    TWeakObjectPtr<UGameInstance> GameInstance;
    FString InstanceName;
    mutable TOptional<FString> Udid;
    
    FGamebaseConfiguration Configuration;
    bool bIsSandbox = false;
    FGamebaseLaunchingInfoPtr LaunchingInfo;
    FGamebaseAuthTokenPtr AuthToken;
    FGamebaseDisplayLanguagePtr DisplayLanguage;
};
