﻿#pragma once

#include "Templates/SubclassOf.h"
#include "UObject/GCObject.h"

class UGamebaseAdapterBase;

class GAMEBASECORE_API FGamebaseAdapterBaseCollection : public FGCObject
{
public:
    void Initialize(UObject* NewOuter);
    void Deinitialize();
    
protected:
    /** protected constructor - for use by the template only(TGamebaseAdapterCollection<TBaseType>) */
    explicit FGamebaseAdapterBaseCollection(TSubclassOf<UGamebaseAdapterBase> InBaseType);

    /** protected constructor - Use the FSubsystemCollection<TBaseType> class */
    FGamebaseAdapterBaseCollection();
    
    UGamebaseAdapterBase* GetAdapterInternal(const FString& AdapterKey) const;
    UGamebaseAdapterBase* GetAdapterInternal(TSubclassOf<UGamebaseAdapterBase> AdapterClass) const;
    const TArray<UGamebaseAdapterBase*>& GetAdapterArrayInternal(TSubclassOf<UGamebaseAdapterBase> AdapterClass) const;
    
private:
    UGamebaseAdapterBase* AddAndInitializeAdapter(UClass* AdapterClass);
    void RemoveAndDeinitializeAdapter(const UGamebaseAdapterBase* Adapter);

public:
    virtual void AddReferencedObjects(FReferenceCollector& Collector) override;
    virtual FString GetReferencerName() const override;
    virtual bool GetReferencerPropertyName(UObject* Object, FString& OutPropertyName) const override;

protected:
    TMap<TSubclassOf<UGamebaseAdapterBase>, UGamebaseAdapterBase*> AdapterMap;

private:
    TMap<FString, UGamebaseAdapterBase*> AdapterKeyMap;
    mutable TMap<TSubclassOf<UGamebaseAdapterBase>, TArray<UGamebaseAdapterBase*>> AdapterArrayMap;

    TSubclassOf<UGamebaseAdapterBase> BaseType;

    UObject* Outer;
    bool bPopulating;
};

template<typename TBaseType>
class TGamebaseAdapterCollection final : public FGamebaseAdapterBaseCollection
{
public:
    void ForEach(TFunctionRef<void(TBaseType*)> Predicate)
    {
        for (auto& Element : AdapterMap)
        {
            Predicate(Cast<TBaseType>(Element.Value));
        }
    }
    
    TBaseType* GetAdapter(const FString& AdapterKey) const
    {
        return Cast<TBaseType>(GetAdapterInternal(AdapterKey));
    }
    
    template <typename TAdapterClass>
    TAdapterClass* GetAdapter(TSubclassOf<TAdapterClass> AdapterClass) const
    {
        static_assert(TIsDerivedFrom<TAdapterClass, TBaseType>::IsDerived, "TAdapterClass must be derived from TBaseType");
        return Cast<TAdapterClass>(GetAdapterInternal(AdapterClass));
    }

    template <typename TAdapterClass>
    const TArray<TAdapterClass*>& GetAdapterArray(TSubclassOf<TAdapterClass> AdapterClass) const
    {
        // Force a compile time check that TAdapterClass derives from TBaseType, the internal code only enforces it's a UGamebaseAdapter
        TSubclassOf<TBaseType> AdapterBaseClass = AdapterClass;

        const TArray<UGamebaseAdapterBase*>& Array = GetAdapterArrayInternal(AdapterBaseClass);
        const TArray<TAdapterClass*>* SpecificArray = reinterpret_cast<const TArray<TAdapterClass*>*>(&Array);
        return *SpecificArray;
    }

    /** Construct a FSubsystemCollection, pass in the owning object almost certainly (this). */
    TGamebaseAdapterCollection()
        : FGamebaseAdapterBaseCollection(TBaseType::StaticClass())
    {
    }
};