#include "GamebaseWindowsMics.h"

#if PLATFORM_WINDOWS
#include "GamebaseDebugLogger.h"

#include "Windows/AllowWindowsPlatformTypes.h"
#include "Windows/AllowWindowsPlatformAtomics.h"
    #include <Wbemidl.h>
    #include <comdef.h>
    #include <Wininet.h>
    #pragma comment(lib, "wbemuuid.lib")
    #pragma comment(lib, "wininet.lib")
#include "Windows/HideWindowsPlatformAtomics.h"
#include "Windows/HideWindowsPlatformTypes.h"

namespace GamebaseWindowsUtils
{
    FGamebaseMicsResult GetWmiData(const TCHAR* Query)
    {
        HRESULT hres;
        bool bDidInitializeCOM = false;
        FString OutData;

        hres = CoInitializeEx(nullptr, COINIT_MULTITHREADED);
        if (hres == S_OK)
        {
            bDidInitializeCOM = true;
        }
        else if (hres == S_FALSE || hres == RPC_E_CHANGED_MODE)
        {
            GAMEBASE_LOG_GLOBAL_DEBUG("COM already initialized or mode changed! Proceeding without reinitialization.");
        }
        else
        {
            return FGamebaseMicsResult::Failure({ FString::Printf(TEXT("CoInitializeEx failed: 0x%08X"), hres) });
        }

        IWbemLocator* pLocator = nullptr;
        hres = CoCreateInstance(CLSID_WbemLocator, nullptr, CLSCTX_INPROC_SERVER, IID_IWbemLocator, (void**)&pLocator);
        if (FAILED(hres))
        {
            if (bDidInitializeCOM)
            {
                CoUninitialize();
            }
            return FGamebaseMicsResult::Failure({ FString::Printf(TEXT("CoCreateInstance failed: 0x%08X"), hres) });
        }

        IWbemServices* pServices = nullptr;
        hres = pLocator->ConnectServer(BSTR(L"ROOT\\CIMV2"), nullptr, nullptr, nullptr, 0, nullptr, nullptr, &pServices);
        pLocator->Release();
        if (FAILED(hres))
        {
            if (bDidInitializeCOM)
            {
                CoUninitialize();
            }
            return FGamebaseMicsResult::Failure({ FString::Printf(TEXT("WMI Connection failed: 0x%08X"), hres) });
        }

        hres = CoSetProxyBlanket(pServices, RPC_C_AUTHN_WINNT, RPC_C_AUTHZ_NONE, nullptr, RPC_C_AUTHN_LEVEL_CALL,
                                 RPC_C_IMP_LEVEL_IMPERSONATE, nullptr, EOAC_NONE);
        if (FAILED(hres))
        {
            pServices->Release();
            if (bDidInitializeCOM)
            {
                CoUninitialize();
            }
            return FGamebaseMicsResult::Failure({ FString::Printf(TEXT("CoSetProxyBlanket failed: 0x%08X"), hres) });
        }

        IEnumWbemClassObject* pEnumerator = nullptr;
        hres = pServices->ExecQuery(BSTR(L"WQL"), BSTR(Query), WBEM_FLAG_FORWARD_ONLY | WBEM_FLAG_RETURN_IMMEDIATELY,
                                    nullptr, &pEnumerator);
        pServices->Release();
        if (FAILED(hres))
        {
            if (bDidInitializeCOM)
            {
                CoUninitialize();
            }
            return FGamebaseMicsResult::Failure({ FString::Printf(TEXT("ExecQuery failed: 0x%08X"), hres) });
        }

        IWbemClassObject* pClassObject = nullptr;
        ULONG uReturn = 0;
        while (pEnumerator)
        {
            hres = pEnumerator->Next(WBEM_INFINITE, 1, &pClassObject, &uReturn);
            if (FAILED(hres) || uReturn == 0)
            {
                break;
            }

            VARIANT vtProp;
            hres = pClassObject->Get(L"SerialNumber", 0, &vtProp, nullptr, nullptr);
            if (SUCCEEDED(hres) && vtProp.vt == VT_BSTR)
            {
                OutData = FString(vtProp.bstrVal);
            }
            VariantClear(&vtProp);
            pClassObject->Release();
        }
        pEnumerator->Release();

        if (bDidInitializeCOM)
        {
            CoUninitialize();
        }

        if (OutData.IsEmpty())
        {
            return FGamebaseMicsResult::Failure({ TEXT("No data retrieved from WMI query.") });
        }

        return FGamebaseMicsResult::Success(OutData);
    }
}

FGamebaseMicsResult FGamebaseWindowsMics::GetDeviceId()
{
    using namespace GamebaseWindowsUtils;

    FString ConcatStr = "";
    
    FGamebaseMicsResult WmiResult = GetWmiData(L"SELECT * FROM Win32_BaseBoard");
    if (WmiResult.IsError())
    {
        return FGamebaseMicsResult::Failure({ FString::Printf(TEXT("BaseBoard SerialNumber is missing (message: %s)"), *WmiResult.GetErrorValue().Message) });
    }
    ConcatStr += WmiResult.GetOkValue();

    WmiResult = GetWmiData(L"SELECT * FROM Win32_BIOS");
    if (WmiResult.IsError())
    {
        return FGamebaseMicsResult::Failure({ FString::Printf(TEXT("BIOS SerialNumber is missing (message: %s)"), *WmiResult.GetErrorValue().Message) });
    }
    ConcatStr += WmiResult.GetOkValue();
    
    WmiResult = GetWmiData(L"SELECT * FROM Win32_OperatingSystem");
    if (WmiResult.IsError())
    {
        return FGamebaseMicsResult::Failure({ FString::Printf(TEXT("OS SerialNumber is missing (message: %s)"), *WmiResult.GetErrorValue().Message) });
    }
    ConcatStr += WmiResult.GetOkValue();

    FSHA1 Sha1;
    Sha1.Update(reinterpret_cast<const uint8*>(TCHAR_TO_UTF8(*ConcatStr)), ConcatStr.Len());
    Sha1.Final();

    uint8 Hash[FSHA1::DigestSize];
    Sha1.GetHash(Hash);

    return FGamebaseMicsResult::Success(BytesToHex(Hash, FSHA1::DigestSize).ToLower());
}

FGamebaseMicsResult FGamebaseWindowsMics::GetLocale()
{
    const GEOID GeoId = GetUserGeoID(GEOCLASS_NATION);
    if (GeoId != GEOID_NOT_AVAILABLE)
    {
        WCHAR CountryCode[3] = {};
        if (GetGeoInfoW(GeoId, GEO_ISO2, CountryCode, 3, 0))
        {
            return FGamebaseMicsResult::Success(CountryCode);
        }
    }
    
    return FGamebaseMicsResult::Failure({ TEXT("Failed to retrieve GeoId or CountryCode.") });
}

bool FGamebaseWindowsMics::IsNetworkConnected()
{
    DWORD Flags;
    return InternetGetConnectedState(&Flags, 0);
}

#endif
