#include "GamebaseJsonSerializerMacros.h"

FGamebaseVariant GamebaseJsonUtils::ConvertJsonToVariant(const TSharedPtr<FJsonValue>& JsonValue)
{
    switch (JsonValue->Type)
    {
        case EJson::String:     return FGamebaseVariant(JsonValue->AsString());
        case EJson::Number:     return FGamebaseVariant(JsonValue->AsNumber());
        case EJson::Boolean:    return FGamebaseVariant(JsonValue->AsBool());
        case EJson::Array:
        {
            FGamebaseVariantArray ArrayVariants;
            for (const auto& Elem : JsonValue->AsArray())
            {
                ArrayVariants.Add(ConvertJsonToVariant(Elem));
            }
            return FGamebaseVariant(ArrayVariants);
        }
        case EJson::Object:
        {
            const TSharedPtr<FJsonObject> JsonObj = JsonValue->AsObject();
            FGamebaseVariantMap MapVariant;
            for (const auto& Pair : JsonObj->Values)
            {
                MapVariant.Add(FName(*Pair.Key), ConvertJsonToVariant(Pair.Value));
            }
            return FGamebaseVariant(MapVariant);
        }
        case EJson::Null:
        case EJson::None:
        default:
            return FGamebaseVariant();
    }
}

void GamebaseJsonUtils::DeserializeVariantMap(
    const FString& JsonName,
    FGamebaseVariantMap& VariantMap,
    FJsonSerializerBase& Serializer)
{
    if (Serializer.GetObject()->HasTypedField<EJson::Object>(JsonName))
    {
        const TSharedPtr<FJsonObject> JsonObj = Serializer.GetObject()->GetObjectField(JsonName);
        VariantMap = FGamebaseVariantMap();
        for (const auto& Pair : JsonObj->Values)
        {
            FName Key(*Pair.Key);
            FGamebaseVariant Value = ConvertJsonToVariant(Pair.Value);
            VariantMap.Add(Key, Value);
        }
    }
}

void GamebaseJsonUtils::SerializeVariantMap(
    const FString& JsonName,
    FGamebaseVariantMap& VariantMap,
    FJsonSerializerBase& Serializer)
{
    Serializer.StartObject(JsonName);
        
    for (const auto& Pair : VariantMap.GetMap())
    {
        const FString Key = Pair.Key.ToString();
        const FGamebaseVariant& Value = Pair.Value;
        switch (Value.GetType())
        {
            case EGamebaseVariantType::Int32:
                {
                    int32 TempValue = Value.GetValue<int32>();
                    Serializer.Serialize(*Key, TempValue);
                    break;
                }
            case EGamebaseVariantType::Float:
                {
                    float TempValue = Value.GetValue<float>();
                    Serializer.Serialize(*Key, TempValue);
                    break;
                }
            case EGamebaseVariantType::Bool:
                {
                    bool TempValue = Value.GetValue<bool>();
                    Serializer.Serialize(*Key, TempValue);
                    break;
                }
            case EGamebaseVariantType::String:
                {
                    FString TempValue = Value.GetValue<FString>();
                    Serializer.Serialize(*Key, TempValue);
                    break;
                }
            case EGamebaseVariantType::Double:
                {
                    double TempValue = Value.GetValue<double>();
                    Serializer.Serialize(*Key, TempValue);
                    break;
                }
            case EGamebaseVariantType::Int64:
                {
                    int64 TempValue = Value.GetValue<int64>();
                    Serializer.Serialize(*Key, TempValue);
                    break;
                }
            case EGamebaseVariantType::UInt64:
                {
                    //TODO: 엔진에서 uint64를 지원하지 않아서 int64로 변환하여 저장
                    int64 TempValue = static_cast<int64>(Value.GetValue<uint64>());
                    Serializer.Serialize(*Key, TempValue);
                    break;
                }
            case EGamebaseVariantType::Name:
                {
                    FString TempValue = Value.GetValue<FName>().ToString();
                    Serializer.Serialize(*Key, TempValue);
                    break;
                }
            default:
                GAMEBASE_LOG_GLOBAL_WARNING("Unsupported FGamebaseVariant type for key: %s", *Key); break;
        }
    }
    Serializer.EndObject();
}
