#include "GamebaseInternalReport.h"

#include "GamebaseInternalData.h"
#include "GamebaseProvider.h"
#include "GamebaseSystemUtils.h"
#include "GpLoggerInstance.h"
#include "Report/GamebaseInternalReportDefines.h"
#include "Types/GamebaseConfiguration.h"
#include "Types/GamebasePurchaseConfiguration.h"

using namespace GamebaseInternal::InternalReport;

namespace GamebaseInternalReport
{
    struct FReportData
    {
        FString LogType;
        FString StabilityCode;
        FString LogLevel;
        TOptional<TMap<FString, FString>> CustomFields;
        bool bIsUserCanceled = false;
        bool bIsExternalLibraryError = false;
        const FGamebaseError* Error = nullptr;
    };

    void Report(const IGamebaseInternalData& InternalData, const FReportData& Data)
    {
        TMap<FString, FString> UserFields;
        
        UserFields.Emplace(StabilityKey::StabilityCode, Data.StabilityCode);

        if (Data.Error != nullptr)
        {
            auto Error = *Data.Error;
            
            FString ErrorJsonString = EscapeJsonString(Error.ToJson()).TrimQuotes();
            
            UserFields.Emplace(StabilityKey::Exception, ErrorJsonString);
            UserFields.Emplace(StabilityKey::ErrorCode, FString::FromInt(Error.Code));
            UserFields.Emplace(StabilityKey::ErrorDomain, Error.Domain);

            if (Error.TransactionId.IsSet())
            {
                UserFields.Emplace(StabilityKey::ExceptionID, Error.TransactionId.GetValue().Replace(TEXT("-"), TEXT("")));
            }
            
            if (Error.Error.IsValid())
            {
                const auto InternalError = Error.Error.Get();
                UserFields.Emplace(StabilityKey::DetailErrorCode, FString::FromInt(InternalError->Code));
                UserFields.Emplace(StabilityKey::DetailErrorMessage, InternalError->Message);
            }
        }
        
        UserFields.Emplace(StabilityKey::GameEngine,        TEXT("UNREAL"));
        UserFields.Emplace(StabilityKey::GameName,          FApp::GetProjectName());
        UserFields.Emplace(StabilityKey::Platform,          GamebaseSystemUtils::GetPlatform());
        UserFields.Emplace(StabilityKey::ProjectAppId,      InternalData.GetAppId());
        UserFields.Emplace(StabilityKey::AppClientVersion,  InternalData.GetAppVersion());
        UserFields.Emplace(StabilityKey::LaunchingZone,     InternalData.GetZoneType());
        UserFields.Emplace(StabilityKey::UnrealSDKVersion,  GamebaseSystemUtils::GetPluginVersion());
        UserFields.Emplace(StabilityKey::ServerApiVersion,  TEXT("v1.3.4"));
        UserFields.Emplace(StabilityKey::GuestUuid,         InternalData.GetUuid());
        UserFields.Emplace(StabilityKey::DeviceLanguageCode,GamebaseSystemUtils::GetDeviceLanguage());;
        UserFields.Emplace(StabilityKey::DisplayLanguageCode, InternalData.GetDisplayLanguageCode());
        UserFields.Emplace(StabilityKey::CountryCodeUSIM,   GamebaseSystemUtils::GetSimCode());
        UserFields.Emplace(StabilityKey::CountryCodeDevice, GamebaseSystemUtils::GetCountryCode());
        UserFields.Emplace(StabilityKey::Carrier,           TEXT("NONE"));
        UserFields.Emplace(StabilityKey::DeviceModel,       GamebaseSystemUtils::GetDeviceModel());
        UserFields.Emplace(StabilityKey::DeviceKey,         GamebaseSystemUtils::GetDeviceKey());
        
        if (Data.bIsUserCanceled)
        {
            UserFields.Emplace(StabilityKey::IsUserCanceled, TEXT("Y"));
        }

        if (Data.bIsExternalLibraryError)
        {
            UserFields.Emplace(StabilityKey::IsExternalLibraryError, TEXT("Y"));
        }

        if (Data.CustomFields.IsSet())
        {
            for (const auto& Fields : Data.CustomFields.GetValue())
            {
                UserFields.Emplace(Fields.Key, Fields.Value);
            }
        }

        if (const auto Provider = UGamebaseProvider::GetInstance(InternalData.GetInstanceName()))
        {
            if (const auto Reporter = Provider->GetInternalReporter())
            {
                if (Data.LogLevel == LogLevel::Info)
                {
                    Reporter->Info(Data.StabilityCode, UserFields);
                }
                else if (Data.LogLevel == LogLevel::Warn)
                {
                    Reporter->Warn(Data.StabilityCode, UserFields);
                }
                else if (Data.LogLevel == LogLevel::Error)
                {
                    Reporter->Error(Data.StabilityCode, UserFields);
                }
                else if (Data.LogLevel == LogLevel::Fatal)
                {
                    Reporter->Fatal(Data.StabilityCode, UserFields);
                }
                else
                {
                    Reporter->Debug(Data.StabilityCode, UserFields);
                }
            }
        }
    }
}

void GamebaseInternalReport::Network::ChangeDomainSuccess(const IGamebaseInternalData& InternalData, const FString& Domain)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Network;
    ReportData.StabilityCode = StabilityCode::Network::ChangeDomainSuccess;
    ReportData.LogLevel = LogLevel::Info;
    ReportData.CustomFields = {
        { AdditionalKey::Domain, Domain }
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Network::DomainConnectionFailed(const IGamebaseInternalData& InternalData, const FGamebaseError* Error)
{
    if (Error == nullptr)
    {
        return;
    }
    
    FReportData ReportData;
    ReportData.LogType = LogType::Network;
    ReportData.StabilityCode = StabilityCode::Network::DomainConnectionFailed;
    ReportData.LogLevel = LogLevel::Error;
    ReportData.Error = Error;
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Init::EngineInformation(const IGamebaseInternalData& InternalData)
{
    using namespace GamebaseInternal::GameInformationReport;
    
    const FString EngineVersion = FEngineVersion::Current().ToString();
    
    TMap<FString, FString> UserFields;
    
    UserFields.Emplace(Key::BasicDataKeyAppId,      InternalData.GetAppId());
    UserFields.Emplace(Key::BasicDataKeyAppVersion, InternalData.GetAppVersion());
    UserFields.Emplace(Key::BasicDataKeyUuid,       InternalData.GetUuid());
    
    UserFields.Emplace(Key::EngineType,              TEXT("Unreal"));
    UserFields.Emplace(Key::EngineVersion,           EngineVersion);
    UserFields.Emplace(Key::EngineSdkVersion,        GamebaseSystemUtils::GetPluginVersion());
    UserFields.Emplace(Key::CountryCode,             GamebaseSystemUtils::GetCountryCode());
    UserFields.Emplace(Key::Platform,                GamebaseSystemUtils::GetPlatform());
    UserFields.Emplace(Key::PlatformVersion,         GamebaseSystemUtils::GetOSVersion());
    UserFields.Emplace(Key::PlatformSdkVersion,      GamebaseSystemUtils::GetPluginVersion());
    UserFields.Emplace(Key::LaunchingZone,           InternalData.GetZoneType());

    if (const auto Provider = UGamebaseProvider::GetInstance(InternalData.GetInstanceName()))
    {
        if (const auto Reporter = Provider->GetStatsReporter())
        {
            Reporter->Info(Body::GameInformation, UserFields);
        }
    }
}

void GamebaseInternalReport::Init::FailedMultipleTime(const IGamebaseInternalData& InternalData, const FGamebaseConfiguration& Configuration, const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Init;
    ReportData.StabilityCode = StabilityCode::Init::FailedMultipleTimes;
    ReportData.LogLevel = LogLevel::Warn;
    ReportData.CustomFields = {
        { AdditionalKey::Configuration, EscapeJsonString(Configuration.ToJson()).TrimQuotes() }
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Auth::LastProviderLogin(const IGamebaseInternalData& InternalData, const FString& Provider, const FGamebaseError* Error)
{
    //AUTH_LAST_PROVIDER_LOGIN_SUCCESS
    //AUTH_LAST_PROVIDER_LOGIN_FAILED
    //AUTH_LAST_PROVIDER_LOGIN_CANCELED
}

void GamebaseInternalReport::Auth::LoginWithProvider(const IGamebaseInternalData& InternalData, const FString& Provider, const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Auth;
    ReportData.CustomFields = {
        { AdditionalKey::SubCategory1, SubCategory::Login },
        { AdditionalKey::LoginIdp, Provider }
    };
    ReportData.Error = Error;
    
    if (Error == nullptr)
    {
        ReportData.StabilityCode = StabilityCode::Auth::LoginSuccess;
        ReportData.LogLevel = LogLevel::Info;
    }
    else
    {
        if (Error->Code == GamebaseErrorCode::AUTH_USER_CANCELED)
        {
            ReportData.StabilityCode = StabilityCode::Auth::LoginCanceled;
            ReportData.LogLevel = LogLevel::Info;
        }
        else
        {
            ReportData.StabilityCode = StabilityCode::Auth::LoginFailed;
            ReportData.LogLevel = LogLevel::Error;
        }
    }
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Auth::LoginWithCredential(const IGamebaseInternalData& InternalData, const FGamebaseVariantMap& CredentialInfo, const FGamebaseError* Error)
{
    if (!CredentialInfo.Contains(GamebaseAuthProviderCredential::ProviderName))
    {
        return;
    }
    
    const FString ProviderName = CredentialInfo.FindRef(GamebaseAuthProviderCredential::ProviderName);
    
    FReportData ReportData;
    ReportData.LogType = LogType::Auth;
    ReportData.CustomFields = {
        { AdditionalKey::SubCategory1, SubCategory::Login },
        { AdditionalKey::LoginIdp, ProviderName },
        //{ AdditionalKey::Credential, CredentialInfo.EncodeJson() }
    };
    ReportData.Error = Error;
    
    if (Error == nullptr)
    {
        ReportData.StabilityCode = StabilityCode::Auth::CredentialLoginSuccess;
        ReportData.LogLevel = LogLevel::Info;
    }
    else
    {
        if (Error->Code == GamebaseErrorCode::AUTH_USER_CANCELED)
        {
            ReportData.StabilityCode = StabilityCode::Auth::LoginCanceled;
            ReportData.LogLevel = LogLevel::Info;
        }
        else
        {
            ReportData.StabilityCode = StabilityCode::Auth::CredentialLoginFailed;
            ReportData.LogLevel = LogLevel::Error;
        }
    }
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Auth::ChangeLogin(const IGamebaseInternalData& InternalData, const FGamebaseVariantMap& Credential, const FGamebaseForcingMappingTicket* ForcingMappingTicket, const FGamebaseError* Error)
{
    //AUTH_CHANGE_LOGIN_SUCCESS
    //AUTH_CHANGE_LOGIN_FAILED
}

void GamebaseInternalReport::Auth::Logout(const IGamebaseInternalData& InternalData, const FGamebaseError* Error, const FGamebaseVariantMap& AdditionalInfo)
{
    // if (AdditionalInfo)
    // {
    //     //AUTH_INTERNAL_LOGOUT_SUCCESS
    //     //AUTH_INTERNAL_LOGOUT_FAILED
    //     //TODO: 내부 로그아웃 처리 추가 시 추가
    // }
    // else
    {
        FReportData ReportData;
        ReportData.LogType = LogType::Auth;
        ReportData.CustomFields = {
            { AdditionalKey::SubCategory1, SubCategory::Logout },
        };
        ReportData.Error = Error;

        if (Error == nullptr)
        {
            ReportData.StabilityCode = StabilityCode::Auth::LogoutSuccess;
            ReportData.LogLevel = LogLevel::Info;
        }
        else
        {
            ReportData.StabilityCode = StabilityCode::Auth::LogoutFailed;
            ReportData.LogLevel = LogLevel::Error;
        }
        
        Report(InternalData, ReportData);
    }
}

void GamebaseInternalReport::Auth::Withdraw(const IGamebaseInternalData& InternalData, const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Auth;
    ReportData.CustomFields = {
        { AdditionalKey::SubCategory1, SubCategory::Withdraw },
    };
    ReportData.Error = Error;

    if (Error == nullptr)
    {
        ReportData.StabilityCode = StabilityCode::Auth::WithdrawSuccess;
        ReportData.LogLevel = LogLevel::Info;
    }
    else
    {
        ReportData.StabilityCode = StabilityCode::Auth::WithdrawFailed;
        ReportData.LogLevel = LogLevel::Error;
    }
        
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Auth::RequestTemporaryWithdrawal(const IGamebaseInternalData& InternalData, const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Auth;
    ReportData.CustomFields = {
        { AdditionalKey::SubCategory1, SubCategory::TemporaryWithdrawal },
    };

    if (Error == nullptr)
    {
        ReportData.StabilityCode = StabilityCode::Auth::RequestTemporaryWithdrawalSuccess;
        ReportData.LogLevel = LogLevel::Info;
    }
    else
    {
        ReportData.StabilityCode = StabilityCode::Auth::RequestTemporaryWithdrawalFailed;
        ReportData.LogLevel = LogLevel::Error;
    }
        
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Auth::CancelTemporaryWithdrawal(const IGamebaseInternalData& InternalData, const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Auth;
    ReportData.CustomFields = {
        { AdditionalKey::SubCategory1, SubCategory::TemporaryWithdrawal },
    };
    ReportData.Error = Error;

    if (Error == nullptr)
    {
        ReportData.StabilityCode = StabilityCode::Auth::CancelTemporaryWithdrawalSuccess;
        ReportData.LogLevel = LogLevel::Info;
    }
    else
    {
        ReportData.StabilityCode = StabilityCode::Auth::CancelTemporaryWithdrawalFailed;
        ReportData.LogLevel = LogLevel::Error;
    }
        
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Event::ObserverBannedMember(const IGamebaseInternalData& InternalData, const FString& ObserverData)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Event;
    ReportData.StabilityCode = StabilityCode::Event::ObserverBannedMember;
    ReportData.LogLevel = LogLevel::Info;
    ReportData.CustomFields = {
        { AdditionalKey::ObserverData, ObserverData },
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Event::ObserverInvalidMember(const IGamebaseInternalData& InternalData, const FString& ObserverData)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Event;
    ReportData.StabilityCode = StabilityCode::Event::ObserverInvalidMember;
    ReportData.LogLevel = LogLevel::Info;
    ReportData.CustomFields = {
        { AdditionalKey::ObserverData, ObserverData },
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Event::LoggedOut(const IGamebaseInternalData& InternalData, const FString& LoggedOutData)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Event;
    ReportData.StabilityCode = StabilityCode::Event::LoggedOut;
    ReportData.LogLevel = LogLevel::Warn;
    ReportData.CustomFields = {
        { AdditionalKey::EventLoggedOutData, LoggedOutData },
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::TAA::SetGameUserData(const IGamebaseInternalData& InternalData, const int32 UserLevel, const FGamebaseAnalyticsUserData& UserData, const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::TAA;
    ReportData.StabilityCode = Error == nullptr ? StabilityCode::Taa::SetGameUserDataSuccess : StabilityCode::Taa::SetGameUserDataFailed;
    ReportData.LogLevel = LogLevel::Debug;
    ReportData.CustomFields = {
        { AdditionalKey::TaaUserLevel, FString::FromInt(UserLevel) },
        { AdditionalKey::GameUserData, EscapeJsonString(UserData.ToJson()).TrimQuotes() }
    };
    ReportData.Error = Error;
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::TAA::TraceLevelUp(const IGamebaseInternalData& InternalData, int32 UserLevel, const FGamebaseAnalyticsLevelUpData& LevelUpData, const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::TAA;
    ReportData.StabilityCode = Error == nullptr ? StabilityCode::Taa::TraceLevelUpSuccess : StabilityCode::Taa::TraceLevelUpFailed;
    ReportData.LogLevel = LogLevel::Debug;
    ReportData.CustomFields = {
        { AdditionalKey::TaaUserLevel, FString::FromInt(UserLevel) },
        { AdditionalKey::GameUserData, EscapeJsonString(LevelUpData.ToJson()).TrimQuotes() }
    };
    ReportData.Error = Error;
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::TAA::ResetUserLevel(const IGamebaseInternalData& InternalData, const FString& Message)
{
    FReportData ReportData;
    ReportData.LogType = LogType::TAA;
    ReportData.StabilityCode = StabilityCode::Taa::ResetUserLevel;
    ReportData.LogLevel = LogLevel::Debug;
    
    //Gamebase-Client/959
    if (Message.IsEmpty() == false)
    {
        ReportData.CustomFields = {
            { TEXT("GBAdditionalMessage"), Message },      //TODO: Android Only, 필요한지 생각해보기
        };
    }
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::TAA::SendPurchaseCompleteAnalytics(const IGamebaseInternalData& InternalData, const int32 UserLevel, const FString& StoreCode, const FString& PaymentSeq, const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Webview;
    ReportData.LogLevel = LogLevel::Debug;
    ReportData.StabilityCode = Error == nullptr ? StabilityCode::Taa::PurchaseCompleteSuccess : StabilityCode::Taa::PurchaseCompleteFailed;
    ReportData.CustomFields = {
        { AdditionalKey::TaaUserLevel, FString::FromInt(UserLevel) },
        { AdditionalKey::StoreCode, StoreCode },
        { AdditionalKey::PaymentSeq, PaymentSeq }
    };
    ReportData.Error = Error;
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Purchase::InitFailed(
    const IGamebaseInternalData& InternalData,
    const FGamebasePurchaseConfiguration& Configuration,
    const FString& ErrorMessage)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Purchase;
    ReportData.StabilityCode = StabilityCode::Iap::InitFailed;
    ReportData.LogLevel = LogLevel::Error;
    ReportData.CustomFields = {
        { AdditionalKey::TcIapAppKey, Configuration.IapAppkey },
        { AdditionalKey::PurchaseInitSettings, EscapeJsonString(Configuration.ToJson()).TrimQuotes() },
        { AdditionalKey::ErrorLog, ErrorMessage }
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Purchase::Purchase(
    const IGamebaseInternalData& InternalData,
    const FGamebasePurchaseConfiguration& Configuration,
    const FString& GamebaseProductId,
    const FGamebasePurchasableReceipt* Receipt,
    const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Purchase;
    ReportData.Error = Error;
    ReportData.CustomFields = {
        { AdditionalKey::TcIapAppKey, Configuration.IapAppkey },
        { AdditionalKey::PurchaseInitSettings, EscapeJsonString(Configuration.ToJson()).TrimQuotes() },
        { AdditionalKey::GamebaseProductId, GamebaseProductId }
    };
    
    if (Error == nullptr)
    {
        ReportData.StabilityCode = StabilityCode::Iap::PurchaseSuccess;
        ReportData.LogLevel = LogLevel::Info;

        if (Receipt != nullptr)
        {
            ReportData.CustomFields.GetValue().Add(AdditionalKey::PaymentSeq, Receipt->PaymentSeq);
            ReportData.CustomFields.GetValue().Add(AdditionalKey::PurchasePayload, Receipt->Payload.Get(""));
            ReportData.CustomFields.GetValue().Add(AdditionalKey::PurchasableReceipt, EscapeJsonString(Receipt->ToJson()).TrimQuotes());
            ReportData.CustomFields.GetValue().Add(AdditionalKey::Price, FString::SanitizeFloat(Receipt->Price));    
        }
    }
    else
    {
        if (Error->Code == GamebaseErrorCode::PURCHASE_USER_CANCELED)
        {
            ReportData.StabilityCode = StabilityCode::Iap::PurchaseCanceled;
            ReportData.LogLevel = LogLevel::Info;
        }
        else
        {
            ReportData.StabilityCode = StabilityCode::Iap::PurchaseFailed;
            ReportData.LogLevel = LogLevel::Error;
        }

        if (Receipt != nullptr)
        {
            ReportData.CustomFields.GetValue().Add(AdditionalKey::PaymentSeq, Receipt->PaymentSeq);
            ReportData.CustomFields.GetValue().Add(AdditionalKey::PurchasePayload, Receipt->Payload.Get(""));
            ReportData.CustomFields.GetValue().Add(AdditionalKey::PurchasableReceipt, EscapeJsonString(Receipt->ToJson()).TrimQuotes());
        }
    }
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Purchase::ItemListFailed(
    const IGamebaseInternalData& InternalData,
    const FString& IapAppkey,
    const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Purchase;
    ReportData.StabilityCode = StabilityCode::Iap::ItemListFailed;
    ReportData.LogLevel = LogLevel::Error;
    ReportData.Error = Error;
    ReportData.CustomFields = {
        { AdditionalKey::TcIapAppKey, IapAppkey }
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Purchase::WrongChangedStoreCode(
    const IGamebaseInternalData& InternalData,
    const FString& CurrentStoreCode,
    const FString& NewStoreCode)
{
    // Currently not used as store reset is not supported on Windows.

    FReportData ReportData;
    ReportData.LogType = LogType::Purchase;
    ReportData.StabilityCode = StabilityCode::Iap::WrongChangedStoreCode;
    ReportData.LogLevel = LogLevel::Error;
    ReportData.CustomFields = {
        { AdditionalKey::CurrentStoreCode, CurrentStoreCode },
        { AdditionalKey::NewStoreCode, NewStoreCode }
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Purchase::PurchaseInvalidReceipt(
    const IGamebaseInternalData& InternalData,
    const FGamebasePurchasableReceipt& Receipts)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Purchase;
    ReportData.StabilityCode = StabilityCode::Iap::PurchaseInvalidReceipt;
    ReportData.LogLevel = LogLevel::Error;
    ReportData.CustomFields = {
        { AdditionalKey::PurchasableReceipt, EscapeJsonString(Receipts.ToJson()).TrimQuotes() },
        { AdditionalKey::Price, FString::SanitizeFloat(Receipts.Price) }
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Purchase::ConsumableListNotEmpty(
    const IGamebaseInternalData& InternalData,
    const FString& IapAppkey,
    const FString& ReceiptListJsonString)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Purchase;
    ReportData.StabilityCode = StabilityCode::Iap::ConsumableListNotEmpty;
    ReportData.LogLevel = LogLevel::Info;
    ReportData.CustomFields = {
        { AdditionalKey::TcIapAppKey, IapAppkey },
        { AdditionalKey::PurchasableReceipts, EscapeJsonString(ReceiptListJsonString).TrimQuotes() }
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Purchase::ConsumableListFailed(
    const IGamebaseInternalData& InternalData,
    const FString& IapAppkey,
    const FGamebaseError* Error)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Purchase;
    ReportData.StabilityCode = StabilityCode::Iap::ConsumableListFailed;
    ReportData.LogLevel = LogLevel::Error;
    ReportData.Error = Error;
    ReportData.CustomFields = {
        { AdditionalKey::TcIapAppKey, IapAppkey }
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::WebView::Open(const IGamebaseInternalData& InternalData, const FString& Url, const FGamebaseWebViewConfiguration* Configuration, const FGamebaseError* Error)
{
    if (Error == nullptr)
    {
        return;
    }
    
    FReportData ReportData;
    ReportData.LogType = LogType::Webview;
    ReportData.StabilityCode = StabilityCode::WebView::OpenFailed;
    ReportData.LogLevel = LogLevel::Error;
    ReportData.CustomFields = {
        { AdditionalKey::Url, Url }
    };

    if (Configuration != nullptr)
    {
        ReportData.CustomFields.GetValue().Add(AdditionalKey::WebviewConfiguration, EscapeJsonString(Configuration->ToJson()).TrimQuotes());
    }
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Common::WrongUsage(const IGamebaseInternalData& InternalData, const FString& FunctionName, const FString& ErrorLog, const FGamebaseError* Error, const FString& Payload)
{
    //TODO: Error값 사용?
    FReportData ReportData;
    ReportData.LogType = LogType::Common;
    ReportData.StabilityCode = StabilityCode::Common::WrongUsage;
    ReportData.LogLevel = LogLevel::Error;
    ReportData.CustomFields = {
        { AdditionalKey::FunctionName, FunctionName },
        { AdditionalKey::ErrorLog, ErrorLog },
    };
    ReportData.Error = Error;

    if (Payload.IsEmpty() == false)
    {
        ReportData.CustomFields.GetValue().Add(AdditionalKey::WrongUsagePayload, Payload);
    }
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Common::DebugMessage(const IGamebaseInternalData& InternalData, const FString& Message)
{
    //TODO: SendDataKeyClientLogTime 필요여부
    FReportData ReportData;
    ReportData.LogType = LogType::Common;
    ReportData.StabilityCode = StabilityCode::LogDebugReport;
    ReportData.LogLevel = LogLevel::Debug;
    ReportData.CustomFields = {
        { AdditionalKey::GamebaseLog, Message },
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Common::WarnMessage(const IGamebaseInternalData& InternalData, const FString& Message)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Common;
    ReportData.StabilityCode = StabilityCode::LogWarnReport;
    ReportData.LogLevel = LogLevel::Warn;
    ReportData.CustomFields = {
        { AdditionalKey::GamebaseLog, Message },
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Common::ErrorMessage(const IGamebaseInternalData& InternalData, const FString& Message)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Common;
    ReportData.StabilityCode = StabilityCode::LogErrorReport;
    ReportData.LogLevel = LogLevel::Error;
    ReportData.CustomFields = {
        { AdditionalKey::GamebaseLog, Message },
    };
    
    Report(InternalData, ReportData);
}

void GamebaseInternalReport::Common::CheckAPI(const IGamebaseInternalData& InternalData, const FString& ApiName)
{
    FReportData ReportData;
    ReportData.LogType = LogType::Common;
    ReportData.StabilityCode = StabilityCode::LogDebugReport;
    ReportData.LogLevel = LogLevel::Debug;
    ReportData.CustomFields = {
        { AdditionalKey::FunctionName, ApiName },
    };
    
    Report(InternalData, ReportData);
}