#include "GamebaseDeviceInfo.h"

#include "GamebaseDebugLogger.h"
#include "Utils/GamebaseMics.h"

namespace GamebaseSaveInfo
{
    const FString StoreId { TEXT("NHN") };
    const FString Section { TEXT("GamePlatform") };
    
    void SetInternalValue(const FString& Key, const FString& Value)
    {
        FPlatformMisc::SetStoredValue(StoreId, Section, Key, Value);
    }

    TOptional<FString> GetInternalValue(const FString& Key)
    {
        FString GetValue;
        if (FPlatformMisc::GetStoredValue(StoreId, Section, Key, GetValue))
        {
            return GetValue;
        }

        return TOptional<FString>();
    }
}

namespace GamebaseSystemUtils
{
    FGamebaseMicsResult ReduceHexStringWithXOR(const FString& HexString)
    {
        if (HexString.Len() != 40)
        {
            return FGamebaseMicsResult::Failure({
                FString::Printf(TEXT("Failed to generate UUID due to invalid Hex String Length. (Len: %d)"), HexString.Len())
            });
        }

        constexpr int32 BufferSize = 10;
        uint8 Buffer[BufferSize] = { 0 };

        for (int32 i = 0; i < BufferSize; i++)
        {
            const uint8 PreByte = FParse::HexDigit(HexString[i * 2]) << 4 | FParse::HexDigit(HexString[i * 2 + 1]);
            const uint8 PostByte = FParse::HexDigit(HexString[i * 2 + 20]) << 4 | FParse::HexDigit(HexString[i * 2 + 21]);

            Buffer[i] = PreByte ^ PostByte;
        }

        TArray<FString> HexArray;

        for (int32 i = 0; i < BufferSize; i++)
        {
            HexArray.Add(FString::Printf(TEXT("%02x"), Buffer[i]));
        }

        const FString Result = FString::Join(HexArray, TEXT(""));

        return FGamebaseMicsResult::Success(Result);
    }
    
    FGamebaseMicsResult GenerateUUID(const FString& Key)
    {
        static constexpr int32 UdidLength = 40;
    
        FString UdidString;
        if (Key.IsEmpty())
        {
            return FGamebaseMicsResult::Failure({ TEXT("Failed to generate UUID because DeviceKey is empty.")} );
        }

        if (Key.Len() != UdidLength)
        {
            const FTCHARToUTF8 Converter(*Key);
            const uint8* Data = reinterpret_cast<const uint8*>(Converter.Get());
            const int32 DataLength = Converter.Length();
        
            FSHA1 SHA1;
            SHA1.Update(Data, DataLength);
            SHA1.Final();
        
            uint8 Hash[FSHA1::DigestSize];
            SHA1.GetHash(Hash);
        
            TArray<FString> HexArray;
            for (int32 i = 0; i < FSHA1::DigestSize; ++i)
            {
                HexArray.Add(FString::Printf(TEXT("%02x"), Hash[i]));
            }

            UdidString = FString::Join(HexArray, TEXT(""));
        }
        else
        {
            UdidString = Key;
        }
        
        const auto Result = ReduceHexStringWithXOR(UdidString);
    
        return Result.IsOk() ?
            FGamebaseMicsResult::Success(Result.GetOkValue()) :
            FGamebaseMicsResult::Failure(Result.GetErrorValue());
    }
    
    FString GetDeviceIdHash(const FString& DeviceId)
    {
        const FString Input = DeviceId.IsEmpty()
            ? TEXT("{00000000-0000-0000-0000-000000000000}")
            : DeviceId;

        const FTCHARToUTF8 Utf8String(*Input);
        const uint8* Bytes = reinterpret_cast<const uint8*>(Utf8String.Get());
        const int32 Length = Utf8String.Length();

        uint8 HashBytes[20];
        FSHA1::HashBuffer(Bytes, Length, HashBytes);
        
        return BytesToHex(HashBytes, UE_ARRAY_COUNT(HashBytes)).ToLower();
    }
}

void FGamebaseDeviceInfo::Initialize()
{
   auto AddErrorIfFailed = [this](const FGamebaseMicsResult& Result, TOptional<FString>& Target, TFunction<FString()>&& ErrorProcess = {}) -> void
    {
        if (Result.IsOk())
        {
            Target = Result.GetOkValue();
        }
        else
        {
            CachedErrors.Add(Result.GetErrorValue().Message);

            if (ErrorProcess)
            {
                Target = ErrorProcess();
            }
        }
    };
    
    AddErrorIfFailed(FGamebaseMics::GetLocale(), CountryCode);
    AddErrorIfFailed(FGamebaseMics::GetDeviceId(), DeviceKey, []() -> FString
    {
        static const FString StoreKey { TEXT("DeviceId") };
        
        GAMEBASE_LOG_GLOBAL_DEBUG("Failed to get DeviceKey from FGamebaseMics::GetDeviceId(). Using OS ID as fallback.");
        
        if (TOptional<FString> SavedDeviceId = GamebaseSaveInfo::GetInternalValue(StoreKey))
        {
            return GamebaseSystemUtils::GetDeviceIdHash(*SavedDeviceId);
        }

        const FString NewDeviceId = FGuid::NewGuid().ToString(EGuidFormats::DigitsWithHyphensInBraces);
        GamebaseSaveInfo::SetInternalValue(StoreKey, NewDeviceId);

        return GamebaseSystemUtils::GetDeviceIdHash(*NewDeviceId);
    });
    
    if (DeviceKey.IsSet())
    {
        const auto UUIDResult = GamebaseSystemUtils::GenerateUUID(DeviceKey.GetValue());
        AddErrorIfFailed(UUIDResult, UUID);
    }
    else
    {
        CachedErrors.Add(TEXT("Failed to generate UUID because DeviceKey is empty."));
    }
}

void FGamebaseDeviceInfo::InitializeCompleted()
{
    for (const FString& Error : CachedErrors)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("%s", *Error);
    }
    
    CachedErrors.Empty();
}
