﻿#include "GamebaseCoreLogger.h"

#include "GamebaseDebugLogger.h"
#include "GpLoggerConfiguration.h"
#include "GpLoggerSubsystem.h"

EGamebaseLogLevel ToLogLevel(const EGpLogLevel Level)
{
    switch (Level)
    {
        case EGpLogLevel::Debug:    return EGamebaseLogLevel::Debug;
        case EGpLogLevel::Info:     return EGamebaseLogLevel::Info;
        case EGpLogLevel::Warn:     return EGamebaseLogLevel::Warn;
        case EGpLogLevel::Error:    return EGamebaseLogLevel::Error;
        case EGpLogLevel::Fatal:    return EGamebaseLogLevel::Fatal;
        default:
            GAMEBASE_LOG_GLOBAL_WARNING("Check the value of GpLogLevel. (LogLevel= %d)", Level);
        break;
    }

    return EGamebaseLogLevel::None;
}

void UGamebaseCoreLogger::Initialize(const FGamebaseLoggerConfiguration& LoggerConfiguration)
{
    auto ConvertZone = [](const FString& Zone)-> EGpLoggerServiceZone
    {
        if (Zone.Equals(TEXT("alpha"), ESearchCase::IgnoreCase))
        {
            return EGpLoggerServiceZone::Alpha;
        }
        if (Zone.Equals(TEXT("beta"), ESearchCase::IgnoreCase))
        {
            return EGpLoggerServiceZone::Beta;
        }

        return EGpLoggerServiceZone::Real;
    };
    
    FGpLoggerConfiguration Configuration;
    Configuration.Appkey = LoggerConfiguration.AppKey;
    Configuration.ServiceZone = ConvertZone(LoggerConfiguration.ZoneType);
    Configuration.bEnableCrashReporter = LoggerConfiguration.bEnableCrashReporter;
    
    if (UGpLoggerSubsystem* LoggerSubsystem = UGameInstance::GetSubsystem<UGpLoggerSubsystem>(GetWorld()->GetGameInstance()))
    {
        LoggerSubsystem->Initialize(Configuration);
        
        FGpLoggerEventListener EventListener;
        EventListener.OnSuccessDelegate.BindUObject(this, &UGamebaseCoreLogger::OnSuccess);
        EventListener.OnSaveDelegate.BindUObject(this, &UGamebaseCoreLogger::OnSave);
        EventListener.OnFilterDelegate.BindUObject(this, &UGamebaseCoreLogger::OnFilter);
        EventListener.OnErrorDelegate.BindUObject(this, &UGamebaseCoreLogger::OnError);
        LoggerSubsystem->SetLoggerListener(EventListener);
        
        FGpLoggerCrashListener CrashListener;
        CrashListener.OnCrashFilterDelegate.BindUObject(this, &UGamebaseCoreLogger::IsCrashFilter);
        LoggerSubsystem->SetCrashListener(CrashListener);
    }
}

void UGamebaseCoreLogger::Debug(const FString& Message, const TMap<FString, FString>& UserFields)
{
    if (UGpLoggerSubsystem* LoggerSubsystem = UGameInstance::GetSubsystem<UGpLoggerSubsystem>(GetWorld()->GetGameInstance()))
    {
        LoggerSubsystem->Debug(Message, UserFields);
    }
}

void UGamebaseCoreLogger::Info(const FString& Message, const TMap<FString, FString>& UserFields)
{
    if (UGpLoggerSubsystem* LoggerSubsystem = UGameInstance::GetSubsystem<UGpLoggerSubsystem>(GetWorld()->GetGameInstance()))
    {
        LoggerSubsystem->Info(Message, UserFields);
    }
}

void UGamebaseCoreLogger::Warn(const FString& Message, const TMap<FString, FString>& UserFields)
{
    if (UGpLoggerSubsystem* LoggerSubsystem = UGameInstance::GetSubsystem<UGpLoggerSubsystem>(GetWorld()->GetGameInstance()))
    {
        LoggerSubsystem->Warn(Message, UserFields);
    }
}

void UGamebaseCoreLogger::Error(const FString& Message, const TMap<FString, FString>& UserFields)
{
    if (UGpLoggerSubsystem* LoggerSubsystem = UGameInstance::GetSubsystem<UGpLoggerSubsystem>(GetWorld()->GetGameInstance()))
    {
        LoggerSubsystem->Error(Message, UserFields);
    }
}

void UGamebaseCoreLogger::Fatal(const FString& Message, const TMap<FString, FString>& UserFields)
{
    if (UGpLoggerSubsystem* LoggerSubsystem = UGameInstance::GetSubsystem<UGpLoggerSubsystem>(GetWorld()->GetGameInstance()))
    {
        LoggerSubsystem->Fatal(Message, UserFields);
    }
}

void UGamebaseCoreLogger::SetUserField(const FString& Key, const FString& Value)
{
    if (UGpLoggerSubsystem* LoggerSubsystem = UGameInstance::GetSubsystem<UGpLoggerSubsystem>(GetWorld()->GetGameInstance()))
    {
        LoggerSubsystem->SetUserField(Key, Value);
    }
}

void UGamebaseCoreLogger::SetListener(const FGamebaseLoggerListener& Listener)
{
    GamebaseEventListener = Listener;
}

void UGamebaseCoreLogger::SetCrashFilter(const FGamebaseLoggerCrashFilterDelegate& FilterDelegate)
{
    GamebaseCrashFilter = FilterDelegate;
}

void UGamebaseCoreLogger::OnSuccess(const FGpLogEntry& Entry)
{
    FGamebaseLogEntry GbEntry;
    GbEntry.Message = Entry.Message;
    GbEntry.CreateTime = Entry.CreateTime;
    GbEntry.LogType = Entry.LogType;
    GbEntry.LogLevel = static_cast<EGamebaseLogLevel>(Entry.LogLevel);
    GbEntry.TransactionId = Entry.TransactionId;
    GbEntry.UserFields = Entry.UserFields;
        
    GamebaseEventListener.OnSuccessDelegate.ExecuteIfBound(GbEntry);
}

void UGamebaseCoreLogger::OnSave(const FGpLogEntry& Entry)
{
    FGamebaseLogEntry GbEntry;
    GbEntry.Message = Entry.Message;
    GbEntry.CreateTime = Entry.CreateTime;
    GbEntry.LogType = Entry.LogType;
    GbEntry.LogLevel = static_cast<EGamebaseLogLevel>(Entry.LogLevel);
    GbEntry.TransactionId = Entry.TransactionId;
    GbEntry.UserFields = Entry.UserFields;
        
    GamebaseEventListener.OnSaveDelegate.ExecuteIfBound(GbEntry);
}

void UGamebaseCoreLogger::OnFilter(const FGpLogEntry& Entry, const FGpLogFilter& Filter)
{
    FGamebaseLogEntry GbEntry;
    GbEntry.Message = Entry.Message;
    GbEntry.CreateTime = Entry.CreateTime;
    GbEntry.LogType = Entry.LogType;
    GbEntry.LogLevel = static_cast<EGamebaseLogLevel>(Entry.LogLevel);
    GbEntry.TransactionId = Entry.TransactionId;
    GbEntry.UserFields = Entry.UserFields;

    FGamebaseLogFilter GbFilter;
    GbFilter.Name = Filter.Name;
        
    GamebaseEventListener.OnFilterDelegate.ExecuteIfBound(GbEntry, GbFilter);
}

void UGamebaseCoreLogger::OnError(const FGpLogEntry& Entry, const FString& ErrorMessage)
{
    FGamebaseLogEntry GbEntry;
    GbEntry.Message = Entry.Message;
    GbEntry.CreateTime = Entry.CreateTime;
    GbEntry.LogType = Entry.LogType;
    GbEntry.LogLevel = static_cast<EGamebaseLogLevel>(Entry.LogLevel);
    GbEntry.TransactionId = Entry.TransactionId;
    GbEntry.UserFields = Entry.UserFields;
        
    GamebaseEventListener.OnErrorDelegate.ExecuteIfBound(GbEntry, ErrorMessage);
}

bool UGamebaseCoreLogger::IsCrashFilter(const FGpLoggerCrashData& CrashData)
{
    if (GamebaseCrashFilter.IsBound())
    {
        FGamebaseLoggerEngineCrashData GbCrashData;
        GbCrashData.LogLevel = ToLogLevel(CrashData.LogLevel);
        GbCrashData.Condition = CrashData.Condition;
        GbCrashData.StackTrace = CrashData.StackTrace;
    
        return GamebaseCrashFilter.Execute(GbCrashData);
    }
    
    return false;
}
