#include "GamebaseBridgeWebView.h"
#include "GamebaseCommunicationRequestTypes.h"
#include "GamebaseCommunicatorAsyncSender.h"

namespace GamebaseScheme
{
    const FName OpenWebBrowser(TEXT("gamebase://openWebBrowser"));
    const FName ShowWebView(TEXT("gamebase://showWebView"));
    const FName CloseWebView(TEXT("gamebase://closeWebView"));
    const FName WebViewSchemeEvent(TEXT("gamebase://schemeEvent"));
}

namespace
{
    int32 WebViewHandleNumber = 0;
}

FGamebaseBridgeWebView::FGamebaseBridgeWebView(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
    Communicator->RegisterEvent(GamebaseScheme::WebViewSchemeEvent, [=](const FGamebaseCommunicatorReceiveDataPtr& ReceiveData)
    {
        if (OnSchemeEventMap.Contains(ReceiveData->Handle))
        {
            OnSchemeEventMap[ReceiveData->Handle].ExecuteIfBound(ReceiveData->JsonData, GetGamebaseError(ReceiveData.ToSharedRef()).Get());
        }
    });
}

void FGamebaseBridgeWebView::OpenWebBrowser(const FString& Url)
{
    FGamebaseCommunicatorSendData Data{ GamebaseScheme::OpenWebBrowser };
    Data.JsonData = Url;

    Communicator->GetAsyncWithoutResponse(Data.ToJson(false));
}

void FGamebaseBridgeWebView::ShowWebView(const FString& Url, const FGamebaseWebViewConfiguration& Configuration, const FGamebaseErrorDelegate& CloseCallback, const TArray<FString>& SchemeList, const FGamebaseSchemeEventDelegate& SchemeEvent)
{
    FWebviewConfiguration ConfigData;
    ConfigData.Url = Url;
    ConfigData.Configuration = Configuration;

    FWebviewSchemeExtraData SchemeData;
    SchemeData.SchemeList = SchemeList;
    SchemeData.SchemeEvent = SetSchemeEvent(SchemeEvent);
    
    FGamebaseCommunicatorSendData Data { GamebaseScheme::ShowWebView };
    Data.JsonData = ConfigData.ToJson();
    Data.ExtraData = SchemeData.ToJson();
    
    GamebaseCommunicatorAsyncSender::UseScheme(Data, Communicator.Get(), CloseCallback, FGamebaseExtraDataDelegate::CreateRaw(this, &FGamebaseBridgeWebView::RemoveSchemeEvent));
}

void FGamebaseBridgeWebView::CloseWebView()
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::CloseWebView };

    Communicator->GetAsyncWithoutResponse(Data.ToJson(false));
}

int32 FGamebaseBridgeWebView::SetSchemeEvent(const FGamebaseSchemeEventDelegate& SchemeEvent)
{
    OnSchemeEventMap.Emplace(++WebViewHandleNumber, SchemeEvent);

    return WebViewHandleNumber;
}

void FGamebaseBridgeWebView::RemoveSchemeEvent(const FString& ExtraData)
{
    const int32 SchemeHandle = FCString::Atoi(*ExtraData);
    OnSchemeEventMap.Remove(SchemeHandle);
}
