#include "GamebaseBridgeTerms.h"
#include "GamebaseCommunicatorAsyncSender.h"
#include "GamebaseCommunicatorManager.h"

namespace GamebaseScheme
{
    const FName ShowTermsView (TEXT("gamebase://showTermsView"));
    const FName ShowTermsViewWithConfiguration (TEXT("gamebase://showTermsViewWithConfiguration"));
    const FName UpdateTerms(TEXT("gamebase://updateTerms"));
    const FName QueryTerms(TEXT("gamebase://queryTerms"));
    const FName IsShowingTermsView(TEXT("gamebase://isShowingTermsView"));
}

namespace GamebaseCommunicatorAsyncSender
{   
    void QueryTerms(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseQueryTermsResultDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }
            
            const FGamebaseQueryTermsResultPtr Result = MakeShared<FGamebaseQueryTermsResult, ESPMode::ThreadSafe>();
            Result->FromJson(ReceiveData->JsonData);
            
            Callback.ExecuteIfBound(Result.Get(), Error.Get());
        });
    }
}


FGamebaseBridgeTerms::FGamebaseBridgeTerms(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
}

void FGamebaseBridgeTerms::ShowTermsView(const FGamebaseDataContainerDelegate& CloseCallback)
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::ShowTermsView };

    GamebaseCommunicatorAsyncSender::DataContainer(Data, Communicator.Get(), CloseCallback);
}

void FGamebaseBridgeTerms::ShowTermsView(const FGamebaseTermsConfiguration& Configuration, const FGamebaseDataContainerDelegate& CloseCallback)
{
    FGamebaseCommunicatorSendData Data{ GamebaseScheme::ShowTermsViewWithConfiguration };
    Data.JsonData = Configuration.ToJson();

    GamebaseCommunicatorAsyncSender::DataContainer(Data, Communicator.Get(), CloseCallback);
}

void FGamebaseBridgeTerms::UpdateTerms(const FGamebaseUpdateTermsConfiguration& Configuration, const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData Data{ GamebaseScheme::UpdateTerms };
    Data.JsonData = Configuration.ToJson();

    GamebaseCommunicatorAsyncSender::Default(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgeTerms::QueryTerms(const FGamebaseQueryTermsResultDelegate& Callback)
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::QueryTerms };

    GamebaseCommunicatorAsyncSender::QueryTerms(Data, Communicator.Get(), Callback);
}

bool FGamebaseBridgeTerms::IsShowingTermsView()
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::IsShowingTermsView };

    const FString GetJsonString = Communicator->GetSync(Data.ToJson(false));
    if (GetJsonString.IsEmpty())
    {
        return false;
    }

    return GetJsonString.ToBool();
}