#include "GamebaseBridgeSystemInfo.h"
#include "GamebaseCommunicator.h"
#include "GamebaseCommunicatorAsyncSender.h"
#include "GamebaseCommunicatorManager.h"

namespace GamebaseScheme
{
    const FName RequestLastLoggedInProvider(TEXT("gamebase://requestLastLoggedInProvider"));
    const FName GetLastLoggedInProvider(TEXT("gamebase://getLastLoggedInProvider"));

    const FName GetUserID(TEXT("gamebase://getUserID"));
    const FName GetAccessToken(TEXT("gamebase://getAccessToken"));

    const FName GetSDKVersion(TEXT("gamebase://getSDKVersion"));
    const FName IsSandbox(TEXT("gamebase://isSandbox"));

    const FName SetDisplayLanguageCode(TEXT("gamebase://setDisplayLanguageCode"));
    const FName GetDisplayLanguageCode(TEXT("gamebase://getDisplayLanguageCode"));

    const FName GetDeviceLanguageCode(TEXT("gamebase://getDeviceLanguageCode"));
    const FName GetCarrierCode(TEXT("gamebase://getCarrierCode"));
    const FName GetCarrierName(TEXT("gamebase://getCarrierName"));
    const FName GetCountryCode(TEXT("gamebase://getCountryCode"));
    const FName GetCountryCodeOfUSIM(TEXT("gamebase://getCountryCodeOfUSIM"));
    const FName GetCountryCodeOfDevice(TEXT("gamebase://getCountryCodeOfDevice"));
}

namespace GamebaseCommunicatorAsyncSender
{
    void RequestLastLoggedInProvider(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseLastLoggedInProviderDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(FString(), Error.Get());
                return;
            }

            Callback.ExecuteIfBound(ReceiveData->JsonData, Error.Get());
        });
    }
}

FGamebaseBridgeSystemInfo::FGamebaseBridgeSystemInfo(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
}

void FGamebaseBridgeSystemInfo::RequestLastLoggedInProvider(const FGamebaseLastLoggedInProviderDelegate& Callback) const
{
    const FGamebaseCommunicatorSendData SendData { GamebaseScheme::RequestLastLoggedInProvider };

    GamebaseCommunicatorAsyncSender::RequestLastLoggedInProvider(SendData, Communicator.Get(), Callback);
}

FString FGamebaseBridgeSystemInfo::GetLastLoggedInProvider() const
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::GetLastLoggedInProvider };
    return Communicator->GetSync(Data.ToJson(false));
}

FString FGamebaseBridgeSystemInfo::GetUserID() const
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::GetUserID };
    return Communicator->GetSync(Data.ToJson(false));
}

FString FGamebaseBridgeSystemInfo::GetAccessToken() const
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::GetAccessToken };
    return Communicator->GetSync(Data.ToJson(false));
}

FString FGamebaseBridgeSystemInfo::GetSDKVersion() const
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::GetSDKVersion };
    return Communicator->GetSync(Data.ToJson(false));
}

bool FGamebaseBridgeSystemInfo::IsSandbox() const
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::IsSandbox };
    const FString GetJsonString = Communicator->GetSync(Data.ToJson(false));
    if (GetJsonString.IsEmpty())
    {
        return false;
    }

    const TSharedRef<TJsonReader<>> JsonReader = TJsonReaderFactory<>::Create(GetJsonString);
    TSharedPtr<FJsonObject> JsonObject = MakeShareable(new FJsonObject());

    if (FJsonSerializer::Deserialize(JsonReader, JsonObject) == false)
    {
        return false;
    }

    return JsonObject->GetBoolField("isSandbox");
}

FString FGamebaseBridgeSystemInfo::GetDisplayLanguageCode() const
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::GetDisplayLanguageCode };
    return Communicator->GetSync(Data.ToJson(false));
}

void FGamebaseBridgeSystemInfo::SetDisplayLanguageCode(const FString& LanguageCode)
{
    FGamebaseCommunicatorSendData Data{ GamebaseScheme::SetDisplayLanguageCode };
    Data.JsonData = LanguageCode;
    
    Communicator->GetSync(Data.ToJson(false));
}

FString FGamebaseBridgeSystemInfo::GetDeviceLanguageCode() const
{
    const FGamebaseCommunicatorSendData Data { GamebaseScheme::GetDeviceLanguageCode };
    return Communicator->GetSync(Data.ToJson(false));
}

FString FGamebaseBridgeSystemInfo::GetCarrierCode() const
{
    const FGamebaseCommunicatorSendData Data { GamebaseScheme::GetCarrierCode };
    return Communicator->GetSync(Data.ToJson(false));
}

FString FGamebaseBridgeSystemInfo::GetCarrierName() const
{
    const FGamebaseCommunicatorSendData Data { GamebaseScheme::GetCarrierName };
    return Communicator->GetSync(Data.ToJson(false));
}

FString FGamebaseBridgeSystemInfo::GetCountryCode() const
{
    const FGamebaseCommunicatorSendData Data { GamebaseScheme::GetCountryCode };
    return Communicator->GetSync(Data.ToJson(false));
}

FString FGamebaseBridgeSystemInfo::GetCountryCodeOfUSIM() const
{
    const FGamebaseCommunicatorSendData Data { GamebaseScheme::GetCountryCodeOfUSIM };
    return Communicator->GetSync(Data.ToJson(false));
}

FString FGamebaseBridgeSystemInfo::GetCountryCodeOfDevice() const
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::GetCountryCodeOfDevice };
    return Communicator->GetSync(Data.ToJson(false));
}