#include "GamebaseBridgePush.h"
#include "GamebaseCommunicatorAsyncSender.h"
#include "GamebaseCommunicatorManager.h"

namespace GamebaseScheme
{
    const FName RegisterPush(TEXT("gamebase://registerPush"));
    const FName QueryPush(TEXT("gamebase://queryPush"));
    const FName RegisterPushWithOption(TEXT("gamebase://registerPushWithOption"));
    const FName QueryTokenInfo(TEXT("gamebase://queryTokenInfo"));
    const FName GetNotificationOptions(TEXT("gamebase://getNotificationOptions"));
    const FName QueryNotificationAllowed(TEXT("gamebase://queryNotificationAllowed"));
}

namespace GamebaseCommunicatorAsyncSender
{
    void QueryPush(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebasePushConfigurationDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }
            
            const FGamebasePushConfigurationPtr PushConfiguration = MakeShared<FGamebasePushConfiguration, ESPMode::ThreadSafe>();
            PushConfiguration->FromJson(ReceiveData->JsonData);

            Callback.ExecuteIfBound(PushConfiguration.Get(), Error.Get());
        });
    }

    void QueryTokenInfo(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebasePushTokenInfoDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }

            const FGamebasePushTokenInfoPtr PushTokenInfo = MakeShared<FGamebasePushTokenInfo, ESPMode::ThreadSafe>();
            PushTokenInfo->FromJson(ReceiveData->JsonData);
            
            Callback.ExecuteIfBound(PushTokenInfo.Get(), Error.Get());
        });
    }

    void QueryNotificationAllowed(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseQueryNotificationAllowedDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(false, Error.Get());
                return;
            }

            Callback.ExecuteIfBound(ReceiveData->JsonData.ToBool(), Error.Get());
        });
    }
}

FGamebaseBridgePush::FGamebaseBridgePush(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
}

void FGamebaseBridgePush::RegisterPush(const FGamebasePushConfiguration& Configuration, const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData Data{ GamebaseScheme::RegisterPush };
    Data.JsonData = Configuration.ToJson();

    GamebaseCommunicatorAsyncSender::Default(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePush::RegisterPush(const FGamebasePushConfiguration& Configuration, const FGamebaseNotificationOptions& NotificationOptions, const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData Data{ GamebaseScheme::RegisterPushWithOption };
    Data.JsonData = Configuration.ToJson();
    Data.ExtraData = NotificationOptions.ToJson();

    GamebaseCommunicatorAsyncSender::Default(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePush::QueryTokenInfo(const FGamebasePushTokenInfoDelegate& Callback)
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::QueryTokenInfo };

    GamebaseCommunicatorAsyncSender::QueryTokenInfo(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePush::QueryPush(const FGamebasePushConfigurationDelegate& Callback)
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::QueryPush };

    GamebaseCommunicatorAsyncSender::QueryPush(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePush::SetSandboxMode(bool bIsSandbox)
{
    GAMEBASE_NOT_SUPPORT_API();
}

FGamebaseNotificationOptionsPtr FGamebaseBridgePush::GetNotificationOptions()
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::GetNotificationOptions };

    const FString GetJsonString = Communicator->GetSync(Data.ToJson(false));
    if (GetJsonString.IsEmpty())
    {
        return nullptr;
    }
    const FGamebaseNotificationOptionsPtr Options = MakeShared<FGamebaseNotificationOptions, ESPMode::ThreadSafe>();
    Options->FromJson(GetJsonString);

    return Options;
}

void FGamebaseBridgePush::QueryNotificationAllowed(const FGamebaseQueryNotificationAllowedDelegate& Callback)
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::QueryNotificationAllowed };

    GamebaseCommunicatorAsyncSender::QueryNotificationAllowed(Data, Communicator.Get(), Callback);
}
